import logging
from sandbox import sdk2
from sandbox.sandboxsdk.environments import PipEnvironment

from sandbox.projects.collections.StartrekBugWeightMetrics.wiki_helpers import update_wiki_tags_doc
from sandbox.projects.collections.StartrekBugWeightMetrics.startrek_helpers import calculate_issue_weight, \
    calculate_issue_priority, calculate_config


class StartrekBugWeightMetrics(sdk2.Task):
    class Requirements(sdk2.Requirements):
        ram = 2 * 1024
        cores = 1

        environments = (
            PipEnvironment('yandex-tracker-client', '1.9', custom_parameters=['--upgrade-strategy only-if-needed']),
            PipEnvironment('startrek-client', '2.5', custom_parameters=['--upgrade-strategy only-if-needed']),
        )

        class Caches(sdk2.Requirements.Caches):
            pass

    class Parameters(sdk2.Parameters):
        with sdk2.parameters.Group('General') as general:
            project = sdk2.parameters.String(
                'Project name',
                required=True,
            )

        with sdk2.parameters.Group('Startrek') as startrek:
            query = sdk2.parameters.String(
                'Startrek query',
                required=True,
            )
            tags_config = sdk2.parameters.JSON(
                'Tags config',
                required=True,
            )
            priority_config = sdk2.parameters.JSON(
                'Task priority by weight',
                required=True,
            )

        with sdk2.parameters.Group('Wiki') as wiki:
            wiki_doc_page = sdk2.parameters.String(
                'Wiki autogenerated doc page',
                required=False,
            )

    def on_execute(self):
        from startrek_client import Startrek

        client = Startrek(
            token=sdk2.Vault.data('collections_startrek_token'),
            useragent='Sandbox StartrekBugWeightMetrics Task',
        )

        with sdk2.helpers.ProgressMeter('Update wiki doc'):
            if self.Parameters.wiki_doc_page:
                update_wiki_tags_doc(
                    page=self.Parameters.wiki_doc_page,
                    token=sdk2.Vault.data('collections_wiki_token'),
                    config=self.Parameters.tags_config,
                )

        with sdk2.helpers.ProgressMeter('Update startrek issues weight'):
            tags_config = calculate_config(self.Parameters.tags_config)
            priority_config = self.Parameters.priority_config

            msg = 'Ticket updated {key} [priority: {prev_priority} -> {priority}] [weight: {prev_weight} -> {weight}]'
            debug_msg = 'Ticket {key} [priority: {prev_priority} -> {priority}] [weight: {prev_weight} -> {weight}]'
            issues_weight = {}

            issues = client.issues.find(
                query=self.Parameters.query,
            )

            for issue in issues:
                weight = calculate_issue_weight(issue, tags_config)
                priority = calculate_issue_priority(weight, priority_config)

                issues_weight[issue.key] = weight

                logging.debug(debug_msg.format(
                    key=issue.key,
                    prev_weight=issue.weight,
                    weight=weight,
                    prev_priority=issue.priority.key,
                    priority=priority,
                ))

                if issue.weight != weight or issue.priority.key != priority:
                    prev_weight = issue.weight
                    prev_priority = issue.priority.key

                    issue.update(
                        priority=priority,
                        weight=weight,
                    )

                    logging.info(msg.format(
                        key=issue.key,
                        prev_weight=prev_weight,
                        weight=weight,
                        prev_priority=prev_priority,
                        priority=priority,
                    ))
