import requests


WIKI_API = 'https://wiki-api.yandex-team.ru/_api/frontend'


def update_wiki_tags_doc(page, token, config):
    return update_wiki_page(page, token, body=generate_tags_doc(config))


def generate_tags_doc(config):
    columns = [
        ('env', 'Env'),
        ('feature', 'Feature (only one)'),
        ('action', 'Action (only one)'),
        ('place', 'Place'),
        ('mod', 'Mod'),
    ]

    hints = config['hints']
    fixes = config['fixes']
    weights = config['weights']

    head = [bold(title) for (_, title) in columns]
    content = ['\n'.join(tags_message(weights[key], hints, fixes)) for (key, _) in columns]

    return table([head, content])


def update_wiki_page(page, token, body):
    prev_body = get_wiki_page(page, token)
    if prev_body != body:
        post_wiki_page(page, token, body)


def get_wiki_page(page, token):
    response = get_json(requests.get(
        WIKI_API + page + '.raw',
        headers={'Authorization': 'OAuth ' + token}
    ))

    return response['data']['body']


def post_wiki_page(page, token, body):
    return get_json(requests.post(
        WIKI_API + page,
        headers={
            'Authorization': 'OAuth ' + token,
            'Content-Type': 'application/json',
        },
        json={
            'body': body,
        }
    ))


def get_json(response):
    response.raise_for_status()
    return response.json()


def bold(text):
    return '**' + text + '**'


def code(text):
    return '%%' + text + '%%'


def table(data):
    result = ['|| ' + ' | '.join(row) + ' ||' for row in data]
    return '#|\n' + '\n'.join(result) + '\n|#'


def tags_message(tags, hints, fixes):
    for tag, value in sorted(tags.iteritems()):
        tag_value = value
        tag_hint = ''

        if tag in hints:
            tag_hint = hints[tag]
        elif tag in fixes:
            tag_hint = 'weight * {val}'.format(val=fixes[tag])

        if tag in fixes:
            tag_value *= fixes[tag]

        yield '{tag} = {val}{hint}'.format(
            tag=code(tag),
            val=round(tag_value, 5),
            hint='\n \'- ({hint})'.format(hint=tag_hint) if tag_hint else ''
        )
