from datetime import datetime, timedelta

from sandbox import sdk2
import sandbox.common.types.task as ctt
from sandbox.projects.collections.LaunchComplaintsTasks import LaunchYasapPdbBackendComplaintsTasks
from sandbox.sandboxsdk.environments import PipEnvironment


YQL_TEMPLATE = """
PRAGMA yt.InferSchema;
PRAGMA yt.Pool = "collections_production";

DECLARE $cards as String;
DECLARE $boards as String;
DECLARE $output as String;
DECLARE $min_cards_number as Int64;
DECLARE $min_vq as Int64;


$cards_by_board = (
    select
        board as board_id,
        count(*) as cards_count
    from $cards
    group by board
);

$corgie_cards_by_board = (
    select
        board as board_id,
        count(*) as cards_count
    from $cards
    where Yson::ConvertToBool(Yson::YPath(document, "/is_private"))
    and Yson::ConvertToString(Yson::YPath(document, "/content/0/source_type")) == "image"
    and Yson::ConvertToInt64(Yson::YPath(document, "/content/0/content/avatars_meta/NeuralNetClasses/good_quality", Yson::Options(false as Strict))) >= $min_vq
    group by board
);

$extended_boards = (
    select
        b.id as id,
        Yson::ConvertToBool(Yson::YPath(b.document, "/additional_attributes/suggested_to_unprivate")) ?? False as is_corgie,
        Yson::ConvertToBool(Yson::YPath(b.document, "/is_private")) as is_private,
        c.cards_count ?? 0 as cards_count
    from $boards as b
    left join $cards_by_board as c
    on b.id == c.board_id
);

$boards_with_corgie = (
    select
        b.id as id,
        b.is_corgie as is_corgie,
        b.is_private as is_private,
        b.cards_count as cards_count,
        c.cards_count ?? 0 as corgie_cards_count
    from $extended_boards as b
    left join $corgie_cards_by_board as c
    on b.id == c.board_id
);

$updated_boards = (
    select
        id as id,
        is_corgie as old_is_corgie,
        corgie_cards_count == cards_count and is_private and cards_count >= $min_cards_number as new_is_corgie
    from $boards_with_corgie
);

insert into $output WITH TRUNCATE
select
    id as board,
    "suggested_to_unprivate" as attribute,
    IF(new_is_corgie == True, True, NULL) as value
from $updated_boards
where new_is_corgie != old_is_corgie;
"""


class CollectionsUpdateBoardAttributes(sdk2.Task):
    class Requirements(sdk2.Task.Requirements):
        environments = (
            PipEnvironment('yql'),
            PipEnvironment('yandex-yt'),
            PipEnvironment('yandex-yt-yson-bindings-skynet'),
        )

    class Parameters(sdk2.Task.Parameters):
        dump_dir = sdk2.parameters.String('YT directory with PDB Dump')
        result_dir = sdk2.parameters.String('YT directory for result table')

        yql_token_secret = sdk2.parameters.String('YQL token secret')
        yt_token_secret = sdk2.parameters.String('YT token secret')
        yt_proxy = sdk2.parameters.String('YT proxy (cluster)')
        admin_url = sdk2.parameters.String('Admin tasks endpoint')

        expiration_time = sdk2.parameters.Integer('Expiration time of output table in days', default=7)
        min_cards_number = sdk2.parameters.Integer('Minimum number of cards in board', default=8)
        min_vq = sdk2.parameters.Integer('Minimum value of good_quality attribute of card', default=90)

    def build_table_with_updates(self):
        if self.Context.table_with_updates:
            return self.Context.table_with_updates

        from yt.wrapper import YtClient, ypath_join
        yt_token = sdk2.Vault.data(self.owner, self.Parameters.yt_token_secret)
        yt_client = YtClient(proxy=self.Parameters.yt_proxy, token=yt_token)

        now = datetime.utcnow()
        output_table = ypath_join(self.Parameters.result_dir, now.isoformat())
        yt_client.create(
            type='table',
            path=output_table,
            attributes={
                'expiration_time': (now + timedelta(days=self.Parameters.expiration_time)).isoformat(),
            }
        )

        from yql.api.v1.client import YqlClient
        from yql.client.parameter_value_builder import YqlParameterValueBuilder as ValueBuilder
        yql_client = YqlClient(
            db=self.Parameters.yt_proxy,
            token=sdk2.Vault.data(self.owner, self.Parameters.yql_token_secret)
        )

        parameters = {
            '$cards': ValueBuilder.make_string(ypath_join(self.Parameters.dump_dir, 'card')),
            '$boards': ValueBuilder.make_string(ypath_join(self.Parameters.dump_dir, 'board')),
            '$min_cards_number': ValueBuilder.make_int64(self.Parameters.min_cards_number),
            '$min_vq': ValueBuilder.make_int64(self.Parameters.min_vq),
            '$output': ValueBuilder.make_string(output_table),
        }
        request = yql_client.query(YQL_TEMPLATE, syntax_version=1).run(parameters=ValueBuilder.build_json_map(parameters))
        errors = request.get_results(wait=True).errors
        if errors:
            raise RuntimeError([error.message for error in errors])

        self.Context.table_with_updates = output_table
        self.Context.save()
        return self.Context.table_with_updates

    def run_admin_task(self, table):
        if self.Context.admin_task:
            return

        task = LaunchYasapPdbBackendComplaintsTasks(
            task='update_additional_attributes',
            base_url=self.Parameters.admin_url,
            additional_params={
                'path': table,
            },
            description='Updating additional attributes for board for {}'.format(table),
            kill_timeout=5 * 60,
            max_restarts=0,
            parent=self,
        )
        task.save()
        task_id = task.enqueue().id

        self.Context.admin_task = task_id
        self.Context.save()

        raise sdk2.WaitTask(
            [task_id],
            (ctt.Status.Group.FINISH + ctt.Status.Group.BREAK),
            wait_all=True
        )

    def on_execute(self):
        table = self.build_table_with_updates()
        self.run_admin_task(table)
