from sandbox import sdk2
import sandbox.common.types.task as ctt
from sandbox.projects.collections.LaunchComplaintsTasks import LaunchYasapPdbBackendComplaintsTasks
from sandbox.sandboxsdk.environments import PipEnvironment


class CollectionsBanBoardsBySlices(sdk2.Task):
    class Requirements(sdk2.Task.Requirements):
        environments = (
            PipEnvironment('yandex-yt'),
            PipEnvironment('yandex-yt-yson-bindings-skynet'),
        )

    class Parameters(sdk2.Task.Parameters):
        source_table = sdk2.parameters.String('YT table to slice')
        slice_size = sdk2.parameters.Integer('Slice size', default=400)
        processed_rows_attribute = sdk2.parameters.String('Processed rows attribute', default='processed_rows')

        yt_token_secret = sdk2.parameters.String('YT token secret')
        yt_proxy = sdk2.parameters.String('YT proxy (cluster)')
        admin_url = sdk2.parameters.String('Admin tasks endpoint')
        admin_task_name = sdk2.parameters.String('Admin task name', default='ban_with_antifraud')

    def build_table_with_slice(self):
        if self.Context.launch_task_params:
            return self.Context.launch_task_params

        from yt.wrapper import YtClient, ypath_join
        yt_token = sdk2.Vault.data(self.owner, self.Parameters.yt_token_secret)
        yt_client = YtClient(proxy=self.Parameters.yt_proxy, token=yt_token)

        table = self.Parameters.source_table

        processed_rows = yt_client.get_attribute(
            path=table,
            attribute=self.Parameters.processed_rows_attribute,
            default=0
        )
        row_count = yt_client.get_attribute(path=table, attribute='row_count')
        if processed_rows == row_count:
            return
        slice_end = min(processed_rows + self.Parameters.slice_size, row_count)
        path_parts = table.strip('/').split('/')
        directory = '//' + "/".join(path_parts[:-2])
        state = path_parts[-2]
        slice = "{source}[#{slice_begin}:#{slice_end}]".format(
            source=path_parts[-1],
            slice_begin=processed_rows,
            slice_end=slice_end,
        )
        yt_client.set_attribute(
            path=table,
            attribute=self.Parameters.processed_rows_attribute,
            value=slice_end
        )

        self.Context.launch_task_params = {
            'yt_path': directory,
            'state': state,
            'yt_table': slice,
        }
        self.Context.save()
        return self.Context.launch_task_params

    def run_admin_task(self, params):
        if self.Context.admin_task:
            return

        task = LaunchYasapPdbBackendComplaintsTasks(
            task=self.Parameters.admin_task_name,
            base_url=self.Parameters.admin_url,
            additional_params=params,
            description='Ban boards for table {}'.format(params['yt_table']),
            kill_timeout=5 * 60,
            max_restarts=0,
            parent=self,
        )
        task.save()
        task_id = task.enqueue().id

        self.Context.admin_task = task_id
        self.Context.save()

        raise sdk2.WaitTask(
            [task_id],
            (ctt.Status.Group.FINISH + ctt.Status.Group.BREAK),
            wait_all=True
        )

    def on_execute(self):
        params = self.build_table_with_slice()
        if not params:
            return
        self.run_admin_task(params)


__TASK__ = CollectionsBanBoardsBySlices
