# -*- encoding: utf-8 -*-

import logging
import subprocess

from sandbox import sdk2
from sandbox.projects.common import utils
from sandbox.sandboxsdk.environments import PipEnvironment

from sandbox.projects.collections.resources import CollectionsPrepareAntifraudBanTablesBinary


class CollectionsPrepareAntifraudBanTables(sdk2.Task):
    class Requirements(sdk2.Task.Requirements):
        environments = (
            PipEnvironment('yandex-yt'),
        )
        cores = 1
        ram = 1024

        class Caches(sdk2.Requirements.Caches):
            pass

    class Parameters(sdk2.Task.Parameters):
        with sdk2.parameters.Group('YT common parameters') as common_parameters:
            yt_token_vault = sdk2.parameters.String(
                'YT token vault',
                required=True,
                default='yt_token'
            )
            yt_proxy = sdk2.parameters.String(
                'YT proxy',
                required=True,
                default='arnold'
            )

        with sdk2.parameters.RadioGroup("Anti-all type") as anti_all_type:
            anti_all_type.values["antifraud"] = anti_all_type.Value("antifraud", default=True)
            anti_all_type.values["clean-web"] = anti_all_type.Value("clean-web")

        with sdk2.parameters.Group('YT source parameters') as source_parameters:
            anti_all_dir = sdk2.parameters.String(
                'Path to anti-all node',
                required=True,
                default='//home/antifraud/export/collections',
            )

        with sdk2.parameters.Group('YT result parameters') as result_parameters:
            ban_unban_dir = sdk2.parameters.String(
                'Path to ban unban queue',
                required=True,
                default='//home/collections/ban_unban/queue',
            )
            result_ttl = sdk2.parameters.Integer(
                'Ttl (seconds) of result table (0=inf)',
                required=True,
                default=15*24*60*60,
            )
        report_timelag = sdk2.parameters.String(
            'Report timelag for monit service',
            default='',
        )
        notify = sdk2.parameters.Bool(
            'Ban with notify',
            default=True,
        )

    def get_parameters(self):
        tool_id = utils.get_and_check_last_released_resource_id(CollectionsPrepareAntifraudBanTablesBinary)
        tool_path = str(sdk2.ResourceData(sdk2.Resource[tool_id]).path)

        args = [
            tool_path, self.Parameters.anti_all_type,
            '--yt-proxy', self.Parameters.yt_proxy,
            '--anti-all-dir', self.Parameters.anti_all_dir,
            '--ban-unban-dir', self.Parameters.ban_unban_dir,
            '--result-ttl', str(self.Parameters.result_ttl),
        ]
        if self.Parameters.report_timelag:
            args.extend(
                ['--report-timelag', self.Parameters.report_timelag]
            )
        if self.Parameters.notify:
            args.append('--notify')
        return args

    def on_execute(self):
        yt_token = sdk2.Vault.data(self.owner, self.Parameters.yt_token_vault)

        args = self.get_parameters()

        with sdk2.helpers.ProcessLog(self, logging.getLogger('prepare_antifraud_bans')) as pl:
            process = subprocess.Popen(
                args,
                stdout=pl.stdout,
                stderr=pl.stderr,
                env={'YT_TOKEN': yt_token}
            )
            process.wait()
            if process.returncode:
                raise Exception('Preparation finished unsuccessfully')


__TASK__ = CollectionsPrepareAntifraudBanTables
