import os
import time
from sandbox import sdk2
import sandbox.common.types.task as ctt
from sandbox.sandboxsdk.environments import PipEnvironment
from sandbox.projects.collections.LaunchComplaintsTasks import LaunchYasapPdbBackendComplaintsTasks

import logging


class CollectionsCheckAntispamBansDirectory(sdk2.Task):
    class Parameters(sdk2.Task.Parameters):
        # if data from antispam is put into collections directories just fill
        # the fields for antispam with the same data as for collections
        antispam_proxy = sdk2.parameters.String('Antispam YT cluster (proxy)')
        antispam_directory = sdk2.parameters.String('Path to Antispam export')
        antispam_backup = sdk2.parameters.String('Path to backup in Antispam')

        collections_proxy = sdk2.parameters.String('Collections YT cluster (proxy)')
        collections_directory = sdk2.parameters.String('Path to Collections Antispam dir')
        collections_backup = sdk2.parameters.String('Path to backup Collections Antispam')
        ttl = sdk2.parameters.Float('TTL for copied tables (minutes)', default=360)

        yt_token = sdk2.parameters.String('YT token secret')
        url = sdk2.parameters.String('Collections admin tasks enpoint')

    class Requirements(sdk2.Task.Requirements):
        environments = (
            PipEnvironment('yandex-yt'),
            PipEnvironment('yandex-yt-yson-bindings-skynet'),
            PipEnvironment('yandex-yt-transfer-manager-client'),
        )
        cores = 1
        ram = 1024

        class Caches(sdk2.Requirements.Caches):
            pass

    def get_yt_client(self, proxy):
        from yt.wrapper import YtClient
        yt_token = sdk2.Vault.data(self.Parameters.yt_token)
        yt_client = YtClient(
            proxy=proxy,
            token=yt_token,
        )
        return yt_client

    def get_tables(self):
        if self.Context.data:
            return self.Context.data

        yt_client = self.get_yt_client(self.Parameters.antispam_proxy)
        children = yt_client.list(
            self.Parameters.antispam_directory,
            attributes=['type'],
            absolute=True
        )
        paths_to_be_processed = [
            str(item)
            for item in children
            if (
                str(item) != self.Parameters.antispam_backup
                and item.attributes['type'] == 'table'
            )
        ]

        self.Context.data = paths_to_be_processed
        self.Context.save()

        return paths_to_be_processed

    def clean_tables(self, tables):
        yt_client = self.get_yt_client(self.Parameters.antispam_proxy)
        for table in tables:
            yt_client.remove(table, force=True)

    def _move_tables_cross_cluster(self, tables):
        from yt.transfer_manager.client import TransferManager
        from yt.wrapper.ypath import ypath_join

        yt_token = sdk2.Vault.data(self.Parameters.yt_token)
        client = TransferManager(token=yt_token)

        table_pairs = []
        new_tables = []
        for source_table in tables:
            destination_table = ypath_join(
                self.Parameters.collections_directory,
                os.path.basename(source_table),
            )
            table_pairs.append(
                (source_table, destination_table)
            )
            new_tables.append(destination_table)

        client.add_tasks_from_src_dst_pairs(
            table_pairs,
            self.Parameters.antispam_proxy,
            self.Parameters.collections_proxy,
            enable_failed_tasks_restarting=True,
            sync=True,
        )

        self.clean_tables(tables)

        return new_tables

    def _move_tables_one_cluster(self, tables):
        from yt.wrapper.ypath import ypath_join

        if self.Parameters.collections_directory == self.Parameters.antispam_directory:
            return tables

        yt_client = self.get_yt_client(self.Parameters.collections_proxy)
        new_tables = []
        for table in tables:
            new_table = ypath_join(
                self.Parameters.collections_directory,
                os.path.basename(table),
            )
            yt_client.copy(table, new_table)
            new_tables.append(new_table)

        self.clean_tables(tables)

        return new_tables

    def _set_tables_ttl(self, tables):
        yt_client = self.get_yt_client(self.Parameters.collections_proxy)
        ts = int((time.time() + self.Parameters.ttl * 60) * 1000)
        for table in tables:
            yt_client.set_attribute(table, 'expiration_time', ts)

    def move_tables(self, tables_to_process):
        if self.Context.tables:
            return self.Context.tables

        if self.Parameters.antispam_proxy == self.Parameters.collections_proxy:
            tables = self._move_tables_one_cluster(tables_to_process)
        else:
            tables = self._move_tables_cross_cluster(tables_to_process)

        if self.Parameters.ttl:
            self._set_tables_ttl(tables)

        self.Context.tables = tables
        self.Context.save()

        return self.Context.tables

    def create_tasks(self, tables_to_process):
        if self.Context.subtasks:
            return

        subtasks = []
        for item in tables_to_process:
            task = LaunchYasapPdbBackendComplaintsTasks(
                task='process_antispam_table',
                base_url=self.Parameters.url,
                additional_params={
                    'antispam_table': item,
                    'backup_table': self.Parameters.collections_backup
                },
                description='Launching table processing {}'.format(item),
                kill_timeout=30 * 60,
                max_restarts=0,
                parent=self,
            )
            task.save()
            task_id = task.enqueue().id
            subtasks.append(str(task_id))

        self.Context.subtasks = subtasks
        self.Context.save()

        if not subtasks:
            return

        raise sdk2.WaitTask(
            subtasks,
            (ctt.Status.Group.FINISH + ctt.Status.Group.BREAK),
            wait_all=True
        )

    def on_execute(self):
        tables_to_process = self.get_tables()
        logging.warning('Tables to process: %s', tables_to_process)
        moved_tables = self.move_tables(tables_to_process)
        logging.warning('Moved tables: %s', moved_tables)
        self.create_tasks(moved_tables)
