import datetime
import os
from sandbox import sdk2
from sandbox.sandboxsdk.environments import PipEnvironment


YQL_QUERY = '''
pragma yson.DisableStrict;


DECLARE $antispam_backup AS String;
DECLARE $cards_dump AS String;
DECLARE $output AS String;


$antispam  = (
    SELECT DISTINCT card_id AS card_id
    FROM $antispam_backup
);


$results = (
    SELECT 
        card.id AS card_id
    FROM
        $antispam AS antispam
        RIGHT SEMI JOIN $cards_dump AS card
            ON antispam.card_id == card.id
    WHERE
        Yson::LookupBool(card.document, 'is_private') == false AND 
        Yson::Lookup(card.document, 'ban') IS null
);


INSERT INTO $output WITH TRUNCATE (
    SELECT * FROM $results
);
'''


class CollectionsResendNotBannedCards(sdk2.Task):
    class Requirements(sdk2.Task.Requirements):
        environments = (
            PipEnvironment('yql'),
            PipEnvironment('yandex-yt'),
            PipEnvironment('yandex-yt-yson-bindings-skynet'),
        )

    class Parameters(sdk2.Task.Parameters):
        yql_token_secret = sdk2.parameters.String('Yql token secret')
        yt_cluster = sdk2.parameters.String('Yt cluster (proxy)')
        dump_dir = sdk2.parameters.String('Collections dump directory')
        antispam_dir = sdk2.parameters.String('Cards antispam directory')
        antispam_backup_table = sdk2.parameters.String('Cards antispam backup table')

    def get_yql_client(self):
        from yql.api.v1.client import YqlClient
        from yql.config import config
        yql_token = sdk2.Vault.data(self.Parameters.yql_token_secret)
        yql_client = YqlClient(db=self.Parameters.yt_cluster, token=yql_token)
        config.no_color = True
        config.is_tty = True
        return yql_client

    def generate_output_path(self):
        output_path = os.path.join(
            self.Parameters.antispam_dir,
            datetime.datetime.utcnow().isoformat(),
        )
        return output_path

    def on_execute(self):
        from yql.client.parameter_value_builder import YqlParameterValueBuilder as ValueBuilder

        cards_path = os.path.join(self.Parameters.dump_dir, 'card')
        output_path = self.generate_output_path()

        parameters = {
            '$antispam_backup': ValueBuilder.make_string(self.Parameters.antispam_backup_table),
            '$cards_dump': ValueBuilder.make_string(cards_path),
            '$output': ValueBuilder.make_string(output_path),
        }

        yql_client = self.get_yql_client()
        request = yql_client.query(query=YQL_QUERY)
        request.run(parameters=ValueBuilder.build_json_map(parameters))
        _ = request.get_results()
