import datetime
import logging
import os
from sandbox import sdk2
from sandbox.sandboxsdk.environments import PipEnvironment
from sandbox.sdk2.helpers import subprocess


class CollectionsSendCardsToAntispamCheck(sdk2.Task):
    class Requirements(sdk2.Task.Requirements):
        environments = (
            PipEnvironment('yandex-yt'),
            PipEnvironment('yandex-yt-yson-bindings-skynet'),
        )

    class Parameters(sdk2.Task.Parameters):
        proxy = sdk2.parameters.String('YT cluster (proxy)')
        directory = sdk2.parameters.String('Collections cards export dir')
        backups = sdk2.parameters.String('Collections backups dir')
        yt_token = sdk2.parameters.String('YT token secret')
        send_diff = sdk2.parameters.Bool('Send dumps diff', default=True)
        diff_builder = sdk2.parameters.Resource(
            'Diff builder binary', default='779356901', required=True
        )

    def load_resource(self, resource_id):
        """return path to file in resource
        """
        resource_data = sdk2.ResourceData(sdk2.Resource[resource_id])
        path = str(resource_data.path)
        return path

    def get_yt_client(self, proxy):
        from yt.wrapper import YtClient
        yt_token = sdk2.Vault.data(self.Parameters.yt_token)
        yt_client = YtClient(
            proxy=proxy,
            token=yt_token,
        )
        return yt_client

    def get_dumps(self):
        yt_client = self.get_yt_client(self.Parameters.proxy)
        backups = yt_client.list(
            self.Parameters.backups,
            attributes=['type'],
            absolute=True
        )

        last, prev = sorted(filter(
            lambda node: node.attributes['type'] == 'map_node',
            backups,
        ), reverse=True)[:2]

        return last, prev

    def run_diff_builder(self, last_table, prev_table, out_table):
        yt_token = sdk2.Vault.data(self.Parameters.yt_token)
        executable = self.load_resource(self.Parameters.diff_builder)
        with sdk2.helpers.ProcessLog(self, logging.getLogger('diff_builder')) as pl:
            cmd = [
                executable,
                '--proxy', self.Parameters.proxy,
                '--new-table', last_table,
                '--out-table', out_table,
            ]
            if self.Parameters.send_diff:
                cmd.extend([
                    '--old-table', prev_table,
                ])
            process = subprocess.Popen(
                cmd,
                env={
                    'YT_TOKEN': yt_token,
                },
                stdout=pl.stdout,
                stderr=pl.stderr,
            )
            process.wait()
            if process.returncode:
                raise Exception(
                    'MR filter returned non-zero code [{}]'.format(
                        process.returncode
                    )
                )

    def on_execute(self):
        last_dump, prev_dump = self.get_dumps()

        last_table = os.path.join(last_dump, 'card')
        prev_table = os.path.join(prev_dump, 'card')

        out_table = os.path.join(
            self.Parameters.directory,
            datetime.datetime.utcnow().isoformat(),
        )
        self.run_diff_builder(last_table, prev_table, out_table)
