import logging
from sandbox import sandboxsdk
from sandbox.sandboxsdk.channel import channel

from sandbox.projects.collections import resources as collections_resources
from sandbox.projects.common import error_handlers as eh
from sandbox.projects.common.search import components as search_components
from sandbox.projects import resource_types

DEFAULT_BASESEARCH_PORT = 17171
FUSION_WAIT_INDEX = "wait_index"
DEFAULT_START_TIMEOUT = 600
DEFAULT_SHUTDOWN_TIMEOUT = 360


def create_boards_search_params(n=None):
    fusion_params = search_components.create_fusion_params(n, old_task=True)
    fusion_params.Binary.resource_type = [collections_resources.CollectionsBoardsRtyServer]
    fusion_params.Config.resource_type = [collections_resources.CollectionsBoardsRtyServerConfigs]
    fusion_params.Config.name = 'basesearch{}_config_resource_id'.format("" if n is None else n)
    fusion_params.Database.resource_type.append(resource_types.RTYSERVER_SEARCH_DATABASE)

    class BoardsSearchParams(object):
        Database = fusion_params.Database
        Binary = fusion_params.Binary
        Config = fusion_params.Config
        StaticData = fusion_params.StaticData
        MaxDocs = fusion_params.MaxDocs
        DbSource = fusion_params.DbSource
        StartTimeout = fusion_params.StartTimeout
        Models = fusion_params.Models

        params = (
            Binary, Config, StaticData,
            DbSource, Database, MaxDocs,
            StartTimeout, Models
        )

    return BoardsSearchParams


def get_boards_search(
    params=create_boards_search_params(),
    shard=0,
    shards_count=1,
    use_profiler=False,
    port=DEFAULT_BASESEARCH_PORT,
    component_creator=None,
    get_db=True,
    max_documents=None,
    default_wait=FUSION_WAIT_INDEX,
    event_log=False,
    run_count=None
):
    ctx = channel.task.ctx

    sys_info = sandboxsdk.util.system_info()
    resource = channel.sandbox.get_resource(ctx[params.Binary.name])

    if not sandboxsdk.util.is_arch_compatible(resource.arch, sys_info['arch']):
        eh.check_failed(
            'Cannot execute task. resource: {0} has incorrect arch {1} instead of {2}'.format(
                resource.id, resource.arch, sys_info['arch']
            )
        )

    binary_path = channel.task.sync_resource(ctx[params.Binary.name])

    if not get_db:
        database_path = None
    else:
        database_path = channel.task.sync_resource(ctx[params.Database.name])

    configs_path = channel.task.sync_resource(ctx[params.Config.name])
    logging.info("configs path:" + configs_path)

    fusion_config_path = configs_path + "/rtyserver.conf-common"
    basesearch_cfg_path = configs_path + "/BoardsTier0.cfg"

    oxygen_config_path = None

    static_data_path = None
    if ctx.get(params.StaticData.name, None):
        static_data_path = channel.task.sync_resource(ctx[params.StaticData.name])

    mxnet_models = None
    if ctx.get(params.Models.name, None):
        mxnet_models = channel.task.sync_resource(ctx[params.Models.name])

    extra_params = {
        "CONFIG_PATH": configs_path,
        "CTYPE": "testing",
        "SERVICE": "boards",
        "LOG_LEVEL": "6",
        "DETACH_DIRECTORY": ".",
        "STATE_ROOT": "."
    }
    patch_params = {
        "Server.ModulesConfig.DOCFETCHER.Enabled": "false",
        "Server.ModulesConfig.Synchronizer.Enabled": "false",
        "Server.Repair.Enabled": "true",
    }

    return search_components.FusionSearch(
        work_dir=channel.task.abs_path(),
        binary=binary_path,
        database_dir=database_path,
        use_profiler=use_profiler,
        port=port,
        shard=shard,
        shards_count=shards_count,
        start_timeout=ctx.get(params.StartTimeout.name) or DEFAULT_START_TIMEOUT,

        fusion_config_path=fusion_config_path,
        config_file=basesearch_cfg_path,
        max_documents=max_documents,
        default_wait=default_wait,
        oxygen_config=oxygen_config_path,
        mxnet_models=mxnet_models,
        event_log=event_log,
        static_data_path=static_data_path,
        extra_params=extra_params,
        patch_params=patch_params,
        run_count=run_count
    )
