from sandbox import sdk2

from sandbox.common.types.misc import NotExists
import sandbox.common.types.task as ctt
from sandbox import common
from sandbox.projects import DeployNannyDashboard as deploy_nanny_dashboard
from sandbox.projects.collections.card_recommender import ShardAcceptanceTest
from sandbox.projects.collections.resources import CollectionsCardRecommenderShard


TEST = 'test'
PRIEMKA = 'priemka'
PROD = 'prod'

TASK_RELEASE_TYPE = {
    TEST: ctt.ReleaseStatus.TESTING,
    PRIEMKA: ctt.ReleaseStatus.PRESTABLE,
    PROD: ctt.ReleaseStatus.STABLE
}

NANNY_PARAMS = {
    TEST: {
        'deployment_nanny_dashboard_name': 'pdb_cg',
        'deployment_nanny_dashboard_recipe': 'collections_card_recommender_testing',
        'deployment_nanny_dashboard_filter': 'card_recommender',
    },
    PRIEMKA: {
        'deployment_nanny_dashboard_name': 'collections_quality_priemka',
        'deployment_nanny_dashboard_recipe': 'deploy_card_recommender',
        'deployment_nanny_dashboard_filter': 'card_recommender',
    },
    PROD: {
        'deployment_nanny_dashboard_name': 'pdb_production',
        'deployment_nanny_dashboard_recipe': 'deploy_collections_card_recommender',
        'deployment_nanny_dashboard_filter': 'card_recommender',
    },
}


class CardRecommenderShardAutoDeploy(sdk2.Task):
    """ Task for card recommender shard acceptance testing and deploy """

    class Parameters(sdk2.Task.Parameters):
        shardmap_task_id = sdk2.parameters.Integer("Id of publish shards task", required=True)
        shard_env = sdk2.parameters.String(
            "Shard environment",
            required=True,
            choices=[('Production', PROD), ('Test', TEST)]
        )
        deploy_shard = sdk2.parameters.Bool("Deploy shard")
        acceptance_testing = sdk2.parameters.Bool("Need acceptance testing before shard deployment")
        items_max_delta = sdk2.parameters.Integer(
            "Maximum percent of different items in priemka and production responses",
            default=10,
            required=True
        )
        relevance_max_delta = sdk2.parameters.Integer(
            "Maximum percent of average relevance diff in priemka and production responses",
            default=10,
            required=True
        )
        priemka_address = sdk2.parameters.String(
            "Priemka address",
            required=True,
        )
        production_address = sdk2.parameters.String(
            "Production address",
            required=True,
        )
        exp_name = sdk2.parameters.String(
            "Recommender experiment name",
            required=False
        )

    def find_shard_task_and_resource(self):
        shardmap_task = sdk2.Task.find(id=self.Parameters.shardmap_task_id).first()
        if shardmap_task is None:
            raise common.errors.TaskFailure("Can't find shardmap task")

        shard_task = shardmap_task.find().first()
        if shard_task is None:
            raise common.errors.TaskFailure("Can't find build shard task")
        self.Context.shard_task_id = shard_task.id

        shard_resource = sdk2.Resource.find(
            CollectionsCardRecommenderShard,
            task=shard_task
        ).first()
        if shard_resource is None:
            raise common.errors.TaskFailure("Can't find shard resource")
        self.Context.shard_resource_id = shard_resource.id

        self.Context.save()

    def release_shard_task(self, release_type):
        self.server.release(
            task_id=self.Context.shard_task_id,
            type=release_type,
            subject="Card recommender shard autodeploy from sandbox task {}".format(self.id)
        )

    def run_dashboard_task(self, env):
        deploy_task_params = NANNY_PARAMS[env]
        deploy_task_params.update({
            'deployment_task_id': self.Context.shard_task_id,
            'deployment_release_status': {
                TEST: "testing",
                PRIEMKA: "prestable",
                PROD: "stable",
            }[env],
            'vault_name': 'nanny_oauth_token',
            'vault_owner': "YASAP",
            'deployment_nanny_bool_wait': True,
            'semaphore_name': 'COLLECTIONS_CARD_RECOMMENDER_DEPLOY'
        })
        deploy_task = sdk2.Task[deploy_nanny_dashboard.DeployNannyDashboard.type](
            self,
            description='Run {} deployment.'.format(env),
            **deploy_task_params
        )
        deploy_task.save().enqueue()
        self.Context.deploy_task_id = deploy_task.id
        self.Context.save()
        raise sdk2.WaitTask(
            [self.Context.deploy_task_id],
            ctt.Status.Group.FINISH | ctt.Status.Group.BREAK,
            wait_all=True
        )

    def deploy(self, env):
        self.release_shard_task(TASK_RELEASE_TYPE[env])
        self.run_dashboard_task(env)

    def make_acceptance(self):
        acceptance_task = ShardAcceptanceTest.CardRecommenderShardAcceptanceTest(
            self,
            description='Autodeploy acceptance test for shard {}'.format(self.Context.shard_task_id),
            items_max_delta=self.Parameters.items_max_delta,
            relevance_max_delta=self.Parameters.relevance_max_delta,
            priemka_address=self.Parameters.priemka_address,
            production_address=self.Parameters.production_address,
            exp_name=self.Parameters.exp_name
        )
        acceptance_task.save().enqueue()
        self.Context.acceptance_task_id = acceptance_task.id
        self.Context.save()
        raise sdk2.WaitTask(
            [self.Context.acceptance_task_id],
            ctt.Status.Group.FINISH | ctt.Status.Group.BREAK,
            wait_all=True
        )

    def check_deploy_task(self):
        if self.Context.deploy_task_id is not NotExists:
            child = sdk2.Task[self.Context.deploy_task_id]
            if child.status != ctt.Status.SUCCESS:
                raise common.errors.TaskFailure("Deploy task was finished with status {}".format(child.status))

    def check_acceptance_task(self):
        if self.Context.acceptance_task_id is not NotExists:
            child = sdk2.Task[self.Context.acceptance_task_id]
            if child.status != ctt.Status.SUCCESS:
               raise common.errors.TaskFailure("Acceptance task was finished with status {}".format(child.status))

    def on_execute(self):
        with self.memoize_stage.find_shard:
            self.find_shard_task_and_resource()

        with self.memoize_stage.deploy_acceptance:
            if self.Parameters.acceptance_testing and self.Parameters.shard_env == PROD:
                self.deploy(PRIEMKA)

        with self.memoize_stage.make_acceptance:
            if self.Parameters.acceptance_testing and self.Parameters.shard_env == PROD:
                self.check_deploy_task()
                self.make_acceptance()

        with self.memoize_stage.final_deploy:
            if self.Parameters.deploy_shard:
                self.check_acceptance_task()
                self.deploy(self.Parameters.shard_env)

        with self.memoize_stage.check_deploy:
            self.check_deploy_task()
