from sandbox import sdk2

from sandbox.projects.collections.resources import CollectionsCardRecommenderResponses
from sandbox.projects.collections.resources import CollectionsCardRecommenderResponsesDiff

import json


class CardRecommenderCompareResponses(sdk2.Task):
    """ Task for comparing Card Recommender responses """

    class Parameters(sdk2.Task.Parameters):
        task1_resource = sdk2.parameters.Resource(
            "Resource with Card Recommender responses",
            resource_type=CollectionsCardRecommenderResponses,
            required=True
        )
        task2_resource = sdk2.parameters.Resource(
            "Resource with Card Recommender responses",
            resource_type=CollectionsCardRecommenderResponses,
            required=True
        )

    def load_responses(self, file_path):
        with open(file_path, 'r') as f:
            raw_responses = f.readlines()
            return [json.loads(r.strip()) for r in raw_responses]

    def build_diff_message(self, field_name, card1, card2):
        return "Different value of field '{}'. Left: {}. Right: {}".format(
            field_name,
            card1[field_name],
            card2[field_name]
        )

    def compare_cards(self, card1, card2):
        for field in ("id", "dup_id", "channel_id", "board_id", "semidup_id"):
            if (field in card1) != (field in card2):
                return "Different presence of field '{}' in responses".format(field)
            if field in card1 and card1[field] != card2[field]:
                return self.build_diff_message(field, card1, card2)

        if abs(card1["rel"] - card2["rel"]) > 1e-5:
            return self.build_diff_message("rel", card1, card2)

    def compare_responses(self, response1, response2):
        cards1 = response1["cards"]
        cards2 = response2["cards"]

        if len(cards1) != len(cards2):
            return "Different count of cards in responses. Left: {}. Right: {}".format(
                len(cards1),
                len(cards2)
            )

        for j in range(0, len(cards1)):
            cards_diff = self.compare_cards(cards1[j], cards2[j])
            if cards_diff is not None:
                return cards_diff

        return None

    def calc_diff(self, responses1, responses2):
        diff = []

        if len(responses1) != len(responses2):
            diff.append("Different count of responses")
            return diff

        for i in range(0, len(responses1)):
            response_diff = self.compare_responses(responses1[i], responses2[i])
            if response_diff is not None:
                diff.append("Diff in responses number {}. {}".format(i, response_diff))

        return diff

    def on_execute(self):
        task1_resource_data = sdk2.ResourceData(self.Parameters.task1_resource)
        task2_resource_data = sdk2.ResourceData(self.Parameters.task2_resource)

        responses1 = self.load_responses(str(task1_resource_data.path))
        responses2 = self.load_responses(str(task2_resource_data.path))

        diff = self.calc_diff(responses1, responses2)

        if len(diff) > 0:
            diff_resource = CollectionsCardRecommenderResponsesDiff(
                self,
                "Card recommender responses diff",
                "responses_diff.txt"
            )
            diff_resouce_data = sdk2.ResourceData(diff_resource)
            with open(str(diff_resouce_data.path), 'w') as diff_file:
                diff_file.write(json.dumps(diff))
            diff_resouce_data.ready()

        self.Context.has_diff = len(diff) > 0
