import json
import logging
import os

from sandbox import common
import sandbox.common.types.misc as ctm

from sandbox.sandboxsdk import parameters
from sandbox.sandboxsdk.channel import channel

from sandbox.projects.collections.informers import resource_types as informers_resource
from sandbox.projects.common import utils
from sandbox.projects.common import cgroup as cgroup_api
from sandbox.projects.common.search import components as search_components
from sandbox.projects.common.search import settings as search_settings
from sandbox.projects.common.search.components import component as components_common


class Daemon(components_common.ProcessComponentMixinWithShutdown,
             components_common.WaitPortComponentMixin,
             components_common.Component):
    """Basic class for all new style components"""

    def __init__(self, args, start_timeout=60, cgroup=None):
        self.port = components_common.try_get_free_port()
        self.__name = os.path.basename(args[0])
        self.__cgroup = cgroup
        logging.info("Using port {} for {}".format(self.port, self.__name))

        components_common.ProcessComponentMixinWithShutdown.__init__(
            self,
            args=[a.format(port=self.port) for a in args],
            shutdown_url="http://localhost:{}{}".format(self.port, self.get_shutdown_path_and_query()),
            log_prefix=self.__name,
            preexec_fn=lambda: self.__cgroup.set_current() if self.__cgroup else None
        )
        components_common.WaitPortComponentMixin.__init__(
            self,
            endpoints=[("localhost", self.port)],
            wait_timeout=start_timeout
        )

    def get_shutdown_path_and_query(self):
        return '/admin?action=shutdown'

    def set_cgroup(self, cgroup):
        """
            Change component's cgroup

            Used by sanitizer library to reset cgroup
        """
        self.__cgroup = cgroup

    def replace_config_parameter(self, *args, **kwargs):
        """
            Useless method called by sanitizer library
        """
        pass


class InformersDaemon(Daemon):
    """
        New experimental component based on common mixins
    """

    def __init__(self,
                 binary_path,
                 config_path,
                 default_channels_path,
                 back_mongo_url,
                 start_timeout=search_components.DEFAULT_START_TIMEOUT,
                 shutdown_timeout=120,
                 cgroup=None):

        args = [
            binary_path,
            '--port', '{port}',
            '--config', config_path,
            '--default-channels-path', default_channels_path,
            '--back-mongo-url', back_mongo_url
        ]
        Daemon.__init__(self, args, cgroup=cgroup, start_timeout=start_timeout)

    def get_shutdown_path_and_query(self):
        return '/shutdown'


def create_informers_daemon_params(n='', group_name=None, component_name=None):
    if group_name is None:
        group_name = "InformersDaemon{} parameters".format(n)

    if component_name is None:
        component_name = "informersdaemon{}".format(n)

    class Parameters:
        class Binary(parameters.ResourceSelector):
            name = '{}_executable_resource_id'.format(component_name)
            description = 'Infoerms executable:'
            resource_type = informers_resource.CollectionsInformers
            group = group_name
            required = True

        class Config(parameters.ResourceSelector):
            name = '{}_config_resource_id'.format(component_name)
            description = 'Informers config:'
            resource_type = informers_resource.CollectionsInformersConfig
            group = group_name
            required = True

        class BackMongoUrl(parameters.SandboxStringParameter):
            name = '{}_back_mongo_url'.format(component_name)
            description = 'BackendMongo url vault'
            group = group_name
            required = True

        class StartTimeout(parameters.SandboxIntegerParameter):
            name = '{}_start_timeout'.format(component_name)
            description = 'Start timeout (sec)'
            group = group_name
            default_value = search_components.DEFAULT_START_TIMEOUT

        class Cgroup(parameters.SandboxStringParameter):
            name = '{}_cgroup'.format(component_name)
            description = 'Cgroup'
            group = group_name

        params = (Binary, Config, BackMongoUrl, StartTimeout, Cgroup)

    return Parameters


def get_informers_daemon(params=create_informers_daemon_params(), **kwargs):
    task = channel.task
    cgroup = _create_basesearch_cgroup(
        search_settings.INDEX_ALL,
        utils.get_or_default(task.ctx, params.Cgroup)
    )

    return InformersDaemon(
        binary_path=task.sync_resource(task.ctx[params.Binary.name]),
        config_path=task.sync_resource(task.ctx[params.Config.name]),
        back_mongo_url=task.get_vault_data(task.ctx[params.BackMongoUrl.name]),
        start_timeout=utils.get_or_default(task.ctx, params.StartTimeout),
        cgroup=cgroup,
        **kwargs
    )


def _create_basesearch_cgroup(index_type, cgroup_props):
    if common.config.Registry().common.installation == ctm.Installation.LOCAL:
        logging.info("Skipping cgroups creation on local sandbox")
        return None

    if cgroup_props:
        cgroup_props = json.loads(cgroup_props)
    else:
        cgroup_props = search_settings.ImagesSettings.CGROUPS[index_type]

    return cgroup_api.create_cgroup(index_type, cgroup_props)
