import base64
import logging
import os.path
import tarfile
import urllib2

from sandbox.sandboxsdk import errors
from sandbox.sandboxsdk import parameters
from sandbox.sandboxsdk import paths
from sandbox.sandboxsdk import task

from sandbox.projects import resource_types
from sandbox.projects.common import error_handlers as eh
from sandbox.projects.common.search import requester_compat as search_requester
from sandbox.projects.collections.daemons import components as daemons_components


INFORMERS_DAEMON_PARAMS = daemons_components.create_informers_daemon_params()


class QueriesParameter(parameters.ResourceSelector):
    name = 'queries_resource_id'
    description = 'Queries'
    required = False
    resource_type = resource_types.PLAIN_TEXT_QUERIES


class QueriesLimitParameter(parameters.SandboxIntegerParameter):
    name = 'queries_limit'
    description = 'Limit number of queries to read (0 = all)'
    default_value = 0


class BaseGetResponsesTask(task.SandboxTask):
    """
        Base class for get responses from daemon
    """
    input_parameters = \
        (QueriesParameter, QueriesLimitParameter) + \
        search_requester.create_params()

    allowed_response_http_codes = [200]

    def on_enqueue(self):
        self.create_resource(
            self.descr,
            self._get_responses_dir(),
            resource_types.COLLECTIONS_RESPONSES_ARCHIVE
        )

    def on_execute(self):
        daemon = self._get_daemon()
        with daemon:
            self._save_responses(daemon)

    def _get_responses_dir(self):
        return self.abs_path("responses")

    def _get_daemon(self):
        raise NotImplementedError()

    # TODO: Remove useless tarball 'data.tar'
    def _save_responses(self, component):
        responses_dir = self._get_responses_dir()
        data_dir = os.path.join(responses_dir, "data")
        meta_path = os.path.join(responses_dir, "meta.txt")
        paths.make_folder(data_dir, delete_content=True)

        url_prefix = "http://localhost:{}".format(component.port)
        queries_path = self.sync_resource(self.ctx[QueriesParameter.name])
        queries_limit = self.ctx[QueriesLimitParameter.name]
        eh.ensure(os.stat(queries_path).st_size, "Input file with queries is empty")

        queries_iterator = self._generate_queries(queries_path, queries_limit, url_prefix)

        with open(meta_path, "w") as meta_file:
            for _, status, data in search_requester.save_responses(self.ctx, queries_iterator, url_read=url_read):
                if not status:
                    raise errors.SandboxTaskFailureError("Error during request processing: {}".format(data))
                url, http_headers, http_data, http_code = data
                if http_code not in self.allowed_response_http_codes:
                    raise errors.SandboxTaskFailureError("Error during request processing: {}".format(http_data))
                url = url[len(url_prefix):]

                meta_file.write("-- {} {}\n".format(url, http_code))
                if len(http_headers) > 0:
                    meta_file.write("{}\n".format(http_headers))
                with open(os.path.join(data_dir, base64.b32encode(url)), "w") as data_file:
                    data_file.write(http_data)

        with tarfile.open(os.path.join(responses_dir, "data.tar.gz"), "w|gz") as tar:
            tar.add(data_dir, "data")
        paths.remove_path(data_dir)

    def _generate_queries(self, queries_path, queries_limit, url_prefix):
        try:
            with open(queries_path) as queries_file:
                for num, url in enumerate(queries_file, 1):
                    yield url_prefix + url.strip()
                    if queries_limit and num == queries_limit:
                        break
        except Exception as e:
            logging.info("Problem during query generation: {}".format(e))
            yield None


def url_read(args):
    url, timeout, max_retries = args
    if url is None:
        return False, "Problem in queries generator"

    except_msg = ""
    for attempt in xrange(max_retries + 1):
        try:
            response = urllib2.urlopen(url, timeout=timeout)
            headers = sorted("{}: {}".format(k.lower(), v) for k, v in response.info().items())
            answer = (url, "\n".join(headers), response.read(), response.getcode())
            return True, answer
        except urllib2.HTTPError as e:
            return True, (url, "", str(e), e.code)
        except Exception as e:
            logging.debug("Problems with request '{}' (attempt {})".format(url, attempt))
            except_msg = str(e)
    logging.debug("Unresolved problems with request '{}':\n{}".format(url, except_msg))
    return False, except_msg


class BaseInformersDaemonTask(task.SandboxTask):
    input_parameters = INFORMERS_DAEMON_PARAMS.params

    def on_enqueue(self):
        task.SandboxTask.on_enqueue(self)

    def _get_daemon(self):
        return daemons_components.get_informers_daemon()
