from sandbox import sdk2
from sandbox import common

from sandbox.common.types.misc import NotExists
import sandbox.common.types.task as ctt
from sandbox.projects import DeployNannyDashboard as deploy_nanny_dashboard
from sandbox.projects.collections.resources import CollectionsFeedReaderPumpkinShard
from sandbox.projects.common.nanny import nanny

import logging


TEST = 'test'
PRIEMKA = 'priemka'
PROD = 'prod'

TASK_RELEASE_TYPE = {
    TEST: ctt.ReleaseStatus.TESTING,
    PRIEMKA: ctt.ReleaseStatus.PRESTABLE,
    PROD: ctt.ReleaseStatus.STABLE
}

NANNY_PROD_DEPLOYMENT_PARAMS = {
    'deployment_nanny_dashboard_name': 'pdb_production',
    'deployment_nanny_dashboard_recipe': 'deploy_hotfeed_reader_service_by_locations_without_closing_locations',
    'deployment_nanny_dashboard_filter': 'feed_reader',
}


def ensure_resource(task_id, resource_type):
    resource = resource_type.find(task_id=task_id).first()
    if resource is None:
        raise common.errors.TaskFailure("Can't find resource " + resource_type.name)


def get_services_from_dashboard(nanny_client):
    return next(
        dashboard_group['services']
        for dashboard_group
        in nanny_client.get_dashboard(NANNY_PROD_DEPLOYMENT_PARAMS['deployment_nanny_dashboard_name'])['groups']
        if dashboard_group['id'] == NANNY_PROD_DEPLOYMENT_PARAMS['deployment_nanny_dashboard_filter'])


def check_current_snapshot(nanny_client, service_id):
    head_snapshot = nanny_client.get_service_runtime_attrs(service_id)['_id']
    active_snapshots = nanny_client.get_service_current_state(service_id)['content']['active_snapshots']
    head_snapshot_state = next((
        snapshot['state']
        for snapshot in active_snapshots
        if snapshot['snapshot_id'] == head_snapshot),
        None
    )
    if head_snapshot_state != 'ACTIVE':
        raise common.errors.TaskFailure(
            'Not safe to deploy service {}: current snapshot is in state {}'.format(
                service_id, head_snapshot_state or 'absent')
        )
    logging.info('Service {} is in ACTIVE state.'.format(service_id))


class CollectionsFeedReaderDeployPumpkinShard(sdk2.Task):
    """ Task for feed reader pumpkin shard deployment """

    class Parameters(sdk2.Task.Parameters):
        build_pumpkin_shard_task_id = sdk2.parameters.Integer('Build pumpkin shard task', required=True)
        deploy_env = sdk2.parameters.String(
            'Deploy environment',
            required=True,
            choices=[('Test', TEST), ('Prestable', PRIEMKA), ('Production', PROD)]
        )
        nanny_token_vault = sdk2.parameters.String('Nanny oauth token vault name', required=True)
        nanny_token_owner = sdk2.parameters.String('Nanny oauth token vault owner', required=True)

    def find_shard_task_and_resource(self):
        shard_task = sdk2.Task.find(id=self.Parameters.build_pumpkin_shard_task_id).first()
        if shard_task is None:
            raise common.errors.TaskFailure("Can't find build shard task")

        self.Context.shard_task_id = shard_task.id

        shard_resource = sdk2.Resource.find(
            CollectionsFeedReaderPumpkinShard,
            task=shard_task
        ).first()
        if shard_resource is None:
            raise common.errors.TaskFailure("Can't find shard resource")
        self.Context.shard_resource_id = shard_resource.id

        self.Context.save()

    def create_nanny_client(self):
        nanny_token = sdk2.Vault.data(self.Parameters.nanny_token_owner, self.Parameters.nanny_token_vault)
        return nanny.NannyClient(
            api_url='http://nanny.yandex-team.ru/',
            oauth_token=nanny_token,
        )

    def validate_current_service_state(self):
        """Check feed reader production services.

        It is safe to deploy only if HEAD snapshot is in ACTIVE state.
        Otherwise it is possible to deploy invalid service configuration.
        """
        nanny_client = self.create_nanny_client()
        for service_id in get_services_from_dashboard(nanny_client):
            check_current_snapshot(nanny_client, service_id)

    def release_shard_task(self, release_type):
        self.server.release(
            task_id=self.Context.shard_task_id,
            type=release_type,
            subject='Feed pumkin autodeploy from sandbox task {}'.format(self.id)
        )

    def run_dashboard_task(self):
        deploy_task_params = NANNY_PROD_DEPLOYMENT_PARAMS
        deploy_task_params.update({
            'deployment_task_id': self.Context.shard_task_id,
            'deployment_release_status': 'stable',
            'vault_name': self.Parameters.nanny_token_vault,
            'vault_owner': self.Parameters.nanny_token_owner,
            'deployment_nanny_bool_wait': True,
        })
        deploy_task = sdk2.Task[deploy_nanny_dashboard.DeployNannyDashboard.type](
            self,
            description='Run production deployment.',
            **deploy_task_params
        )
        deploy_task.save().enqueue()
        self.Context.deploy_task_id = deploy_task.id
        self.Context.save()
        raise sdk2.WaitTask(
            [self.Context.deploy_task_id],
            ctt.Status.Group.FINISH | ctt.Status.Group.BREAK,
            wait_all=True
        )

    def deploy(self, env):
        if env == PROD:
            self.validate_current_service_state()
        self.release_shard_task(TASK_RELEASE_TYPE[env])
        if env == PROD:
            self.run_dashboard_task()

    def check_deploy_task(self):
        if self.Context.deploy_task_id is not NotExists:
            child = sdk2.Task[self.Context.deploy_task_id]
            if child.status != ctt.Status.SUCCESS:
                raise common.errors.TaskFailure('Deploy task has finished with status {}'.format(child.status))

    def on_execute(self):
        with self.memoize_stage.find_shard:
            self.find_shard_task_and_resource()

        with self.memoize_stage.deploy:
            self.deploy(self.Parameters.deploy_env)

        with self.memoize_stage.check_deploy:
            self.check_deploy_task()
