from sandbox import sdk2
from sandbox.sandboxsdk import environments

from sandbox.sdk2.helpers import subprocess

from sandbox.projects import resource_types
from sandbox.projects.common import dolbilka

import logging
import os
import sys


class GenerateFeedReaderRequests(sdk2.Task):
    class Parameters(sdk2.Task.Parameters):
        cluster = sdk2.parameters.String('YT server name', default='hahn', required=True)
        yt_token_vault = sdk2.parameters.String('YT token vault', required=True)
        table = sdk2.parameters.String('YT table with feed reader logs', default='')
        requests_count = sdk2.parameters.Integer("Requests count", default=1000)
        rps = sdk2.parameters.Integer("RPS", default=10)
        filtering_coefficient = sdk2.parameters.Float('Filtering coefficient', default=0.2, required=True)

    class Requirements(sdk2.Task.Requirements):
        environments = (
            environments.PipEnvironment('yandex-yt'),
            environments.PipEnvironment('yandex-yt-yson-bindings-skynet')
        )

    def generate_requests(self, requests_file):
        yt_token = sdk2.Vault.data(self.Parameters.yt_token_vault)

        import yt.wrapper as yt
        yt.config.set_proxy(self.Parameters.cluster)
        yt.config['token'] = yt_token

        log_table = self.Parameters.table
        row_count = yt.row_count(log_table)
        if row_count == 0:
            raise RuntimeError("Empty input table")

        logging.info('Start worker')

        file_path = os.path.join(os.path.abspath(os.path.dirname(__file__)), 'worker.py')

        env = os.environ.copy()
        env['YT_TOKEN'] = yt_token

        with sdk2.helpers.ProcessLog(self, logger=logging.getLogger('worker')) as pl:
            return_code = subprocess.Popen([
                sys.executable, file_path,
                '--table', log_table,
                '--cluster', self.Parameters.cluster,
                '--requests_file', requests_file,
                '--requests_count', str(self.Parameters.requests_count),
                '--filtering_coefficient', str(self.Parameters.filtering_coefficient),
            ],
                stdout=pl.stdout, stderr=subprocess.STDOUT, env=env).wait()

            if return_code == 0:
                logging.info('Worker finished successfully')
            else:
                logging.error('Subprocess ended with error: %s', return_code)
                raise Exception('Subprocess failed!')

    def on_execute(self):
        plain_requests_resource = resource_types.PLAIN_TEXT_QUERIES(
            self,
            'Requests in plain text format',
            'requests.txt',
            collections_feed_reader=True
        )
        plain_requests_resource_data = sdk2.ResourceData(plain_requests_resource)
        self.generate_requests(str(plain_requests_resource_data.path))
        plain_requests_resource_data.ready()

        basesearch_plan_resouce = resource_types.BASESEARCH_PLAN(
            self,
            'Requests in dolbilka plan format',
            'requests.plan',
            collections_feed_reader=True
        )
        basesearch_plan_resouce_data = sdk2.ResourceData(basesearch_plan_resouce)
        dolbilka.convert_queries_to_plan(
            str(plain_requests_resource_data.path),
            str(basesearch_plan_resouce_data.path),
            rps=self.Parameters.rps)
        basesearch_plan_resouce_data.ready()
