from sandbox import common
from sandbox import sdk2
from sandbox.projects import resource_types
from sandbox.projects.collections.resources import CollectionsFeedReaderPumpkin
from sandbox.projects.collections.resources import CollectionsFeedReaderPumpkinShard
from sandbox.sandboxsdk.network import is_port_free
from sandbox.sandboxsdk.svn import Arcadia
from sandbox.sdk2.helpers import subprocess as sp

from requests.adapters import HTTPAdapter
from requests.packages.urllib3.util.retry import Retry

import logging
import re
import requests
import tarfile
import time


def find_resource(task_id, resource_type):
    resource = resource_type.find(task_id=task_id).first()
    if resource is None:
        raise common.errors.TaskFailure("Can't find resource " + resource_type.name)
    return resource


def fetch_responses(port, requests_path, requests_count, tolerance):
    retry = Retry(
        total=3,
        backoff_factor=0.3,
    )
    session = requests.Session()
    adapter = HTTPAdapter(max_retries=retry)
    session.mount('http://localhost', adapter)

    address = 'http://localhost:' + str(port)

    logging.info('Start shooting pumpkin')

    total_requests_count = 0
    bad_requests_count = 0

    with open(requests_path, 'r') as requests_file:
        for i in range(requests_count):

            url = requests_file.readline()
            if not url:
                break
            total_requests_count += 1

            response = session.get(address + url)
            if not response.ok or not response.json()['cards']:
                logging.error(
                    'Error in request processing. Url: {}. Request number: {}. Code: {}. Response: {}'.format(
                        url,
                        i,
                        response.status_code,
                        response.content
                    )
                )
                bad_requests_count += 1
            else:
                logging.info('Response #{} contains {} cards'.format(i, len(response.json()['cards'])))

    if bad_requests_count > total_requests_count * tolerance:
        raise common.errors.TaskFailure('Too many bad responses')


class Pumpkin:
    def __init__(self, task, binary_path, config_path):
        self.process_log = sdk2.helpers.ProcessLog(task, logger='pumpkin')
        self.process = sp.Popen([
            binary_path,
            '-c', config_path,
        ],
            stdout=self.process_log.stdout, stderr=sp.STDOUT,
        )
        self.port = self.get_port(config_path)
        while is_port_free(self.port):
            time.sleep(1)
            if self.process.poll() is not None:
                raise RuntimeError('Error in pumpkin starting')

    @staticmethod
    def get_port(config_path):
        with open(config_path, 'r') as f:
            return re.search('(?<=Port: )\d*', f.read()).group(0)

    def close(self):
        self.process.terminate()


class FeedReaderPumpkinShardAcceptance(sdk2.Task):
    """ Task for feed reader pumpkin shard acceptance testing"""

    class Parameters(sdk2.Task.Parameters):
        requests_task_id = sdk2.parameters.Integer(
            'Generate feed reader requests task', required=True)
        build_pumpkin_task_id = sdk2.parameters.Integer(
            'Build feed reader pumpkin task', required=True)
        build_pumpkin_shard_task_id = sdk2.parameters.Integer(
            'Build pumpkin shard task', required=True)
        pumpkin_config_arcadia_path = sdk2.parameters.String(
            'Path to arcadia file with pumpkin config',
            default='arcadia:/arc/trunk/arcadia/yweb/yasap/pdb/quality/hot_feed/tools/pumpkin/config.pbtxt',
            required=True)
        requests_count = sdk2.parameters.Integer(
            'Count of requests for testing', default=10000, required=True)
        tolerance = sdk2.parameters.Float('Bad responses share', default=0.001, required=True)

    def on_execute(self):
        requests_file = str(sdk2.ResourceData(find_resource(
            self.Parameters.requests_task_id,
            resource_types.PLAIN_TEXT_QUERIES
        )).path)

        pumpkin_binary = str(sdk2.ResourceData(find_resource(
            self.Parameters.build_pumpkin_task_id,
            CollectionsFeedReaderPumpkin
        )).path)

        pumpkin_shard = str(sdk2.ResourceData(find_resource(
            self.Parameters.build_pumpkin_shard_task_id,
            CollectionsFeedReaderPumpkinShard
        )).path)
        with tarfile.open(pumpkin_shard, "r:gz") as tar:
            tar.extractall()

        config_path = 'config.pbtxt'
        Arcadia.export(self.Parameters.pumpkin_config_arcadia_path, config_path)

        pumpkin = Pumpkin(self, pumpkin_binary, config_path)

        fetch_responses(pumpkin.port, requests_file, self.Parameters.requests_count, self.Parameters.tolerance)

        pumpkin.close()
