import logging
import os
import tarfile

from sandbox import common
from sandbox import sdk2

from sandbox.projects.common.nanny import nanny
from sandbox.sandboxsdk.svn import Arcadia
import sandbox.common.types.task as ctt


class BuildMasterItemRecommenderBundle(nanny.ReleaseToNannyTask2, sdk2.Task):
    """ Task for building master item recommender bundle """

    class Parameters(sdk2.Task.Parameters):
        bundle_type = sdk2.parameters.String(
            "Bundle type (test/priemka/prod)",
            required=True
        )

        bundle_resource_name = sdk2.parameters.String(
            "Sandbox resource name for bundle",
            required=True
        )

        svn_revision = sdk2.parameters.Integer("SVN revision", required=False)

        binary_resource = sdk2.parameters.Resource(
            "Resource with binary",
            required=False
        )

        binary_resource_name = sdk2.parameters.String(
            "Sandbox resource name for binary",
            default="ARCADIA_PROJECT",
            required=False
        )

        binary_name = sdk2.parameters.String(
            "Binary artifact name",
            required=True
        )

        arcadia_binary_path = sdk2.parameters.String(
            "Path to arcadia project for binary",
            required=True
        )

        arcadia_config_path = sdk2.parameters.String(
            "Path to arcadia folder with config",
            required=True
        )

        arcadia_resource_path = sdk2.parameters.String(
            "Path to arcadia folder with resources",
            required=True
        )

    def build_binary(self):
        logging.info("Start child task for building binary...")

        build_task = sdk2.Task["YA_MAKE"](
            self,
            description="Build master item recommender binary " + self.Parameters.binary_name,
            owner=self.owner,
            priority=self.Parameters.priority,
            checkout_arcadia_from_url="arcadia:/arc/trunk/arcadia@" + str(self.Parameters.svn_revision),
            arts=os.path.join(self.Parameters.arcadia_binary_path, self.Parameters.binary_name),
            targets=self.Parameters.arcadia_binary_path,
            result_rt=self.Parameters.binary_resource_name,
            result_rd="Master item recommender binary",
            result_single_file=True,
            checkout_mode="auto",
            build_type="release",
            build_arch="linux",
        )
        build_task.save().enqueue()
        self.Context.build_task_id = build_task.id
        self.Context.save()
        raise sdk2.WaitTask([self.Context.build_task_id],
                ctt.Status.Group.FINISH | ctt.Status.Group.BREAK, wait_all=True)

    def find_binary_resource_data(self):
        if self.Parameters.binary_resource:
            return sdk2.ResourceData(self.Parameters.binary_resource)
        child = sdk2.Task[self.Context.build_task_id]
        if child.status != ctt.Status.SUCCESS:
            raise common.errors.TaskFailure("Binary build task was finished with status {}".format(child.status))
        resource = sdk2.Resource[self.Parameters.binary_resource_name].find(task=child).first()
        return sdk2.ResourceData(resource)


    def add_config_file(self, tar):
        file_name = "config_" + self.Parameters.bundle_type + ".pbtxt"
        Arcadia.export("arcadia:/arc/trunk/arcadia/" + self.Parameters.arcadia_config_path + "/" + file_name + "@" + str(self.Parameters.svn_revision),
            "./config.pbtxt")
        tar.add("./config.pbtxt", arcname="./config.pbtxt")

    def add_folder(self, tar, folder_name):
        Arcadia.export("arcadia:/arc/trunk/" + self.Parameters.arcadia_resource_path + "/" + folder_name + "@" + str(self.Parameters.svn_revision),
            "./" + folder_name + "/")
        tar.add("./" + folder_name + "/", arcname="resources")

    def add_resource_files(self, tar):
        self.add_folder(tar, "any")
        self.add_folder(tar, self.Parameters.bundle_type)

    def create_bundle(self):
        logging.info("Start bundle building...")

        resource = sdk2.Resource[self.Parameters.bundle_resource_name](
            self, "Master item recommender bundle", "bundle.tar.gz")
        resource_data = sdk2.ResourceData(resource)

        tar = tarfile.open(str(resource_data.path), 'w:gz')
        tar.add(str(self.find_binary_resource_data().path), arcname=self.Parameters.binary_name)
        self.add_config_file(tar)
        self.add_resource_files(tar)
        tar.close()

        resource_data.ready()

    def on_execute(self):
        with self.memoize_stage.create_subtasks:
            self.build_binary()

        with self.memoize_stage.create_bundle:
            self.create_bundle()

    def on_release(self, additional_parameters):
        nanny.ReleaseToNannyTask2.on_release(self, additional_parameters)
        sdk2.Task.on_release(self, additional_parameters)
