from sandbox import sdk2
from sandbox.sandboxsdk import environments

from sandbox.sdk2.helpers import subprocess

from sandbox.projects import resource_types
from sandbox.projects.common import dolbilka

import logging
import json
import os
import sys


class GenerateRequestsBase(sdk2.Task):
    class Parameters(sdk2.Task.Parameters):
        yt_proxy = sdk2.parameters.String("YT server name", default="hahn", required=True)
        yt_pool = sdk2.parameters.String("YT pool", required=True)
        yt_token_vault = sdk2.parameters.String('YT token vault', required=True)
        log_folder = sdk2.parameters.String("YT dir with log tables", default="")
        log_table = sdk2.parameters.String("YT table with logs", default="")
        requests_configs = sdk2.parameters.JSON("Requests configs", required=True)
        rps = sdk2.parameters.Integer("RPS", default=10)

    class Requirements(sdk2.Task.Requirements):
        environments = (
            environments.PipEnvironment('yandex-yt'),
            environments.PipEnvironment("yandex-yt-yson-bindings-skynet")
        )

    def get_log_table(self):
        import yt.wrapper as yt
        yt_table = self.Parameters.log_table
        if len(yt_table) == 0:
            if len(self.Parameters.log_folder) == 0:
                raise RuntimeError("Select log dir or table.")
            list_dir = sorted(yt.list(self.Parameters.log_folder), reverse=True)
            if len(list_dir) == 0:
                raise RuntimeError("Empty dir")
            yt_table = yt.ypath_join(self.Parameters.log_folder, list_dir[0])

        return yt_table

    def generate_requests(self):
        yt_token = sdk2.Vault.data(self.Parameters.yt_token_vault)

        import yt.wrapper as yt
        yt.config.set_proxy(self.Parameters.yt_proxy)
        yt.config['token'] = yt_token

        log_table = self.get_log_table()

        row_count = yt.row_count(log_table)
        if row_count == 0:
            raise RuntimeError("Empty input table")

        logging.info('Start worker')

        env = os.environ.copy()
        env["YT_TOKEN"] = yt_token

        with sdk2.helpers.ProcessLog(self, logger=logging.getLogger("worker")) as pl:
            return_code = subprocess.Popen([
                    sys.executable, self.worker_file_path,
                    "--input_table", log_table,
                    "--server", self.Parameters.yt_proxy,
                    "--pool", self.Parameters.yt_pool,
                    "--requests_configs", json.dumps(self.Parameters.requests_configs)
                ],
                stdout=pl.stdout, stderr=subprocess.STDOUT, env=env).wait()

            if return_code == 0:
                logging.info("Worker finished successfully")
            else:
                logging.error("Subprocess ended with error: %s", return_code)
                raise Exception("Subprocess failed!")

    def on_execute(self):
        requests_resources_data = list()
        for config in self.Parameters.requests_configs["plan_configs"]:
            name = config["name"]
            resource = resource_types.PLAIN_TEXT_QUERIES(self,
                "Requests in plain text format",
                name + ".txt",
                **{self.service_name + "_" + name + "_plan" : True})
            data = sdk2.ResourceData(resource)
            requests_resources_data.append(data)
            config["file_path"] = str(data.path)

        self.generate_requests()
        logging.info(json.dumps(self.Parameters.requests_configs))

        for resource_data in requests_resources_data:
            resource_data.ready()

        for i, config in enumerate(self.Parameters.requests_configs["plan_configs"]):
            name = config["name"]
            resource = resource_types.BASESEARCH_PLAN(self,
                "Requests in dolbilka text format",
                name + ".plan",
                **{self.service_name + "_" + name + "_plan" : True})
            data = sdk2.ResourceData(resource)
            dolbilka.convert_queries_to_plan(str(requests_resources_data[i].path),
                                         str(data.path),
                                         rps=self.Parameters.rps,
                                         loader_type='phantom')
            data.ready()
