import logging
import os

from sandbox import common
from sandbox import sdk2

from sandbox.projects.common.nanny import nanny
from sandbox.projects import resource_types
import sandbox.common.types.resource as ctr
import sandbox.common.types.task as ctt

from sandbox.projects import DeployNannyDashboard as deploy_nanny_dashboard
from sandbox.projects.dj import DjBuildRecommenderBundle


class ModelsAutoDeployBase(sdk2.Task):
    """ Base task for building services production bundle with given models archive"""

    class DefaultParameters(sdk2.Task.Parameters):
        nanny_token_vault = sdk2.parameters.String('Nanny OAuth token vault', required=True)
        service_name = sdk2.parameters.String('Service with production bundle', required=True)

    def create_nanny_client(self):
        nanny_token = sdk2.Vault.data('YASAP', self.Parameters.nanny_token_vault)
        return nanny.NannyClient(
            api_url='http://nanny.yandex-team.ru/',
            oauth_token=nanny_token,
        )

    def find_production_bundle_task(self, nanny_client):
        current_resources = nanny_client.get_service_resources(self.Parameters.service_name)
        for files in current_resources['content']['sandbox_files']:
            if files['resource_type'] == self.resource_type_str:
                return sdk2.Task[files['task_id']]
        logging.warning("Can not find bundle resource")
        return None

    def prepare_bundle_params(self):
        logging.info("Creating nanny client...")
        nanny_client = self.create_nanny_client()

        logging.info("Search for current production bundle...")
        current_production_bundle_task = self.find_production_bundle_task(nanny_client)
        logging.info("Current production bundle task id {}".format(current_production_bundle_task.id))

        logging.info("Prepare new bundle params...")
        params = dict(current_production_bundle_task.Parameters)
        params['models_resource'] = self.Parameters.models_resource
        return params

    def run_bundle_task(self):
        logging.info("Start bundle building...")
        new_bundle_task = DjBuildRecommenderBundle.DjBuildRecommenderBundle(self, **self.prepare_bundle_params())
        new_bundle_task.save().enqueue()
        self.Context.new_bundle_task_id = new_bundle_task.id
        self.Context.save()
        raise sdk2.WaitTask([self.Context.new_bundle_task_id],
                ctt.Status.Group.FINISH | ctt.Status.Group.BREAK, wait_all=True)

    def find_new_bundle_resource(self):
        logging.info("Search for new bundle resource...")
        resource = sdk2.Resource.find(
            self.bundle_resource_type,
            task=sdk2.Task[self.Context.new_bundle_task_id]
        ).first()
        logging.info("New bundle resource id : {}".format(resource.id))
        return resource

    def find_plan_resource(self):
        logging.info("Search for last queries plan...")
        queries_resource = sdk2.Resource.find(
            resource_types.BASESEARCH_PLAN,
            state=ctr.State.READY,
            attrs=dict(collections_card_recommender_general_plan=True)
        ).first()
        logging.info("Queries plan resource id: {}".format(queries_resource.id))
        return queries_resource

    def __test_bundle__(self, test_task):
        logging.info("Start test task...")
        test_task.save().enqueue()
        self.Context.test_task_id = test_task.id
        self.Context.save()
        raise sdk2.WaitTask([self.Context.test_task_id],
                ctt.Status.Group.FINISH | ctt.Status.Group.BREAK, wait_all=True)

    def check_test_task(self):
        logging.info("Check test task status...")
        if self.Context.test_task_id is None:
            logging.info("No test task found")
            return
        child = sdk2.Task[self.Context.test_task_id]
        if child.status != ctt.Status.SUCCESS:
            raise common.errors.TaskFailure("Test task was finished with status {}".format(child.status))

    def release_bundle_task(self):
        logging.info("Release bundle task...")
        self.server.release(
            task_id=self.Context.new_bundle_task_id,
            type=ctt.ReleaseStatus.STABLE,
            subject=self.Parameters.service_name + " models autodeploy from sandbox task {}".format(self.id)
        )

    def run_dashboard_task(self):
        logging.info("Run dashboard task...")
        deploy_task_params = self.NANNY_PARAMS
        deploy_task_params.update({
            'deployment_task_id': self.Context.new_bundle_task_id,
            'deployment_release_status': 'stable',
            'vault_name': self.Parameters.nanny_token_vault,
            'vault_owner': "YASAP",
            'deployment_nanny_bool_wait': True,
            'semaphore_name': self.nanny_semaphore_name
        })
        deploy_task = sdk2.Task[deploy_nanny_dashboard.DeployNannyDashboard.type](
            self,
            description='Run production deployment.',
            **deploy_task_params
        )
        deploy_task.save().enqueue()
        self.Context.deploy_task_id = deploy_task.id
        self.Context.save()
        raise sdk2.WaitTask(
            [self.Context.deploy_task_id],
            ctt.Status.Group.FINISH | ctt.Status.Group.BREAK,
            wait_all=True
        )

    def deploy_bundle(self):
        self.release_bundle_task()
        self.run_dashboard_task()

    def check_deploy(self):
        logging.info("Check deploy task status...")
        child = sdk2.Task[self.Context.deploy_task_id]
        if child.status != ctt.Status.SUCCESS:
            raise common.errors.TaskFailure("Deploy task was finished with status {}".format(child.status))

    def release_models(self):
        logging.info("Release models task...")
        self.server.release(
            task_id=self.Parameters.models_resource.task_id,
            type=ctt.ReleaseStatus.STABLE,
            subject="Autodeploy from sandbox task {}".format(self.id)
        )

    def on_execute(self):
        with self.memoize_stage.build_bundle:
            self.run_bundle_task()

        with self.memoize_stage.test_bundle:
            self.test_bundle()

        with self.memoize_stage.deploy_bundle:
            self.check_test_task()
            self.deploy_bundle()

        with self.memoize_stage.release_models:
            self.check_deploy()
            self.release_models()
