# -*- coding: utf-8 -*-

import os

from sandbox.projects.common.build.CommonBuildTask import CommonBuildTask
import sandbox.projects.common.build.parameters as build_params
import sandbox.projects.common.constants as consts
from sandbox.sandboxsdk.errors import SandboxTaskFailureError
from sandbox.sandboxsdk.parameters import SandboxBoolParameter
from sandbox.sandboxsdk.paths import make_folder
from sandbox.sandboxsdk.svn import Arcadia


class BuildBinariesParameter(SandboxBoolParameter):
    name = 'build_binaries'
    description = 'Build binaries'
    default_value = True


class BuildConfigsParameter(SandboxBoolParameter):
    name = 'build_configs'
    description = 'Build configs'
    default_value = True


class BaseBinariesAndConfigsTask(CommonBuildTask):
    """
        Таск для сборки релизных бандлов поисковых программ.
    """

    input_parameters = build_params.get_common_build_params() + [BuildBinariesParameter, BuildConfigsParameter]

    BUILD_CONFIGS_KEY = 'build_configs'

    def initCtx(self):
        CommonBuildTask.initCtx(self)
        self.ctx[consts.BUILD_BUNDLE_KEY] = True

    def get_target_configs(self):
        """
            Список описаний (тип ресурса, имя результата, путь в Аркадии) конфигов, которые нужно собрать.
        """
        result = []
        config_types_list = getattr(self, 'TARGET_CONFIG_TYPES', [])
        for config_type in config_types_list:
            config_path = config_type.arcadia_path
            config_name = os.path.split(config_path)[1]
            result.append((config_type, config_name, config_path))

        return result

    def get_configs_dir(self):
        return self.abs_path('configs')

    def is_build_binaries_task(self):
        return self.ctx[BuildBinariesParameter.name]

    def is_build_configs_task(self):
        return self.ctx[BuildConfigsParameter.name]

    def on_enqueue(self):
        if not self.is_build_binaries_task() and not self.is_build_configs_task():
            raise SandboxTaskFailureError("Must build either configs or binaries")

        CommonBuildTask.on_enqueue(self)

        if self.is_build_configs_task():
            configs_dir = self.get_configs_dir()
            for config_type, config_name, _ in self.get_target_configs():
                if not getattr(config_type, 'any_arch', False):
                    raise SandboxTaskFailureError("Cannot build arch-specific config %s" % config_type)
                self._create_resource(
                    self.descr,
                    os.path.join(configs_dir, config_name),
                    config_type,
                    arch='any'
                )

    def on_execute(self):
        self.set_info('Building binaries.')
        for arch in self.archs_for_bundle:
            self.additional_bundle_params.setdefault(arch, {})[BuildConfigsParameter.name] = False
        if self.is_build_binaries_task():
            CommonBuildTask.on_execute(self)

        if self.is_build_configs_task():
            self.set_info('Making configs.')

            configs_dir = self.get_configs_dir()
            make_folder(configs_dir, True)

            for config_type, config_name, config_path in self.get_target_configs():
                config_svn_path = Arcadia.append(self.ctx[consts.ARCADIA_URL_KEY], config_path)
                Arcadia.export(config_svn_path, os.path.join(configs_dir, config_name))
        self.set_info('All done.')
