# -*- coding: utf-8 -*-

import logging
import tarfile
import shlex
import shutil
import json
import time
import os
import re
import errno

from sandbox.projects.common.nanny.auto_deploy import AutoNannyDeployTask
from sandbox.projects.common.utils import get_bsconfig
from sandbox.sandboxsdk.task import SandboxTask
import sandbox.sandboxsdk.environments as sdk_environments
import sandbox.sandboxsdk.process as sdk_process
import sandbox.sandboxsdk.paths as sdk_paths
from sandbox import common
from sandbox.sandboxsdk.parameters import SandboxStringParameter
from sandbox.sandboxsdk.parameters import SandboxBoolGroupParameter
from sandbox.sandboxsdk.parameters import SandboxBoolParameter
from sandbox.sandboxsdk import ssh


class GitCommit(SandboxStringParameter):
    """
    Target git branch to clone.
    Since git uses the same command to clone branches and tags you can use tag names in this field.
    """
    name = 'git_branch'
    description = 'GIT repository branch, tag or commit hash to clone'
    default_value = 'master'
    required = True


class PipRequirementsFile(SandboxStringParameter):
    """
    Requirements file name or path relative to repository root.
    """
    name = 'pip_requirements_file'
    description = 'Requirements file name or path to file relative to code repository root'
    default_value = 'requirements.txt'


class PythonBundleBuildOptions(SandboxBoolGroupParameter):
    """
    Group of checkboxes, which determines task behaviour: resources that should be exported, steps that should be
    performed.
    """
    name = 'generate_resources'
    description = 'Common execution steps'
    choices = [('Pack and export the code archive', 'code'),
               ('Build and export virtual environment', 'venv'), ]
    default_value = 'code venv'


class AllowSystemSitePackages(SandboxBoolParameter):
    """
    Should it be allowed system site packages usage from virtual environment built for code?
    """
    name = 'allow_system_site_packages'
    description = 'Allow usage of system site packages from generated virtual environment'
    default_value = False


class CustomResourceAttributes(SandboxStringParameter):
    """
    Attributes, that should be added to every resource exported by task run.
    """
    name = 'custom_resource_attributes'
    description = 'Additional custom resource attributes in JSON format'
    default_value = '{}'
    do_not_copy = True


class Utils(object):
    """
    Common functions. In most cases you will not need this functions at all, but they can be usefull sometimes.
    If you look this module's code (help) for a first time, easily skip this class and go to the
    'BaseBuildPythonBundleTask'.
    """

    @staticmethod
    def _create_tar_filter(exclude_patterns):
        """
        Creates function for filtering files by name before adding them to a tar archive.
        """

        def _filter(tarinfo):
            for pattern in exclude_patterns:
                if re.search(pattern, tarinfo.name):
                    return None

            return tarinfo

        return _filter

    @staticmethod
    def run_cmd(cmd_string, log_message='Executing command: {cmd}', log_prefix=None, work_dir=None):
        """
        Runs command given as string. Parses command string using shlex parser and executes command using
        sdk_process.run_process.

        :param log_message: debug message, that should be printed before command execution.
                            str.format() is used for substitution.
                            Following substitutions are available:
                              * '{cmd}' - parsed command (list)
                              * '{cmd_string}' - original command string before parsing.
        :param log_prefix: log file prefix name for logging command execution process.
                           When 'log_prefix' casts to 'True', then 'run_cmd' waits for command execution and returns
                           nothing. The command's output is logged to file a started with log_prefix value.
                           When 'log_prefix' casts to 'False', then 'run_cmd' returns 'Popen.communicate()' call result.

        :param work_dir: command working directory path.
        """
        command = shlex.split(cmd_string)

        logging.debug(log_message.format(cmd=command,
                                         cmd_str=cmd_string))

        if log_prefix:
            proc = sdk_process.run_process(cmd=command, log_prefix=log_prefix,
                                                  shell=False, work_dir=work_dir, wait=True)
        else:
            proc = sdk_process.run_process(cmd=command, outs_to_pipe=True,
                                                  shell=False, work_dir=work_dir, wait=False, check=False)
            return proc.communicate()

    @staticmethod
    def pack_archive(data_path, archive_path, excludes=None):
        """
        Create archive from 'data_path'.
        Files can be excluded from archive by 'excludes' patterns.
        Exclude patterns are interpreted as python regular expressions.

        :param data_path: path to file or directory to be archived
        :param archive_path: path to archive file should be created
        :param excludes: list of file exclude patterns.
                         Files matched to the pattern should not be added to archive.
        """
        logging.debug("Packing archive... Data: '{}', archive: '{}', excludes: '{}'".format(data_path,
                                                                                            archive_path,
                                                                                            excludes))
        if excludes is not None:
            _exclude_filter = Utils._create_tar_filter(excludes)
        else:
            _exclude_filter = None

        with tarfile.open(archive_path, "w|gz") as tar:
            if os.path.isdir(data_path):
                tar.add(data_path, ".", filter=_exclude_filter)
            else:
                tar.add(data_path, os.path.basename(data_path), filter=_exclude_filter)

    @staticmethod
    def gen_shard_name(name, tag):
        """
        Generate shard name using name and tag.

        :param name: 'name' part of shard name
        :param tag: 'tag' part of shard name

        :rtype: str
        :return: name of shard in 3 parts with '-' delimiter: name, tag and timestamp in YMD_HMS format.
        """
        timestamp = time.strftime("%Y%m%d_%H%M%S")

        shard_name = '{name}-{tag}-{ts}'.format(name=name,
                                                tag=tag,
                                                ts=timestamp)

        return shard_name

    @staticmethod
    def init_shard(shard_root):
        """
        Initialize shard using bsconfig utility.

        :param shard_root: shard root directory
        """

        shard_dir = os.path.dirname(shard_root)
        shard_name = os.path.basename(shard_root)

        # do initialize shard
        cmd = 'perl {bs} shard_init --torrent {sn}'.format(bs=get_bsconfig(),
                                                           sn=shard_name)

        Utils.run_cmd(cmd, log_message='Initializing shard', log_prefix='shard-init', work_dir=shard_dir)

        # Cleanup shard directory removing extra temporary files created by shard_init command
        # (this step is not required but it is better to have the same file list in sandbox shard resource and
        # shard itself)
        for extra_file in ['_lock.tmp', 'shard.state']:
            try:
                os.remove(os.path.join(shard_root, extra_file))
            except OSError as e:
                if e.errno != errno.ENOENT:
                    raise


class GitLocalRepository(object):
    def __init__(self, remote_source, local_path, commit=None):
        """
        Python wrapper around git utility. Can clone repository, list branches, tags, check HEAD reference type.

        :param remote_path: remote repository URL
        :param local_path: path to local repository copy
        :param commit: repository commit hash/branch/tag to clone (not required)
        """
        self.remote_source = remote_source
        self.local_path = local_path
        self.commit = commit

    def _gen_git_command(self, command):
        """
        Generates full git command string using self attributes and given git command.

        :param command: git command with parameters. Will be prepended with 'git --git-dir <dir> ...' prefix.

        :rtype: str
        :return: git command ready to be executed with shell or parsed by shlex.
        """
        cmd = "git --git-dir '{0}/.git' --work-tree '{0}' {1}".format(self.local_path,
                                                    command)

        return cmd

    def _get_current_branch(self):
        """
        Gets current local repository branch name. Returns None when you in 'detached' mode (HEAD is a ref to commit
        hash)

        :rtype: str or None
        :return: current local branch name or None in detached mode.
        """
        cmd = self._gen_git_command('branch')

        stdout, stderr = Utils.run_cmd(cmd, 'Getting list of GIT branches: {cmd}')
        branches = stdout.split('\n')

        current_branch = filter(lambda x: x.startswith('*'), branches)[0]
        current_branch_name = current_branch.lstrip('*').strip()

        # In detached state current commit is displayed in brackets.
        if current_branch_name.startswith('(') and current_branch_name.endswith(')'):
            current_branch_name = None

        return current_branch_name

    def clone(self, commit=None):
        """
        Clone git repository. Clones repository from remote source to local path.

        :param commit: commit to clone. Can be branch, tag or commit hash (git clone -b <commit>).
                       Defaults to remote repository branch name.
        """
        commit = commit or self.commit

        cmd_pattern = "git clone '{repo}' '{path}'"

        cmd = cmd_pattern.format(repo=self.remote_source,
                                 path=self.local_path)

        Utils.run_cmd(cmd, log_message='Cloning git repository: {cmd}', log_prefix='git-clone')

        if commit is None:
            commit = self._get_current_branch()
        else:
            cmd = "checkout '{commit}'".format(commit=commit)
            Utils.run_cmd(self._gen_git_command(cmd), log_message="Checkout ref: {cmd}", log_prefix='git-checkout')

        self.commit = commit

    def list_tags(self):
        """
        List local tags.

        :rtype: list
        :return: list of local repository tags.
        """
        cmd = self._gen_git_command('tag --sort="v:refname"')

        stdout, stderr = Utils.run_cmd(cmd, 'Getting list of GIT tags: {cmd}')
        tags = stdout.strip().split('\n')

        logging.debug('GIT repository tags list: {0}'.format(tags))

        return tags

    def list_branches(self):
        """
        List local branches.

        :rtype: list
        :return: list of local repository branches.
        """
        cmd = self._gen_git_command('branch')

        stdout, stderr = Utils.run_cmd(cmd, 'Getting list of GIT branches: {cmd}')

        branches = stdout.lstrip('*').strip().split('\n')
        branches = filter(lambda x: not (x.startswith('(') and x.endswith(')')), branches)

        logging.debug('GIT repository branches list: {0}'.format(branches))

        return branches

    def list_refs(self):
        """
        Return full list of registered 'hard' refs.
        :return:
        """
        return self.list_branches() + self.list_tags()

    def is_tag(self, symbolic_name=None):
        """
        Checks that given symbolic link name (or self.commit value) is a tag name.

        :param symbolic_name: name of link to check.

        :rtype: bool
        :return: True when 'self.commit' or 'symbolic_name' value is a tag name.
        """
        symbolic_name = symbolic_name or self.commit

        return symbolic_name in self.list_tags()

    def is_branch(self, symbolic_name=None):
        """
        Checks that given symbolic link name (or self.commit value) is a branch name.

        :param symbolic_name: name of link to check.

        :rtype: bool
        :return: True when 'self.commit' or 'symbolic_name' value is a branch name.
        """
        symbolic_name = symbolic_name or self.commit

        return symbolic_name in self.list_branches()

    def is_ref(self, symbolic_name=None):
        """
        Checks, that given symbolic link name (or self.commit value) is a known reference name: tag or branch.
        :param symbolic_name: name to check
        :return: True when the given link name equals to any tag or branch name. False otherwise.
        :rtype: bool
        """
        return symbolic_name in self.list_refs()

    @property
    def current_commit_hash(self):
        """
        Get current local repository full commit hash.

        :rtype: str
        :return: current local repository commit hash.
        """
        # 'git show --format' always prints out last commit full diff in some old git versions.
        cmd = self._gen_git_command("log -n 1 --format='%H'")
        stdout, stderr = Utils.run_cmd(cmd, 'Getting current commit hash')

        return stdout


class BaseBuildPythonBundleTask(AutoNannyDeployTask, SandboxTask):
    """
    Basic task for Python code deployment preparation. It can get the code from git repository, pack it to archive and
    and export as resource. Besides it can build a virtual environment for code execution and export it in separate
    resource.

    To use this code you should create your own task class inherited from this one and define some required parameters
    (class variables). Class variables are used to change final task behaviour.
    List of required and optional parameters (variables) and their meaning is listed below.

    Task also has some hook methods. You can redefine them to add some custom steps and logic to final task.
    Hooks are named using next template:
        <build step>_<before|after>_<step part>
    for example:
        code_before_archivation will be called before code archivation process start

    Special hooks before_execution and after_execution are called at the beginning of task execution and after all steps
    done.

    To enable your task automatically update your service configuration on release process, you should grant aproppriate
    permissions to 'nanny-robot' user: configuration manager and, possibly, operation manager.

    Required parameters (class variables):
        REPO_URL_PARAMETER          git repository URL parameter. Will be used to clone python code.

        RESOURCE_TYPE_CODE          code archive resource type class. When it is not None, task will pack source code to
                                    archive and export it in a resource of given type.

        RESOURCE_TYPE_VENV_ARCHIVE  virtual environment archive resource type class. When it is not None, task will
                                    create resource of given type with virtual environtment archive file.

        RESOURCE_TYPE_VENV_SHARD    virtual environment shard resource type class. When it is not None, task will
                                    create shard with virtual environment archive and export one in a resource of given
                                    type.

    Optional parameters (class variables):
        WHEEL_PACKAGES          packages, that should be installed from wheels stored in sandbox. Dict with package name
                                as key and version as value.

        ADDITIONAL_PARAMETERS   list of additional parameter classes. It is used for adding custom parameters to the
                                final task

        CODE_NAME               code archive name part. Also used as part of resource descriptions.
        CODE_EXCLUDE_PATTERNS   code archive file exlude patterns. List of file names and regexps to exlude files or
                                directories from code archive.

        VENV_SHARD_NAME         name part of shard name. Shard name is lokks like <name>-<tag>-<timestamp>
        VENV_ARCHIVE_FILE       python virtual environment archive file name, added to shard.

        REPO_BRANCH_PARAMETER   checkout branch parameter. This branch will be cloned during task execution. I think,
                                you may need to replace this parameter with your own class only when you want to change
                                default value.

        TEMP_DIR                temporary storage of files, used at task execution time. You will hardly want to change
                                this, I think.
        TEMP_CODE_PATH          temporary directory used for storing your code before archiving (checkout to this path)
        TEMP_VENV_PATH          temporary directory used at virtual environment generation time

        PIP_INSTALL_TIMEOUT     timeout for pip install for other requirements
    """
    execution_space = 200

    REPO_URL_PARAMETER = None
    REPO_BRANCH_PARAMETER = GitCommit
    ADDITIONAL_PARAMETERS = []

    RESOURCE_TYPE_CODE = None
    RESOURCE_TYPE_VENV_ARCHIVE = None
    RESOURCE_TYPE_VENV_SHARD = None

    CODE_NAME = 'python_code'
    CODE_EXCLUDE_PATTERNS = ['^\./tests',
                             '^\./\.git']
    VENV_ARCHIVE_FILE = 'virtualenv.tar.gz'
    VENV_SHARD_NAME = 'python_venv'

    WHEEL_PACKAGES = None

    TEMP_DIR = 'tmp/'
    TEMP_CODE_PATH = os.path.join(TEMP_DIR, CODE_NAME)
    TEMP_VENV_PATH = TEMP_DIR

    PIP_INSTALL_TIMEOUT = 30

    # Don't run deploy process after automatic Nanny service configuration update by default.
    # Just update the service configuration when needed.
    auto_deploy = False

    VAULT_OWNER = None
    SSH_KEY_NAME = None

    INPUT_PARAMETERS = [PythonBundleBuildOptions,
                        PipRequirementsFile,
                        AllowSystemSitePackages,
                        CustomResourceAttributes]

    @common.utils.classproperty
    def input_parameters(cls):
        return (
            [
                cls.REPO_URL_PARAMETER,
                cls.REPO_BRANCH_PARAMETER
            ] +
            cls.INPUT_PARAMETERS +
            cls.ADDITIONAL_PARAMETERS
        )

    def __init__(self, task_id=0):
        self.venv_archive = None
        self.venv_shard = None
        self.code_resource = None
        SandboxTask.__init__(self, task_id)

    def on_execute(self):
        """
        What should it do:
            * checkout repository with specific tag/branch/commit
            * generate virtual environment (optional)
                * create virtual environment and install requirements
                * pack virtual environment to archive
                * export virtual environment archive as a resource (when RESOURCE_TYPE_VENV_ARCHIVE is not None)
                * create shard with virtual environment (when RESOURCE_TYPE_VENV_SHARD is not None)
            * pack and export archive with code as a resource (optional)
        """

        # Get code from repository. Creates 'self.git' attribute.
        if self.VAULT_OWNER and self.SSH_KEY_NAME:
            with ssh.Key(self, self.VAULT_OWNER, self.SSH_KEY_NAME):
                self._get_python_code()
        else:
            self._get_python_code()

        # Generate common attributes list
        common_attributes = self.gen_common_attributes()

        # Hook: before common execution steps
        self.before_execution(common_attributes=common_attributes)

        # Virtual environment generation
        if self._checkBuildOption('venv'):
            requirements_file = self.ctx[PipRequirementsFile.name]
            requirements = os.path.join(self.git.local_path, requirements_file)

            venv = self.create_virtual_environment(self.TEMP_VENV_PATH, requirements)
            Utils.pack_archive(venv.root_dir, self.VENV_ARCHIVE_FILE)

            if self.RESOURCE_TYPE_VENV_ARCHIVE is not None:
                self.export_virtual_environment(venv_archive=self.VENV_ARCHIVE_FILE,
                                                common_attributes=common_attributes)

            if self.RESOURCE_TYPE_VENV_SHARD is not None:
                shard_root = self.create_virtualenv_shard(shard_name=self.venv_shard_name,
                                                          archive_file=self.VENV_ARCHIVE_FILE)
                self.export_virtual_environment_shard(shard_root, common_attributes=common_attributes)

        # Code bundle creation
        if self._checkBuildOption('code') and self.RESOURCE_TYPE_CODE is not None:
            self.create_python_code_resource(self.git.local_path, common_attributes)

        # Hook: after common execution steps
        self.after_execution(common_attributes=common_attributes)

    def _get_python_code(self):
        """
        Get python code from remote source using Git.
        Set 'self.git' attribute for further usage.
        """
        git_url = self.ctx[self.REPO_URL_PARAMETER.name]
        git_branch = self.ctx[self.REPO_BRANCH_PARAMETER.name]
        local_code_path = self.TEMP_CODE_PATH

        self.git = GitLocalRepository(remote_source=git_url, local_path=local_code_path)

        # Clone repository
        self.git.clone(git_branch)

    def _install_wheels(self, venv):
        """
        Install package wheels, listed in task attribute "WHEEL_PACKAGES".
        WHEEL_PACKAGES is a dictionary with package names as keys and package versions as values.

        :param venv: virtual environment object
        :type venv: sdk_environments.VirtualEnvironment

        :return: None
        """
        if self.WHEEL_PACKAGES is not None:
            for package_name, package_version in self.WHEEL_PACKAGES.iteritems():
                sdk_environments.PipEnvironment(package_name, package_version, use_wheel=True,
                                                       venv=venv).prepare()

    def _checkBuildOption(self, opt_name):
        """
        Check that some build option had been chosen.

        :param opt_name: option name to check
        """
        opt_list = self.ctx[PythonBundleBuildOptions.name].strip().split(' ')

        logging.debug('Checking option: {0}. Current options list: {1}'.format(opt_name, opt_list))
        return opt_name in opt_list

    @property
    def venv_shard_name(self):
        """
        Virtual environment shard name.
        Generates new shard name on first call. Any next call will returns this generated name.
        Uses venv shard name, uses target repository branch as tag.
        """
        if 'shard_name' in self.ctx:
            shard_name = self.ctx['shard_name']
        else:
            tag = self.ctx[self.REPO_BRANCH_PARAMETER.name].replace('/', '-')

            shard_name = Utils.gen_shard_name(name=self.VENV_SHARD_NAME, tag=tag)
            self.ctx['shard_name'] = shard_name

        return shard_name

    def gen_common_attributes(self):
        """
        Generate common resources attributes list using task's runtime context and current repository state.
        When local repository copy exists, 'commit_hash' and 'branch' or 'tag' attributes are generated.
        Custom attributes are used to update common attributes list, so you can recover any common attribute from
        'CustomResourceAttributes' task parameter.

        :param repo_path: path to local git repository copy. This copy's state is used to generate attributes.
                          Generated attributes:
                            'commit_hash' (always)
                            'tag' (optional, depends on HEAD)
                            'branch' (optional, depends on HEAD)

                          Current repository HEAD contents results in:
                            * 'tag' attribute generation when HEAD is a symbolic reference to tag.
                            * 'branch' attribute generation when HEAD is a symbolic reference to branch.
        """

        common_attributes = {}

        common_attributes["repository"] = self.git.remote_source

        if os.path.exists(self.git.local_path):
            common_attributes["commit_hash"] = self.git.current_commit_hash

            if self.git.is_tag():
                common_attributes["tag"] = self.git.commit
            elif self.git.is_branch():
                common_attributes["branch"] = self.git.commit

        # Add custom attributes to common list. This allows to override automatically
        # generated common attributes from sandbox task parameters.
        custom_attributes = json.loads(self.ctx[CustomResourceAttributes.name])
        common_attributes.update(custom_attributes)

        return common_attributes

    def create_virtual_environment(self, virtualenv_root, requirements_path):
        """
        Create virtual environment bundle. Virtual environment is built from Skynet python.
        Packages from 'requirements_path' file and WHEEL_PACKAGES dict will be installed into virtual environment.

        :param virtualenv_root: virtual environment root path
        :param requirements_path: path to requirements file to install packages into virtual environment

        :rtype: sdk_environments.VirtualEnvironment
        :return: VirtualEnvironment object associated with generated virtual environment
        """
        use_system = self.ctx[AllowSystemSitePackages.name]

        virtual_environment = sdk_environments.VirtualEnvironment(virtualenv_root, do_not_remove=True,
                                                                         use_system=use_system)

        with virtual_environment:
            # Hook
            self.venv_before_wheels(venv=virtual_environment)

            # Install wheels
            self._install_wheels(virtual_environment)

            # Hook
            self.venv_before_requirements(venv=virtual_environment, requirements_path=requirements_path)

            # Install other requirements
            virtual_environment.pip('-r {0} --timeout {1}'.format(requirements_path, self.PIP_INSTALL_TIMEOUT))

            # Hook
            self.venv_after_requirements(venv=virtual_environment, requirements_path=requirements_path)

            # Make virtual environment relocatable
            virtual_environment.make_relocatable()

        return virtual_environment

    def export_virtual_environment(self, venv_archive, common_attributes):
        if self.venv_archive is None:
            resource_name = "Архив с виртуальным окружением Python ({})".format(self.CODE_NAME)
            self.venv_archive = self.create_resource(resource_name, venv_archive,
                                                     self.RESOURCE_TYPE_VENV_ARCHIVE,
                                                     attributes=common_attributes)

        return self.venv_archive

    def create_virtualenv_shard(self, shard_name, archive_file):
        """
        Create shard with given python virtual environment archive.

        :param shard_name: full shard name to create
        :param archive_file: path to file with virtual environment archive, that should be added to shard.

        :rtype: str
        :return: path to shard directory relative to current task root directory
        """
        shard_path = self.path(shard_name)
        if os.path.exists(shard_path):
            sdk_paths.remove_path(shard_path)
        sdk_paths.make_folder(shard_path)

        # Copy virtual environment archive
        archive_name = os.path.basename(archive_file)
        shard_archive_path = os.path.join(shard_path, archive_name)
        shutil.copy(archive_file, shard_archive_path)

        self.generate_shard_configuration(shard_path=shard_path, archive_name=archive_name)

        # Hook
        self.venv_before_shard_init(shard_root=shard_path)

        # Shard initialization
        Utils.init_shard(shard_path)

        return shard_name

    def generate_shard_configuration(self, shard_path, archive_name):
        """
        Generate shard configuration (shard.conf). This file contains shard installation instructions, so
        shard archive can be automatically unpacked after deployment, for example.

        :param shard_path: shard root path
        :param archive_name: name of archive file to be unpacked on shard deployment process
        """
        with open(os.path.join(shard_path, 'shard.conf'), 'w') as f:
            f.write(
                '%install\n'
                'tar -xzf {}\n'.format(archive_name)
            )

    def export_virtual_environment_shard(self, shard_root, common_attributes):
        if self.venv_shard is None:
            # Create resource
            attributes = common_attributes.copy()
            attributes['shard_name'] = self.venv_shard_name

            resource_name = 'Shard с виртуальным окружением Python ({})'.format(self.CODE_NAME)
            self.venv_shard = self.create_resource(resource_name, shard_root,
                                                   self.RESOURCE_TYPE_VENV_SHARD,
                                                   attributes=attributes)

        return self.venv_shard

    def create_python_code_resource(self, code_path, common_attributes):
        """
        Create python code archive and export it as a resource
        """

        # Resource creation
        code_archive_name = '{}.tar.gz'.format(self.CODE_NAME)

        resource_name = 'Архив с кодом Python ({})'.format(self.CODE_NAME)

        self.code_resource = self.create_resource(resource_name, code_archive_name,
                                                  self.RESOURCE_TYPE_CODE,
                                                  attributes=common_attributes)

        # Hook
        self.code_before_archivation(common_attributes=common_attributes)

        # Archivation
        Utils.pack_archive(code_path, self.code_resource.path, self.CODE_EXCLUDE_PATTERNS)

        # Hook
        self.code_after_archivation(common_attributes=common_attributes)

    def arcadia_info(self):
        """
        Hacky way to allow this task to be released: provide tag, other fields are not checked.
        """
        return None, self.ctx[self.REPO_BRANCH_PARAMETER.name], None

    def before_execution(self, common_attributes):
        """
        This method is called after common task configuration (repository cloning, common attributes list generation,
        etc.) and before any real work (virtual environment generation, code archiving, etc).

        If you want to make some actions here, just redefine method in child class.

        :param common_attributes: list of common attributes that should be set for all task resources.
        :type common_attributes: dict
        """
        pass

    def after_execution(self, common_attributes):
        """
        This method is called when all real work is already done (virtual environment hed been generated, code archive
        had been created, etc).

        If you want to make some actions here, just redefine method in child class.

        :param common_attributes: list of common attributes that should be set for all task resources.
        :type common_attributes: dict
        """
        pass

    def venv_before_wheels(self, venv):
        """
        This method is called before wheels installation to virtual environment.

        If you want to make some actions here, just redefine method in child class.

        :param venv: virtual environment object that will be used for wheels installation.
        :type venv: sdk_environments.VirtualEnvironment
        """
        pass

    def venv_before_requirements(self, venv, requirements_path):
        """
        This method is called before installation of required packages to virtual environment.

        If you want to make some actions here, just redefine method in child class.

        :param venv: virtual environment object that will be used to install packages listed in requirements file.
        :type venv: sdk_environments.VirtualEnvironment

        :param requirements_path: path to requirements file that will be used
        :type requirements_path: str
        """
        pass

    def venv_after_requirements(self, venv, requirements_path):
        """
        This method is called after requirements installation to virtual environment.

        If you want to make some actions here, just redefine method in child class.

        :param venv: virtual environment object that will be used to install packages listed in requirements file.
        :type venv: sdk_environments.VirtualEnvironment

        :param requirements_path: path to requirements file that will be used
        :type requirements_path: str
        """
        pass

    def venv_before_shard_init(self, shard_root):
        """
        This method is called before virtual environment shard initialization.

        If you want to make some actions here, just redefine method in child class.

        :param shard_root: shard root directory path
        :type shard_root: str
        """
        pass

    def code_before_archivation(self, common_attributes):
        """
        This method is called before code archivation process start.

        If you want to make some actions here, just redefine method in child class.

        :param common_attributes: list of common attributes that should be set for all task resources.
        :type common_attributes: dict
        """
        pass

    def code_after_archivation(self, common_attributes):
        """
        This method is called after code archivation process end.

        If you want to make some actions here, just redefine method in child class.

        :param common_attributes: list of common attributes that should be set for all task resources.
        :type common_attributes: dict
        """
        pass
