import json
import re
import logging

from sandbox import sdk2
from sandbox.projects import resource_types
from sandbox.common.types.misc import NotExists


class CompareDiffTestResource(sdk2.Resource):
    """
    Result of comparision
    """
    has_diff = sdk2.Attributes.Bool("Flag for diff in resource", default=False)
    arcanum_review_id = sdk2.Attributes.Integer("Review", default=0)
    test_name = sdk2.Attributes.String("Test name")
    auto_backup = True


class BaseCompareYaMakeOutputsTask(sdk2.Task):

    class Parameters(sdk2.Task.Parameters):
        build_output_id1 = sdk2.parameters.Resource(
            "1st BUILD_OUTPUT",
            resource_type=resource_types.ARCADIA_PROJECT,
            required=True,
        )
        build_output_id2 = sdk2.parameters.Resource(
            "2nd BUILD_OUTPUT",
            resource_type=resource_types.ARCADIA_PROJECT,
            required=True,
        )
        testing_out_stuff_dir = sdk2.parameters.String(
            "Path to test output directory relative to arcadia root",
            required=True,
        )
        result_json_id1 = sdk2.parameters.Resource(
            "1st result.json",
            resource_type=resource_types.TEST_ENVIRONMENT_JSON_V2,
            required=False,
        )
        result_json_id2 = sdk2.parameters.Resource(
            "2nd result.json",
            resource_type=resource_types.TEST_ENVIRONMENT_JSON_V2,
            required=False,
        )
        fix_dir_outputs = sdk2.parameters.Bool(
            "test dir_outputs fix",
            required=False,
            default=False,
        )
        output_filename = sdk2.parameters.String(
            "Filename to write result of compare function.",
            default="diff.txt"
        )
        test_name = sdk2.parameters.String(
            "Testenv test name",
            required=False,
        )

    class Context(sdk2.Task.Context):
        pass

    def _get_tos_dirs(self, result_json):
        tos_dirs = set()
        assert "results" in result_json
        tos_regexp = re.compile(r'proxy\.sandbox\.yandex-team\.ru/\d+/(.*/test-results.*py3?test/.*)')
        for result in result_json["results"]:
            if 'links' not in result:
                continue
            links = result['links']
            logging.debug("links: %s", links)
            for name, value in links.iteritems():
                if name.startswith("logsdir"):
                    logging.debug("testing_out_stuff path: %s", value)
                    tos_link = value[0]
                    match = tos_regexp.search(tos_link)
                    if match:
                        tos_dirs.add(match.group(1))
                    else:
                        logging.info("link %s does not match", tos_link)
        return tos_dirs

    def on_execute(self):
        diff = ''
        if self.Parameters.fix_dir_outputs:
            with open(str(sdk2.ResourceData(self.Parameters.result_json_id1).path), 'r') as afile:
                result1_json = json.load(afile)
            with open(str(sdk2.ResourceData(self.Parameters.result_json_id2).path), 'r') as afile:
                result2_json = json.load(afile)
            tos_dirs_id1 = self._get_tos_dirs(result1_json)
            tos_dirs_id2 = self._get_tos_dirs(result2_json)
            logging.debug("tos_dirs_id1: %s", tos_dirs_id1)
            logging.debug("tos_dirs_id2: %s", tos_dirs_id2)

            for tos_dir in tos_dirs_id1 | tos_dirs_id2:
                diff_part = self.compare(
                    str(sdk2.ResourceData(self.Parameters.build_output_id1).path),
                    str(sdk2.ResourceData(self.Parameters.build_output_id2).path),
                    str(tos_dir),
                )
                if diff_part and isinstance(diff_part, str):
                    diff += diff_part
        else:
            logging.info("tos_dir: {}".format(self.Parameters.testing_out_stuff_dir))
            diff = self.compare(
                str(sdk2.ResourceData(self.Parameters.build_output_id1).path),
                str(sdk2.ResourceData(self.Parameters.build_output_id2).path),
                self.Parameters.testing_out_stuff_dir,
            )
        self.Context.has_diff = True if diff else False
        if diff:
            data = sdk2.ResourceData(
                CompareDiffTestResource(
                    task=self,
                    description="The diff",
                    path=self.Parameters.output_filename,
                    arcanum_review_id=self.Context.arcanum_review_id if self.Context.arcanum_review_id is not NotExists else 0,
                    has_diff=self.Context.has_diff,
                    test_name=self.Parameters.test_name,
                )
            )
            data.path.write_bytes(diff)
            data.ready()

    def compare(self, build_output1, build_output2, testing_out_stuff_dir):
        raise NotImplementedError()
