# -*- coding: utf-8 -*-

import re
import os
import logging

from time import time

from sandbox.projects import resource_types

from sandbox.sandboxsdk.parameters import SandboxSvnUrlParameter
from sandbox.sandboxsdk.parameters import SandboxStringParameter
from sandbox.sandboxsdk.parameters import SandboxBoolParameter

from sandbox.sandboxsdk.task import SandboxTask
from sandbox.sandboxsdk.svn import Arcadia
from sandbox.sandboxsdk.sandboxapi import Sandbox
from sandbox.sandboxsdk.paths import make_folder
from sandbox.sandboxsdk.errors import SandboxTaskFailureError

from sandbox.projects.common.nanny.nanny import ReleaseToNannyTask


class ShardmapSvnUrlParameter(SandboxSvnUrlParameter):
    """
        Путь до SVN директории с шардмапами
    """
    name = 'search_shardmap_url'
    description = 'SVN directory with shardmaps (svn+ssh://url/)'
    default_value = ''


class ShardmapLastNameTemplate(SandboxStringParameter):
    """
        Шаблон имени указателя на последний шардмап
        (то, что между shardmap_ и -last)
    """
    name = 'deployment_shardmap_last_name_template'
    description = 'Шаблон имени указателя на последний шардмап \
        (shardmap_<TEMPLATE>-last.map)'
    default_value = ''


class ShardmapGetLastFromSvnBool(SandboxBoolParameter):
    """
        Получить последний шардмап из SVN
    """
    name = 'deployment_shardmap_last'
    description = 'Получить последний шардмап из SVN'
    default_value = True


class ShardmapNameOverride(SandboxStringParameter):
    """
        Имя шардмапа (для ручной выкладки)
    """
    name = 'deployment_shardmap_name_override'
    description = 'Имя шардмапа (для ручной выкладки, \
        если не выбрано получение последнего из SVN)'
    default_value = ''


class BaseGetLastShardmapTask(ReleaseToNannyTask, SandboxTask):
    """
        Базовый класс для выкачивания шардмапа из SVN
    """
    input_parameters = [
        ShardmapSvnUrlParameter,
        ShardmapLastNameTemplate,
        ShardmapGetLastFromSvnBool,
        ShardmapNameOverride
    ]

    API_HOST = None
    API_PORT = None
    # API_HOST = "sandbox-feliksas.haze.yandex.net"
    # API_PORT = "8080"

    def on_execute(self):
        shardmap_resource_type = self._get_shardmap_resource_type()
        shardmap_svn_path = self.ctx[ShardmapSvnUrlParameter.name]
        shardmap_last_template = self.ctx[ShardmapLastNameTemplate.name]
        shardmap_get_last = self.ctx[ShardmapGetLastFromSvnBool.name]
        shardmap_name_override = self.ctx[ShardmapNameOverride.name]

        shardmap_reg = re.compile(
            'shardmap_(?P<service_name>[a-z_]+)-(?P<shardmap_ts>\d+).map')
        shardmap_service = os.path.basename(
            os.path.normpath(shardmap_svn_path))
        checkout_folder = make_folder('sandboxsvn_%s' % shardmap_service)
        Arcadia.checkout(
            shardmap_svn_path,
            path=checkout_folder,
            depth='files')
        resources = Sandbox(
            host=self.API_HOST,
            port=self.API_PORT).list_resources(
            shardmap_resource_type.name,
            status='READY',
            limit=999)

        already_released_shardmaps = []
        if resources is not None:
            for resource in resources:
                try:
                    already_released_shardmaps.append(
                        resource.attributes['shardmap_name'])
                except:
                    logging.debug("Wrong resource: %s" % resource)
        else:
            logging.debug(
                "Can't find no one released shardmap for %s with status \
                'READY'" %
                shardmap_resource_type.name)

        if shardmap_get_last:
            shardmap_last = 'shardmap_%s-last.map' % shardmap_last_template
            with open(os.path.join(checkout_folder, shardmap_last), "r") as fd:
                shardmap = fd.readline().strip()
        else:
            shardmap = shardmap_name_override
        if shardmap:
            if shardmap_get_last:
                m = shardmap_reg.match(shardmap)
                self.ctx['shardmap_timestamp'] = m.group('shardmap_ts')
            else:
                self.ctx['shardmap_timestamp'] = int(time())
            self.ctx['shardmap_name'] = shardmap
            if self.ctx['shardmap_name'] not in already_released_shardmaps:
                logging.debug(
                    'Publishing new resource: %s' %
                    self.ctx['shardmap_name'])
                resource_class = getattr(
                    resource_types, shardmap_resource_type.name)
                self.create_resource(
                    shardmap,
                    os.path.join(
                        checkout_folder,
                        shardmap),
                    resource_class,
                    attributes={
                        'shardmap_name': self.ctx['shardmap_name'],
                        'shardmap_timestamp': self.ctx['shardmap_timestamp']})
            else:
                raise SandboxTaskFailureError('Resource with name %s \
                                              already published' %
                                              self.ctx['shardmap_name'])

        else:
            logging.debug('Incorrect file name: %s' % shardmap)
