# -*- coding: utf-8 -*-

import re
import os
import logging

from sandbox.projects import resource_types
from sandbox.projects.common.nanny import nanny

from sandbox.sandboxsdk.parameters import SandboxSvnUrlParameter
from sandbox.sandboxsdk.parameters import SandboxStringParameter
from sandbox.sandboxsdk.parameters import SandboxBoolParameter

from sandbox.sandboxsdk.task import SandboxTask
from sandbox.sandboxsdk.svn import Arcadia
from sandbox.sandboxsdk.sandboxapi import Sandbox
from sandbox.sandboxsdk.paths import make_folder


#from sandbox.sandboxsdk.errors import SandboxTaskFailureError


class ShardmapSvnUrlParameter(SandboxSvnUrlParameter):
    """
        Путь до SVN директории с шардмапами
    """
    name = 'search_shardmap_url'
    description = 'SVN directory with shardmaps (svn+ssh://url/)'
    default_value = ''


class ShardmapResourceType(SandboxStringParameter):
    """
        Для какого ресурса релизить полученный шардмап
    """
    name = 'deployment_nanny_service_name'
    description = 'Тип ресурса в sandbox'
    default_value = 'None'


class ShardmapGetLastFromSvnBool(SandboxBoolParameter):
    """
        Получить последний шардмап из svn
    """
    name = 'deployment_shardmap_last'
    description = 'Получить последний шардмап из svn'
    default_value = False


class BaseGetShardmapTask(nanny.ReleaseToNannyTask, SandboxTask):
    """
        Базовый класс для выкачивания шардмапа из SVN
    """
    input_parameters = [
        ShardmapSvnUrlParameter,
        ShardmapResourceType,
        ShardmapGetLastFromSvnBool
    ]

    API_HOST = None
    API_PORT = None
    #API_HOST = "nextus-dev2.haze.yandex.net"
    #API_PORT = "8080"

    def on_execute(self):
        shardmap_resource_type = self._get_shardmap_resource_type()
        shardmap_svn_path = self.ctx[ShardmapSvnUrlParameter.name]

        shardmap_reg = re.compile(
            'shardmap-(?P<shardmap_ts>\d+)-(?P<shardmap_date>\d+)-(?P<shardmap_time>\d+).map')
        shardmap_service = os.path.basename(
            os.path.normpath(shardmap_svn_path))
        checkout_folder = make_folder('sandboxsvn_%s' % shardmap_service)
        Arcadia.checkout(
            shardmap_svn_path,
            path=checkout_folder,
            depth='files')
        resources = Sandbox(
            host=self.API_HOST,
            port=self.API_PORT).list_resources(
            shardmap_resource_type.name,
            status='READY',
            omit_failed=True,
            limit=999)

        already_released_shardmaps = []
        if resources is not None:
            for resource in resources:
                try:
                    task_status = Sandbox(
                        host=self.API_HOST,
                        port=self.API_PORT
                    ).get_task(resource.task_id).new_status
                    if task_status in (self.Status.SUCCESS, self.Status.RELEASED):
                        already_released_shardmaps.append(
                            resource.attributes['shardmap_name'])
                    else:
                        logging.debug("Wrong task status for resource: %s" % resource)
                except:
                    logging.debug("Wrong resource: %s" % resource)
        else:
            logging.debug(
                "Can't find no one released shardmap for %s with status 'READY'" %
                shardmap_resource_type.name)

        for shardmap in sorted(os.listdir(checkout_folder))[-20:]:
            if self.ctx[ShardmapGetLastFromSvnBool.name]:
                shardmap = sorted(os.listdir(checkout_folder))[-1]
            m = shardmap_reg.match(shardmap)
            if m:
                self.ctx['shardmap_name'] = "%s_%s" % (
                    shardmap_service, shardmap)
                self.ctx['shardmap_timestamp'] = m.group('shardmap_ts')
                self.ctx[
                    'shardmap_version'] = "%s-%s" % (m.group('shardmap_date'), m.group('shardmap_time'))
                if self.ctx['shardmap_name'] not in already_released_shardmaps:
                    logging.debug(
                        'Publishing new resource: %s' %
                        self.ctx['shardmap_name'])
                    resource_class = getattr(
                        resource_types, shardmap_resource_type.name)
                    self.create_resource(
                        shardmap,
                        os.path.join(
                            checkout_folder,
                            shardmap),
                        resource_class,
                        attributes={
                            'shardmap_name': self.ctx['shardmap_name'],
                            'shardmap_timestamp': self.ctx['shardmap_timestamp'],
                            'shardmap_version': self.ctx['shardmap_version']})
                    # We must release only one resource for each type for one
                    # task run
                    break
                else:
                    logging.debug(
                        'Resource with name %s already published' %
                        self.ctx['shardmap_name'])
            else:
                logging.debug('Incorrect file name: %s' % shardmap)
