# -*- coding: utf-8 -*-

import sandbox.common.types.misc as ctm
import os
import shutil
import tempfile

import sandbox.sandboxsdk.paths as sdk_paths

from sandbox.common.share import skynet_get, files_torrent

from sandbox.sandboxsdk.channel import channel
from sandbox.sandboxsdk.errors import SandboxTaskFailureError
from sandbox.sandboxsdk.task import SandboxTask

from sandbox.projects.common.BaseImagesBundleReplaces import parse_resource_replace, REPLACED_BY_TORRENT, REPLACED_BY_RESOURCE


class BaseMediaBuildBundle(SandboxTask):

    __ARCH = 'linux'

    def get_built_resources(self):
        """Returns list of tuple [resource type, rename to string] of binary resources from build task to bundle"""

        raise NotImplementedError()

    def get_resources(self):
        """Returns list of tuple [resource type, build task type, rename to] of binary resources to bundle"""

        return ()

    def get_resources_from_tasks(self):
        """Returns list of tuple [resource type, task id, rename to] of binary resources to bundle"""

        return ()

    def get_resource_replaces(self):
        return ()

    def get_resource_type(self):
        """Returns bundle resource type"""

        raise NotImplementedError()

    def get_resource_descr(self):
        """Returns bundle resource description"""

        raise NotImplementedError()

    def get_resource_file_name(self):
        """Returns bundle resource file name"""

        raise NotImplementedError()

    def get_build_task(self):
        """Returns build task"""

        raise NotImplementedError()

    dns = ctm.DnsType.DNS64
    __RESOURCE_ID = 'resource_id'

    def __get_last_released_task(self, task_type, resource_type):
        step = 10
        offset, limit = 0, 10
        while True:
            resources = channel.sandbox.list_releases(resource_type=resource_type,
                    arch=self.__ARCH, offset=offset, limit=limit)
            if not resources:
                return None
            offset += step
            limit += step
            for resource in resources:
                task = channel.sandbox.get_task(resource.task_id)
                if task.type == task_type:
                    return task

    def on_execute(self):
        self.__make_resource()

        dst_resource = channel.sandbox.get_resource(self.ctx[self.__RESOURCE_ID])

        res_replaces = self.__parse_replaced_resources()

        self.__get_from_built_resources(res_replaces, dst_resource.path)
        self.__get_from_resources(res_replaces, dst_resource.path)
        self.__get_from_resources_from_tasks(res_replaces, dst_resource.path)

        channel.task.mark_resource_ready(dst_resource)

    def __get_from_built_resources(self, res_replaces, dst_path):
        build_task = self.get_build_task()
        for resource in self.get_built_resources():
            resource_type = resource[0]
            rename_to = resource[1] if len(resource) > 1 else ''

            if resource_type in res_replaces:
                self.__get_replaced_resource(res_replaces.get(resource_type), resource_type, dst_path, rename_to)
            else:
                self.__get_from_task(build_task, resource_type, dst_path, rename_to)

    def __get_from_resources(self, res_replaces, dst_path):
        for resource in self.get_resources():
            resource_type = resource[0]
            task_type = resource[1]
            rename_to = resource[2] if len(resource) > 2 else ''

            if resource_type in res_replaces:
                self.__get_replaced_resource(res_replaces.get(resource_type), resource_type, dst_path, rename_to)
            else:
                task = self.__get_last_released_task(task_type, resource_type)
                if not task:
                    raise SandboxTaskFailureError('Task {0} releases not found'.format(task_type))

                self.__get_from_task(task, resource_type, dst_path, rename_to)
                self.ctx[task_type] = task.id

    def __get_from_resources_from_tasks(self, res_replaces, dst_path):
        for resource in self.get_resources_from_tasks():
            resource_type = resource[0]
            task_id = resource[1]
            rename_to = resource[2] if len(resource) > 2 else ''
            if resource_type in res_replaces:
                self.__get_replaced_resource(res_replaces.get(resource_type), resource_type, dst_path, rename_to)
            else:
                self.__get_from_task(task_id, resource_type, dst_path, rename_to)

    def __parse_replaced_resources(self):
        replaces = self.get_resource_replaces()
        result = {}
        for res_type, replace_param in replaces:
            if replace_param.name in self.ctx:
                value = self.ctx.get(replace_param.name)
                if value:
                    result[res_type] = parse_resource_replace(value)
        return result

    def __get_replaced_resource(self, replaced_res, resource_type, dst_folder, rename_to):
        url_type, url = replaced_res
        if url_type == REPLACED_BY_RESOURCE:
            url_res_type = channel.sandbox.get_resource(url).type
            if url_res_type != resource_type:
                raise SandboxTaskFailureError('Tried to find resource type {0} found {1}'.format(resource_type, url_res_type))
            self.__get_from_resource(url, dst_folder, rename_to)
        elif url_type == REPLACED_BY_TORRENT:
            self.__get_from_rbtorrent(url, dst_folder, rename_to)
        else:
            raise SandboxTaskFailureError('Unknown resource url type {0}'.format(url_type))

        self.ctx[resource_type.__name__] = url

    def __make_resource(self):
        if self.__RESOURCE_ID in self.ctx:
            return

        resource_id = self.create_resource(
            self.get_resource_descr(),
            self.get_resource_file_name(),
            self.get_resource_type(),
            arch=self.__ARCH
        ).id
        resource = channel.sandbox.get_resource(resource_id)
        sdk_paths.make_folder(resource.path, True)

        self.ctx[self.__RESOURCE_ID] = resource_id

    def __get_from_task(self, task, resource_type, dst_folder, rename_to):
        resources = channel.sandbox.list_resources(
            resource_type=resource_type, limit=1,
            task_id=task, arch=self.__ARCH)

        if not resources:
            raise SandboxTaskFailureError('Resource {0} not found'.format(resource_type))

        self.__get_from_resource(resources[0].id, dst_folder, rename_to)

    def __get_from_resource(self, resource_id, dst_folder, rename_to):
        resource_path = self.sync_resource(resource_id)

        if not rename_to:
            rename_to = os.path.basename(resource_path)

        sdk_paths.copy_path(resource_path, os.path.join(dst_folder, rename_to))

    def __get_rbtorrent_file_name(self, url):
        files = files_torrent(url)

        if len(files) != 1:
            raise SandboxTaskFailureError('Share {0} should have only one file'.format(url))

        return files[0].get('name')

    def __get_from_rbtorrent(self, url, dst_folder, rename_to):
        try:
            if not rename_to:
                rename_to = self.__get_rbtorrent_file_name(url)

            tmp = tempfile.mkdtemp()
            skynet_get(url, tmp)
            src_file_path = os.path.join(tmp, rename_to)
            dst_file_path = os.path.join(dst_folder, rename_to)
            if os.path.isfile(src_file_path):
                shutil.move(src_file_path, dst_file_path)
            else:
                raise SandboxTaskFailureError('File name {0} not found in url {1}'.format(rename_to, url))
        finally:
            shutil.rmtree(tmp)
