# -*- coding: utf-8 -*-

import sandbox.common.types.misc as ctm
import os
import shutil

from sandbox import sdk2

from sandbox.common.errors import TaskFailure


class BaseMediaBuildBundle(sdk2.Task):

    class Requirements(sdk2.Requirements):
        cores = 1
        ram = 512

        class Caches(sdk2.Requirements.Caches):
            pass

    __ARCH = 'linux'

    def get_built_resources(self):
        """Returns list of tuple [resource type, rename to string] of binary resources from build task to bundle"""

        raise NotImplementedError()

    def get_resources(self):
        """Returns list of tuple [resource type, build task type, rename to] of binary resources to bundle"""

        return ()

    def get_resources_from_tasks(self):
        """Returns list of tuple [resource type, task id, rename to] of binary resources to bundle"""

        return ()

    def get_resource_type(self):
        """Returns bundle resource type"""

        raise NotImplementedError()

    def get_resource_descr(self):
        """Returns bundle resource description"""

        raise NotImplementedError()

    def get_resource_file_name(self):
        """Returns bundle resource file name"""

        raise NotImplementedError()

    def get_build_task(self):
        """Returns build task"""

        raise NotImplementedError()

    dns = ctm.DnsType.DNS64
    __RESOURCE_ID = 'resource_id'

    def __get_last_released_task(self, task_type, resource_type):
        resources = sdk2.Resource.find(type=resource_type,
                        arch=self.__ARCH, attrs={"released": "stable"}).order(-sdk2.Resource.id).limit(42)
        for resource in resources:
            if task_type is not None:
                task = sdk2.Task.find(id=resource.task_id).limit(1).first()
                if str(task.type) != str(task_type):
                    continue
            return resource.task_id
        return None

    def on_execute(self):
        dst_resource = self.__make_resource()

        self.__get_from_built_resources(dst_resource.path)
        self.__get_from_resources(dst_resource.path)
        self.__get_from_resources_from_tasks(dst_resource.path)

        dst_resource.ready()

    def __get_from_built_resources(self, dst_path):
        build_task = self.get_build_task()
        for resource in self.get_built_resources():
            resource_type = resource[0]
            rename_to = resource[1] if len(resource) > 1 else ''

            self.__get_from_task(build_task, resource_type, dst_path, rename_to)

    def __get_from_resources(self, dst_path):
        for resource in self.get_resources():
            resource_type = resource[0]
            task_type = resource[1] if len(resource) > 1 else None
            rename_to = resource[2] if len(resource) > 2 else ''

            task_id = self.__get_last_released_task(task_type, resource_type)
            if not task_id:
                raise TaskFailure('Releases of the resource {0} from task {1} was not found'.format(resource_type, task_type))

            self.__get_from_task(task_id, resource_type, dst_path, rename_to)
            # self.Parameters.[task_type] = task.id

    def __get_from_resources_from_tasks(self, dst_path):
        for resource in self.get_resources_from_tasks():
            resource_type = resource[0]
            task_id = resource[1]
            rename_to = resource[2] if len(resource) > 2 else ''
            self.__get_from_task(task_id, resource_type, dst_path, rename_to)

    def __make_resource(self):
        resource = sdk2.ResourceData(self.get_resource_type()(self,
            self.get_resource_descr(),
            self.get_resource_file_name(),
            arch=self.__ARCH))

        return resource

    def __get_from_task(self, task_id, resource_type, dst_folder, rename_to):
        resource = sdk2.Resource.find(type=resource_type, task_id=task_id, arch=self.__ARCH).limit(1).first()

        if not resource:
            raise TaskFailure('Resource {0} not found'.format(resource_type))

        self.__get_from_resource(resource, dst_folder, rename_to)

    def __get_from_resource(self, resource, dst_folder, rename_to):
        resource_data = sdk2.ResourceData(resource)

        if not rename_to:
            rename_to = os.path.basename(str(resource_data.path))

        dst_path = str(dst_folder.joinpath(rename_to))
        dst_dir = os.path.dirname(dst_path)
        if not os.path.exists(dst_dir):
            os.makedirs(dst_dir)
        shutil.copy(str(resource_data.path), dst_path)
