# -*- coding: utf-8 -*-
from sandbox.common import rest

from sandbox.projects import resource_types

from sandbox.sandboxsdk.sandboxapi import ARCH_ANY
from sandbox.sandboxsdk.channel import channel
from sandbox.sandboxsdk.errors import SandboxTaskFailureError

from sandbox.projects.common import utils

from sandbox.projects.common.build import parameters

from sandbox.projects.common.priemka.BasePriemkaTask import BasePriemkaTask
from sandbox.projects.common.BaseImagesBuildBundle import BuildTask


class BaseMediaBuildRobotBundles(BasePriemkaTask):
    """
        Build all robot binaries and make bundle
    """

    __EMPTY_RESOURCE_NAME = 'empty_resource'
    __EMPTY_RESOURCE_ID = 'empty_resource_id'

    def get_project(self):
        """
            Arcadia project branch
        """
        raise NotImplementedError()

    def create_changelog(self):
        # We do not have main resource so can't use this function
        pass

    def get_bundle_types(self):
        """
            Built bundle resource type
            returns [bundle type, task context]
        """
        raise NotImplementedError()

    def on_execute(self):
        BasePriemkaTask.on_execute(self)

        for bundle_type, ctx in self.get_bundle_types():
            self.__build_deploy_bundle(bundle_type, ctx)

        self.__create_empty_resurce()

        utils.check_subtasks_fails(fail_on_first_failure=True)

    def on_release(self, additional_parameters):
        self.create_release(self.get_build_task_id(), status=additional_parameters['release_status'])

        for bundle_type, _ in self.get_bundle_types():
            self.__release_deploy_bundle(bundle_type, additional_parameters)

    def __release_deploy_bundle(self, bundle_type, additional_parameters):
        task = channel.sandbox.get_task(self.ctx[bundle_type])
        if task.new_status in self.Status.Group.SUCCEED:
            release_id = self.create_release(task.id, status=additional_parameters['release_status'])
            if not release_id:
                raise SandboxTaskFailureError("Failed to release task {}".format(task.id))
        else:
            raise SandboxTaskFailureError("Failed to release task {}".format(task.id))

    def __build_deploy_bundle(self, bundle_type, bundle_ctx):
        if bundle_type in self.ctx:
            return
        build_task_id = self.get_build_task_id()
        subtask_ctx = {
            BuildTask.name: build_task_id
        }

        build_task_ctx = rest.Client().task[build_task_id].context.read()
        arcadia_url = build_task_ctx[parameters.ArcadiaUrl.name]
        if arcadia_url:
            subtask_ctx.update({parameters.ArcadiaUrl.name: arcadia_url})

        subtask_ctx.update(bundle_ctx)

        task = channel.task.create_subtask(
            task_type=bundle_type,
            description=self.descr,
            input_parameters=subtask_ctx)

        self.ctx[bundle_type] = task.id

    def __create_empty_resurce(self):
        if self.__EMPTY_RESOURCE_ID in self.ctx:
            return

        resource = self.create_resource(
            self.descr,
            self.__EMPTY_RESOURCE_NAME,
            resource_types.IMAGES_EMPTY_RELEASEABLE_RESOURCE,
            arch=ARCH_ANY,
            attributes={}
        )

        open(resource.path, 'a').close()

        channel.task.mark_resource_ready(resource)

        self.ctx[self.__EMPTY_RESOURCE_ID] = resource.id
