# -*- coding: utf-8 -*-

import os
import time
import shutil
import json

from sandbox.projects import resource_types
from sandbox.projects.app_host import resources as app_host_resources
from sandbox.sandboxsdk.channel import channel
from sandbox.sandboxsdk.task import SandboxTask
from sandbox.sandboxsdk.parameters import LastReleasedResource
from sandbox.sandboxsdk.parameters import SandboxStringParameter
from sandbox.sandboxsdk.svn import Arcadia
from sandbox.sandboxsdk.paths import make_folder
from sandbox.sandboxsdk.paths import copy_path
from sandbox.sandboxsdk.process import run_process
from sandbox.sandboxsdk.errors import SandboxTaskFailureError

SCRIPTS_BSCONFIG = 'arcadia:/arc/trunk/arcadia/yweb/scripts/bsconfig'
TESTS_CONFIGURATION = 'arcadia:/arc/trunk/arcadia/search/garden/runtime_tests/configuration/'


class ChangePrefixParameter(SandboxStringParameter):
    name = 'name_prefix'
    description = 'shard name prefix (leave empty for default value)'
    default_value = ''


class ChangeShardConfParameter(SandboxStringParameter):
    name = 'shard_conf'
    description = 'install scripts folder svn url (leave empty for default value)'
    default_value = ''


def getClassesByResources(resources):
    classes = []
    for name in resources:
        resource_type = getattr(resource_types, name) if hasattr(resource_types, name) else getattr(app_host_resources, name)
        short_name = name.lower().replace('_package', '').replace('_bundle', '')
        attrs = {
            'name': 'res_' + name.lower(),
            'description': short_name.replace('_', ' '),
            'resource_type': resource_type
        }
        ParamClass = type(name, (LastReleasedResource,), attrs)
        classes.append(ParamClass)
    return classes


class BaseReportShardTask(SandboxTask):
    input_parameters = [ChangePrefixParameter, ChangeShardConfParameter, ]

    def on_execute(self):
        shard_name = self.get_shard_prefix() + '-000-' + str(int(time.time()))
        self.ctx['shard_name'] = shard_name
        if self.shard_exists(shard_name):
            raise SandboxTaskFailureError('Shard %s already exists' % shard_name)
        shard_dir = self.abs_path(shard_name)
        make_folder(shard_dir)
        resource_info = self.copy_resources(shard_dir)
        self.save_resource_info(resource_info, shard_dir)
        Arcadia.export(self.get_shard_conf(), shard_dir)
        self.pack_tests(shard_dir)
        run_process([self.get_bsconfig(), 'shard_init', '--torrent', '--logfile',
                     self.log_path('bsconfig.log'), shard_name], log_prefix="bs_config")
        self.create_resource(self.descr, shard_dir, self.resource_type,
                             attributes={'ttl': 'inf'})

    def arcadia_info(self):
        info = 'upper+report shard: %s' % self.ctx['shard_name']
        return None, info, None

    def pack_tests(self, shard_dir):
        Arcadia.export(TESTS_CONFIGURATION, os.path.join(shard_dir, 'tests'))
        shutil.move(os.path.join(shard_dir, 'tests', 'tests'), os.path.join(shard_dir, 'tests', 'configuration'))

    @staticmethod
    def shard_exists(conf_name):
        from api.cms import Registry

        try:
            exists = Registry.getShard(conf_name) is not None
        except Exception:
            exists = False
        return exists

    def get_shard_prefix(self):
        if len(self.ctx['name_prefix']) != 0:
            return self.ctx['name_prefix']
        return self.default_shard_prefix

    def get_shard_conf(self):
        if len(self.ctx['shard_conf']) != 0:
            return self.ctx['shard_conf']
        return self.default_shard_conf

    def copy_resources(self, shard_dir):
        resource_info = {}
        for param in self.input_parameters:
            if not param.name.startswith('res_'):
                continue
            resource_id = self.ctx[param.name]
            if resource_id == 0:
                continue
            resource_path = self.sync_resource(resource_id)
            copy_path(resource_path, os.path.join(shard_dir, os.path.basename(resource_path)))
            resource_info[param.name.replace('res_', '')] = self.get_resource_info(resource_id)
        return resource_info

    @staticmethod
    def get_resource_info(resource_id):
        resource = channel.sandbox.get_resource(resource_id)
        res_info = {'id': resource_id, 'description': resource.description}
        return res_info

    def get_bsconfig(self):
        bsconfig_svn_url = SCRIPTS_BSCONFIG
        bsconfig_path = self.abs_path('bsconfig')
        Arcadia.export(bsconfig_svn_url, bsconfig_path)
        return bsconfig_path

    def get_short_task_result(self):
        if self.ctx.has_key('shard_name'):
            return self.ctx['shard_name']
        return None

    @staticmethod
    def save_resource_info(resource_info, shard_dir):
        with open(os.path.join(shard_dir, 'sandbox_info'), 'w') as resource_info_file:
            json.dump(resource_info, resource_info_file, indent=2)
