# -*- coding: utf-8 -*-

import os
import shutil
import time
import logging

from sandbox.sandboxsdk import parameters

from sandbox.projects import resource_types
from sandbox.sandboxsdk.paths import remove_path
from sandbox.sandboxsdk.process import get_process_info
from sandbox.sandboxsdk.process import throw_subprocess_error
from sandbox.sandboxsdk.process import run_process
from sandbox.sandboxsdk.channel import channel
from sandbox.sandboxsdk.errors import SandboxTaskFailureError
from sandbox.projects.common.dolbilka import stats_parser
from sandbox.projects.common.utils import check_processes
from sandbox.projects.common.dolbilka import DolbilkaExecutor
from sandbox.projects.common.dolbilka import DolbilkaDumper

from sandbox.projects.common.BaseTestTask import base_tester_task


class BaseDolbiloTask(base_tester_task.BaseTesterTask):
    type = 'BASE_PECK_TASK'

    class DolbiloOptionsInfo(parameters.SandboxInfoParameter):
        name = "dolbilo_options_info"
        description = "Dolbilo related options"

    class RequestsLimit(parameters.SandboxIntegerParameter):
        name = "requests_limit"
        description = "Requests limit"
        default_value = 10000

    class TotalSessions(parameters.SandboxIntegerParameter):
        name = "total_sessions"
        description = "Number of runs "
        default_value = 1

    class StartTimeout(parameters.SandboxIntegerParameter):
        name = "start_timeout"
        description = "Search start timeout"
        default_value = 240

    class DExecutorAugmenturl(parameters.SandboxStringParameter):
        name = "d_executor_augmenturl"
        description = "Additional string to append to each request URL"

    class ExecutorMode(parameters.SandboxRadioParameter):
        name = "executor_mode"
        description = "Executor mode"
        per_line = 4
        choices = [(_, _) for _ in ("plan", "devastate", "binary", "finger")]
        sub_fields = {
            "devastate": ["fuckup_mode_max_simultaneous_requests"],
            "plan": ["plan_mode_delay_between_requests_multipliers"],
            "binary": ["binary_min_multiplier", "binary_max_multiplier"],
            "finger": ["fuckup_mode_max_simultaneous_requests"]
        }

    class FuckupModeMaxSimultaneousRequests(parameters.SandboxIntegerParameter):
        name = "fuckup_mode_max_simultaneous_requests"
        description = "Max simultaneous requests"
        default_value = 1

    class PlanModeDelayBetweenReqsMultipliers(parameters.SandboxFloatParameter):
        name = "plan_mode_delay_between_requests_multipliers"
        description = "Plan mode delay"
        default_value = 1.

    class BinaryMinMultiplier(parameters.SandboxIntegerParameter):
        name = "binary_min_multiplier"
        description = "Minimal rps"
        default_value = 123

    class BinaryMaxMultiplier(parameters.SandboxIntegerParameter):
        name = "binary_max_multiplier"
        description = "Maximum rps"
        default_value = 456

    class DoTrace(parameters.SandboxBoolParameter):
        name = "do_trace"
        description = "Trace"

    class Circular(parameters.SandboxBoolParameter):
        name = "circular"
        description = "Repeat shots in circle"
        default_value = False

    class PlanMemory(parameters.SandboxBoolParameter):
        name = "plan_memory"
        description = "Load full plan in memory"
        default_value = False

    class MaxServiceUnavailable(parameters.SandboxIntegerParameter):
        name = "max_service_unavailable"
        description = "Coredump when service unavailable more than (0 - don't check)"

    class SaveLoadLog(parameters.SandboxBoolParameter):
        name = "save_load_log"
        description = "Save loadlog as resource"

    class DoNotRemoveDolbilkaResults(parameters.SandboxBoolParameter):
        name = "do_not_remove_dolbilka_results"
        description = "Store dolbilka results resources forever"

    input_parameters = [
        DolbiloOptionsInfo, RequestsLimit, TotalSessions, StartTimeout, DExecutorAugmenturl,
        ExecutorMode, FuckupModeMaxSimultaneousRequests, PlanModeDelayBetweenReqsMultipliers,
        BinaryMinMultiplier, BinaryMaxMultiplier, DoTrace, Circular, PlanMemory, MaxServiceUnavailable,
        SaveLoadLog, DoNotRemoveDolbilkaResults
    ]

    default_cpu_model = 'e5645'

    def on_enqueue(self):
        base_tester_task.BaseTesterTask.on_enqueue(self)
        if not self.cpu_model_filter:
            self.cpu_model_filter = self.default_cpu_model
        if self.ctx.get('save_load_log', False):
            resource = self._create_resource("load log", "load.log", resource_types.LOAD_LOG)
            self.ctx['load_log_resource_id'] = resource.id

    def _run_session(
        self, session_id, subproc_list, dump_name, stat_name, rss, save_dump,
        plan_resource=None,
        executor_mode=None,
        simultaneous_requests=None,
        delay_between_requests=None,
    ):
        check_processes(subproc_list)
        EXECUTOR_TIMEOUT = '30000000'
        self.ctx['dolbilka_test_sessions_info']['session_%s' % session_id] = {}
        session_info = self.ctx['dolbilka_test_sessions_info']['session_%s' % session_id]
        working_dir = self.abs_path()

        resource_path = self.sync_resource(plan_resource or self.ctx.get('dolbilo_plan_resource_id'))
        params = ''
        # prepare params and start executor
        executor_mode = executor_mode or self.ctx['executor_mode']
        if executor_mode in ('devastate', 'finger'):
            simultaneous_requests = str(simultaneous_requests or self.ctx['fuckup_mode_max_simultaneous_requests'])
            if executor_mode == 'devastate':
                params = '-m devastate -s {}'.format(simultaneous_requests)
            elif executor_mode == 'finger':
                params = '-m finger -s {}'.format(simultaneous_requests)
        elif executor_mode == 'plan':
            delay_between_requests = str(
                delay_between_requests or
                self.ctx[self.PlanModeDelayBetweenReqsMultipliers.name]
            )
            params = '-m plan -M {}'.format(delay_between_requests)
        elif executor_mode == 'binary':
            params = '-m binary -a {} -b {}'.format(
                self.ctx['binary_min_multiplier'], self.ctx['binary_max_multiplier']
            )

        session_descr = (u'{}, session {}, {}'.format(self.descr, str(session_id), params))

        if self.ctx.get(self.Circular.name, False):
            params = "{} --circular".format(params)

        if self.ctx.get(self.PlanMemory.name, False):
            params = "{} --plan-memory".format(params)

        logging.info('Load d-executor: ')
        executor_cmd_template = '{} -Q {} -H {} -P {} -t {} {} -p {} -o {}'
        d_executor_custom_resource_id = self.ctx.get('d_executor_custom_resource_id')
        if d_executor_custom_resource_id:
            dexecutor_path = self.sync_resource(d_executor_custom_resource_id)
        else:
            dexecutor_path = DolbilkaExecutor.get_executor_path()
        executor_cmd = executor_cmd_template.format(
            dexecutor_path,
            self.ctx['requests_limit'],
            'localhost',
            self._get_searcher_port(),
            EXECUTOR_TIMEOUT,
            params,
            resource_path,
            self.abs_path(dump_name)
        )
        if self.ctx.get('dolbilka_get_responses', False) or self.type == 'TEST_REALSEARCH':
            executor_cmd += ' -d'
        if self.ctx.get('d_executor_augmenturl'):
            executor_cmd += ' --augmenturl={0}'.format(self.ctx['d_executor_augmenturl'])
        session_info['d_executor_cmd'] = executor_cmd
        logging.info('Get process info before test session %s', session_id)
        before_info = []
        check_processes(subproc_list)
        for subproc in subproc_list:
            before_info.append(get_process_info(subproc.pid, ('%cpu', '%mem', 'time', 'command', 'majflt', 'minflt')))
        session_info['processes_info_before'] = before_info
        executor_process = run_process(executor_cmd, wait=False, log_prefix='executor')
        # wait
        while (
            (executor_process.poll() is None) and
            not len(filter(lambda x: x is not None, map(lambda x: x.poll(), subproc_list)))
        ):
            time.sleep(2)
        logging.info('Get process info after test session %s', session_id)
        after_info = []
        for subproc in subproc_list:
            after_info.append(get_process_info(subproc.pid, ('%cpu', '%mem', 'time', 'command', 'majflt', 'minflt')))
        session_info['processes_info_after'] = after_info
        for subproc in subproc_list:
            if subproc.poll() is not None:
                throw_subprocess_error(subproc)

            if executor_process.poll() is not None:
                if executor_process.returncode:
                    throw_subprocess_error(subproc)

        d_dumper_custom_resource_id = self.ctx.get('d_dumper_custom_resource_id')
        if d_dumper_custom_resource_id:
            ddumper_path = self.sync_resource(d_dumper_custom_resource_id)
        else:
            ddumper_path = DolbilkaDumper.get_dumper_path()

        # create statistics
        run_process(
            ' '.join([ddumper_path, '-a', '-f', self.abs_path(dump_name)]),
            log_prefix='d-dumper'
        )
        stat_path = self.abs_path(stat_name)
        shutil.move(self.log_path('d-dumper.out.txt'), stat_path)

        # get_responses
        result_resource_attributes = None
        if self.ctx.get(self.DoNotRemoveDolbilkaResults.name):
            result_resource_attributes = {'ttl': 'inf'}

        if self.ctx.get('dolbilka_get_responses', False):
            resource = self._create_resource(
                session_descr, 'dolbilka_responses.txt',
                resource_types.DOLBILKA_RESPONSES,
                attrs=result_resource_attributes
            )
            run_process(
                [ddumper_path, '-u', '-f', self.abs_path(dump_name), ],
                log_prefix='d-dumper-responses'
            )
            shutil.move(self.log_path('d-dumper-responses.out.txt'), resource.abs_path())
            resource.mark_ready()

        # save standard executor dump resources
        resource = self._create_resource(
            session_descr, stat_path,
            resource_types.EXECUTOR_STAT,
            attrs=result_resource_attributes
        )
        resource.mark_ready()
        if save_dump:
            resource = self._create_resource(
                session_descr, dump_name,
                resource_types.EXECUTOR_DUMP,
                attrs=result_resource_attributes
            )
            resource.mark_ready()
        else:
            os.unlink(self.abs_path(dump_name))

        # save additional resources
        for (fname, rs_type, rs_path) in rss:
            fname = fname + '.' + str(session_id)
            shutil.copyfile(rs_path, os.path.join(working_dir, fname))
            resource = self._create_resource(session_descr, fname, rs_type)
            resource.mark_ready()

    def _trace(self, trace_name, subproc_list):
        if not self.ctx['do_trace']:
            return
        if not len(subproc_list):
            return

        from platform import system
        system_name = system()

        pid = subproc_list[0].pid
        trace_resource = self._create_resource(self.descr, trace_name, resource_types.TRACE_OUT)

        if system_name == 'FreeBSD':
            run_process('ktrace -C', wait=False)
            trace_cmd = ' '.join(['ktrace', '-p %d' % pid, '-f %s' % trace_resource.abs_path()])
            trace_process = run_process(trace_cmd)
            if trace_process.wait():
                raise SandboxTaskFailureError('(k)trace process died with exit code %d' % trace_process.returncode)
        elif system_name == 'Linux':
            trace_cmd = ' '.join(['strace', '-p %d' % pid, '-o %s' % trace_resource.abs_path()])
            trace_process = run_process(trace_cmd, wait=False)
            if trace_process.poll() is not None:
                if trace_process.returncode:
                    raise SandboxTaskFailureError('(s)trace process died with exit code %d' % trace_process.returncode)
            subproc_list.append(trace_process)

    # task for runnig
    def run_dolbilo(
        self, ctx, stat_pattern, rss, save_dump=False, start_once=False,
        total_sessions=None, plan_resource=None, warm_up=False,
        run_one_more_fuckup_session=False, **kvargs
    ):
        """
            Функция запускает поиск и обстреливает его доблилкой по данным из контекста таска.
            @ctx - контекст таска
            @stat_pattern - строчка для формирования названия отчётов (например, 'basesearch')
            @rss - ???
            @save_dump - сохранять или нет дамп обстрела. Если установлено в True, то создаётся соответствующий ресурс
            @start_once - перезапускать поиск каждый раз или только в начале обстрелов.
                Если установлен в True, поиск будет запущен перед всеми сессиями обстрелов, а завершён после них.
            @total_sessions - сколько раз обстреливать поиск.
                Если не задано значение, то оно берётся из контекста по ключу 'total_sessions'
            @warm_up: прогревать ли поиск перед первой стрельбой.
                Если равно True, запускается одна стрельба в devastate-режиме
            @run_one_more_fuckup_session: запустить ещё одну тестовую сессию в 4 потока в devastate режиме
        """
        def _parse_stat(ctx, stat_name):
            data = stats_parser.StatsParser(stat_name)

            n = data.vars.get('requests/sec', 0)

            try:
                ctx["requests_per_sec"].append(n)
            except KeyError:
                ctx["requests_per_sec"] = [n]

            return data
        self.ctx['dolbilka_test_sessions_info'] = {}
        subproc_list = []
        try:
            if not total_sessions:
                total_sessions = int(ctx['total_sessions'])
            else:
                total_sessions = int(total_sessions)
        except ValueError:
            raise SandboxTaskFailureError('Incorrect value for "total_sessions" option: %s' % total_sessions)
        if start_once:
            # prepare and start searcher
            subproc_list = self._start_searcher(**kvargs)
            self._trace('trace.out', subproc_list)
        if warm_up:
            logging.info('Run warm-up session')
            session = 'warm_up'
            dump_name = 'dump.%s.bin' % session
            stat_name = 'stat.%s.txt' % session
            self._run_session(
                session, subproc_list, dump_name, stat_name, rss,
                save_dump, plan_resource=plan_resource, executor_mode='devastate',
                simultaneous_requests=self.client_info['ncpu']
            )
        for session in range(0, total_sessions):
            # delete files with extra resources
            for (_, _, rs_path) in rss:
                try:
                    os.remove(rs_path)
                except OSError:  # FIXME: do not ignore
                    pass
            if not start_once:
                # remove load log for first session
                if ctx['save_load_log'] and session == 1:
                    resource_path = channel.sandbox.get_resource(ctx['load_log_resource_id']).path
                    remove_path(resource_path)

                # prepare and start searcher
                subproc_list = self._start_searcher(**kvargs)
                self._trace('trace.%03d.out' % session, subproc_list)

            dump_name = 'dump.%s.%d.bin' % (stat_pattern, session)
            stat_name = 'stat.%s.%d.txt' % (stat_pattern, session)

            # run session
            self._run_session(session, subproc_list, dump_name, stat_name, rss, save_dump, plan_resource=plan_resource)

            data = _parse_stat(ctx, self.abs_path(stat_name))

            if ctx['max_service_unavailable'] > 0:
                unavailable = data.vars.get('Service unavailable', 0)
                if unavailable > ctx['max_service_unavailable']:
                    for subproc in subproc_list:
                        os.kill(subproc.pid, 6)
                    raise SandboxTaskFailureError("Service unavailable = %s in %s" % (unavailable, stat_name))

            # subproc_list always has length of 1
            if len(subproc_list) == 1:
                self._save_memory_usage(subproc_list[0].pid, ctx)

            if not start_once:
                # stop searches
                self._stop_searcher(subproc_list, session, **kvargs)
        if run_one_more_fuckup_session:
            logging.info('Run one more devastate session')
            session = 'one_extra_fuckup_session'
            dump_name = 'dump.%s.bin' % session
            stat_name = 'stat.%s.txt' % session
            self._run_session(
                session, subproc_list, dump_name, stat_name, rss,
                save_dump, plan_resource=plan_resource,
                executor_mode='finger', simultaneous_requests=4
            )
        if start_once:
            self._stop_searcher(subproc_list, 'all', **kvargs)

        if ctx['save_load_log']:
            resource_path = channel.sandbox.get_resource(ctx['load_log_resource_id']).path
            if os.path.getsize(resource_path) == 0:
                raise SandboxTaskFailureError("load log is empty")
            self.mark_resource_ready(ctx['load_log_resource_id'])

    @staticmethod
    def _save_memory_usage(pid, ctx):
        """
            save memory usage after each session for memory leak detection
            save rss - resident set size, the non-swapped physical memory that a task has used
            and vsz - virtual memory size of the process in KiB (1024-byte units).
                Device mappings are currently excluded; this is subject to change.
        """
        info = get_process_info(pid, ['rss', 'vsz'], ignore_errors=False)

        if 'memory_rss' not in ctx:
            ctx['memory_rss'] = []
        ctx['memory_rss'].append(info['rss'])

        if 'memory_vsz' not in ctx:
            ctx['memory_vsz'] = []
        ctx['memory_vsz'].append(info['vsz'])
