import os
import os.path
import subprocess
import logging
import random
import shutil
import copy
import platform

from sandbox.sandboxsdk.channel import channel
from sandbox.sandboxsdk.process import run_process
from sandbox.sandboxsdk.task import SandboxTask
from sandbox.sandboxsdk.svn import Arcadia
from sandbox.sandboxsdk.errors import SandboxTaskFailureError

from sandbox.projects.common import apihelpers


class BuildExternalProgram(SandboxTask):
    type = None

    RESULT_RESOURCE_NAME = None
    RESULT_RESOURCE_CLASS = None

    OS_VERSIONS = ('freebsd-9.0', 'linux-3.2.0')

    def run_subtasks(self):
        platforms_hosts = {}

        for client in channel.sandbox.list_clients():
            logging.info('Check arch: %s, os: %s' % (client['arch'], client['os_version']))

            if not client['alive_now']:
                continue

            os_meta_name = ''
            os_string = '%s-%s' % (client['arch'], client['os_version'])

            for os_prefix in self.OS_VERSIONS:
                if os_string.startswith(os_prefix):
                    os_meta_name = os_prefix
                    break

            if len(os_meta_name) == 0:
                continue

            logging.info('good os_version %s' % client['os_version'])

            if os_meta_name in platforms_hosts:
                platforms_hosts[os_meta_name].append(client['hostname'])
            else:
                platforms_hosts[os_meta_name] = [client['hostname']]

        subtasks = []
        for _platform, hosts in platforms_hosts.items():
            random.shuffle(hosts)

            logging.info('create_subtask for %s' % _platform)

            child_ctx = copy.deepcopy(self.ctx)
            child_ctx['do_not_restart'] = True
            child_ctx['child_task'] = True

            task = self.create_subtask(
                self.type,
                'Building %s for %s' % (self.RESULT_RESOURCE_NAME, _platform),
                input_parameters=child_ctx,
                host=hosts[0],
                important=self.important,
            )
            subtasks.append(task)

        if len(subtasks) == 0:
            raise SandboxTaskFailureError("Have no platforms to run build on")

        self.ctx['subtasks_created'] = True
        self.wait_all_tasks_completed(subtasks)

    def clone_resource(self, resource):
        _platform = resource.attributes['platform']

        src_path = self.sync_resource(resource.id)
        dst_path = os.path.join(_platform, resource.file_name)
        dst_dir = os.path.dirname(dst_path)

        if not os.path.exists(dst_dir):
            os.makedirs(dst_dir)
        if os.path.isdir(src_path):
            shutil.copytree(src_path, dst_dir)
        else:
            shutil.copy(src_path, dst_dir)

        attributes = copy.deepcopy(resource.attributes)
        attributes["do_not_remove"] = True

        self.create_resource(
            description=resource.description,
            resource_path=dst_path,
            resource_type=self.RESULT_RESOURCE_CLASS,
            arch=resource.arch,
            attributes=attributes
        )

    def save_resources(self):
        for task in self.list_subtasks(load=True):
            if task.status == self.Status.FAILURE:
                raise SandboxTaskFailureError('Build in worker %s failed' % task.id)

            sub_resources = [r for r in
                             apihelpers.list_task_resources(task.id)
                             if r.type == self.RESULT_RESOURCE_NAME]

            if len(sub_resources) == 0:
                raise SandboxTaskFailureError("Worker %s hasn't %s resource" % (task.id, self.RESULT_RESOURCE_NAME))

            resource = max(sub_resources, key=lambda r: r.id)
            self.clone_resource(resource)

    def build_binary(self):
        raise NotImplementedError()

    def get_sources_resources(self):
        raise NotImplementedError()

    def build_resource(self):

        # if not os.path.exists('_tmp'):
        #     os.mkdir('_tmp')

        # env = dict(os.environ)

        # drop path set by skynet python binary
        # if 'LD_LIBRARY_PATH' in env:
        #     del env['LD_LIBRARY_PATH']

        # if 'LIBRARY_PATH' in env:
        #     del env['LIBRARY_PATH']

        # env['TMPDIR'] = self.abs_path('_tmp')

        os.chdir(self.abs_path('.'))

        self.get_sources_resources()
        binary_path = self.build_binary()

        if not len(binary_path):
            raise SandboxTaskFailureError("No output of build script (no binary name)")

        if not os.path.exists(binary_path):
            raise SandboxTaskFailureError("Can't find binary in task dir (%s)" % binary_path)

        self.create_resource(
            description="%s for %s" % (self.RESULT_RESOURCE_NAME, platform.platform()),
            resource_path=binary_path,
            resource_type=self.RESULT_RESOURCE_CLASS,
            arch=self.arch,
            attributes={
                "version": self.ctx['program_version'],
                "platform": platform.platform(),
                "isa": os.uname()[4]
            }
        )

    def get_program_version(self):
        resource = channel.sandbox.get_resource(self.ctx['program_source_resource_id'])
        program_version = resource.attributes.get('version')

        if program_version is None:
            raise SandboxTaskFailureError("No `version' attribute in program sources resource")

        return program_version

    def arcadia_info(self):
        return '', None, self.ctx['program_version']

    def on_execute(self):
        if 'child_task' in self.ctx:
            logging.info('Run build_resource')
            self.build_resource()
        else:
            if 'subtasks_created' in self.ctx:
                logging.info('Run save_resources')
                self.save_resources()
            else:
                self.ctx['program_version'] = self.get_program_version()

                logging.info('Run run_subtasks')
                self.run_subtasks()


class BuildExternalProgramByScript(BuildExternalProgram):

    BUILD_SCRIPT_SVN = None

    def build_binary(self):
        build_script_name = os.path.basename(self.BUILD_SCRIPT_SVN)

        Arcadia.export(self.BUILD_SCRIPT_SVN, "./" + build_script_name)

        build_script = channel.task.abs_path(build_script_name)

        if not os.path.exists(build_script):
            raise SandboxTaskFailureError("Can't find %s in task dir" % build_script)

        process = run_process(
            [build_script],
            shell=True,
            log_prefix=build_script_name,
            stdout=subprocess.PIPE,
            outputs_to_one_file=False,
            # environment=env
        )

        binary_path = process.stdout.readline().rstrip()

        return binary_path
