# -*- coding: utf-8 -*-

import logging
import requests
import copy

from sandbox.sandboxsdk.parameters import ResourceSelector
from sandbox.sandboxsdk.parameters import SandboxStringParameter

from sandbox.projects import resource_types

from sandbox.projects.common.BaseGetShardmapTask import ShardmapSvnUrlParameter
from sandbox.projects.common.BaseGetShardmapTask import ShardmapResourceType

from sandbox.projects.DeployNannyShardmap import ShardmapResource
from sandbox.projects.DeployNannyShardmap import NannyServiceNameParameter
from sandbox.projects.DeployNannyShardmap import NannyWaitBeforeDeployParameter
from sandbox.projects.DeployNannyShardmap import NannyWaitDeployParameter
from sandbox.projects.DeployNannyShardmap import NannyAutoDeployParameter

from sandbox.projects.SetItsValueTask import ItsRuchkaNameParameter
from sandbox.projects.SetItsValueTask import ItsValueNameParameter

from sandbox.projects import DeployNannyShardmap as DeployShardmapTask

from sandbox.projects import SetItsValueTask

from sandbox.projects.common import apihelpers
from sandbox.projects.common.nanny import nanny

from sandbox.sandboxsdk.sandboxapi import Sandbox
from sandbox.sandboxsdk.errors import SandboxTaskFailureError
from sandbox.sandboxsdk.task import SandboxTask
from sandbox.sandboxsdk.channel import channel
from sandbox.sandboxsdk.paths import copy_path


class ImportedShardmapResourceParameter(ResourceSelector):
    name = 'search_shardmap'
    description = 'Импортировать шардмап из другого ресурса'
    required = False


class NannyThumbServiceNameParameter(SandboxStringParameter):
    """
        Название сервиса тумбнейлов в nanny для автодеплоя
    """
    name = 'deployment_nanny_thumbs_service_name'
    description = 'Название сервиса тумбнейлов в nanny'
    default_value = False


class DeployMediaShardmap(SandboxTask):
    """
        Отслеживаем новый шардмап в svn директории и деплоим его в сервис.
        После успешного завершения даём ссылку на релиз шардмапа для выкатки
        его в продакшен.
    """

    API_HOST = None
    API_PORT = None
    # API_HOST = "nextus-dev.haze.yandex.net"
    # API_PORT = "8080"

    @property
    def footer(self):
        if self.is_finished():
            if NannyServiceNameParameter.name not in self.ctx:
                result = """
                    There is no new shardmap to deploy
                """
            elif 'deployment_shardmapname' not in self.ctx:
                result = """
                    Link to nanny service: <a href="https://nanny.yandex-team.ru/ui/#/services/catalog/%(nanny_service)s/">%(nanny_service)s</a>
                    There is no new shardmap to deploy
                """ % {
                    'nanny_service': self.ctx[NannyServiceNameParameter.name],
                }
            else:
                result = """
                    Link to nanny service: <a href="https://nanny.yandex-team.ru/ui/#/services/catalog/%(nanny_service)s/">%(nanny_service)s</a>
                    Task to release new shardmap %(shardmap_name)s: <a href="/task/%(shardmap_task)s/view">%(shardmap_task)s</a>
                """ % {
                    'nanny_service': self.ctx[NannyServiceNameParameter.name],
                    'shardmap_name': self.ctx['deployment_shardmapname'],
                    'shardmap_task': self.ctx['get_shardmap_task_id']
                }
        else:
            result = """
                Link to nanny service: <a href="https://nanny.yandex-team.ru/ui/#/services/catalog/%(nanny_service)s/">%(nanny_service)s</a>
            """ % {
                'nanny_service': self.ctx[NannyServiceNameParameter.name],
            }
        return result

    def _get_shardmap_attr(self, runtime_attrs, attr_to_get):
        # get shardmap name
        service_shardmap_dump = runtime_attrs[
            u'content'][u'resources'][u'sandbox_bsc_shard']
        if 'SANDBOX_SHARDMAP' in service_shardmap_dump[u'chosen_type']:
            production_shardmap_task_id = service_shardmap_dump[
                u'sandbox_shardmap'][u'task_id']
            shardmap_resource_type_str = service_shardmap_dump[
                u'sandbox_shardmap'][u'resource_type']
            shardmap_resource_type = getattr(resource_types, shardmap_resource_type_str)
            production_resource_id = apihelpers.get_task_resource_id(
                production_shardmap_task_id,
                shardmap_resource_type)
            attr = Sandbox().get_resource_attribute(
                production_resource_id,
                attr_to_get)
            if not attr:
                raise SandboxTaskFailureError(
                    "Couldn't get sandbox id with production shardmap for service %s" %
                    (self.ctx[NannyServiceNameParameter.name]))
            else:
                return attr
        else:
            raise SandboxTaskFailureError(
                "Selected service doesn't use shardmaps")

    def _get_active_snapshot(self, nanny_client):
        """
            Получить последний активный стейт сервиса из nanny
        """
        if 'production_snapshot' not in self.ctx:
            current_state = nanny_client.get_service_current_state(
                self.ctx[NannyServiceNameParameter.name])
            for snapshot in current_state['content']['active_snapshots']:
                if snapshot[u'state'] in (
                        'GENERATING',
                        'ACTIVATING',
                        'ACTIVE',
                        'PREPARING'):

                    self.ctx['production_snapshot'] = snapshot[u'snapshot_id']
                    return True
            raise SandboxTaskFailureError(
                "You have to set shardmap for service in nanny: %s" %
                (self.ctx[NannyServiceNameParameter.name]))

    def _get_nanny_attrs_by_snapshot(self, nanny_client, snapshot_id):
        """
            Получить атрибуты рантайма по снепшоту из сервисов 2.0
            (этого нет в стандартной библиотеке nanny)
        """
        resources_url = nanny_client._api_url + \
            '/v2/history/services/runtime_attrs/' + snapshot_id
        try:
            r = nanny_client._session.get(resources_url)
            r.raise_for_status()
        except requests.HTTPError as e:
            raise nanny._handle_http_error(e)
        except Exception as e:
            raise nanny.NannyApiException(
                'Failed to call Nanny API. Error: {}'.format(e))

        return r.json()

    def _get_deployment_shardmap_by_sandbox(self, production_timestamp):
        resources = Sandbox(
            host=self.API_HOST,
            port=self.API_PORT).list_resources(
            self._get_shardmap_resource_type(),
            status='READY',
            attribute_name='shardmap_timestamp',
            limit=20,
            order_by="-id")
        if resources is not None:
            for resource in reversed(resources):
                try:
                    resource_task = channel.sandbox.get_task(resource.task_id)
                    if resource_task.new_status == self.Status.SUCCESS or resource_task.new_status == self.Status.NOT_RELEASED:
                        logging.debug("Found new unreleased shardmap in task: %s. Comparing timestamps with production shardmap" % resource.task_id)
                        if production_timestamp < resource.attributes['shardmap_timestamp']:
                            logging.debug("Try to release former shardmap: %s" % resource.id)
                            self.ctx['get_shardmap_task_id'] = resource.task_id
                            self.ctx['deployment_timestamp'] = resource.attributes['shardmap_timestamp']
                            self.ctx['deployment_shardmapname'] = resource.attributes['shardmap_name']
                            self.ctx['deployment_version'] = resource.attributes['shardmap_version']
                            return resource.id
                        else:
                            logging.debug("Unrealesed shardmap %s is older than production %s" % (resource.attributes['shardmap_timestamp'], production_timestamp))
                    else:
                        logging.debug("Task %s has unappropriate status: %s" % (resource.task_id, resource_task.new_status))
                except:
                    logging.debug("Wrong resource: %s" % resource.id)
        logging.debug("There isn't new unreleased shardmaps in sandbox. Trying to get it from svn")
        return False

    def _get_deployment_shardmap_by_svn(self):
        """
            Запускаем таску, которая пытается получить новый шардмап и ждёт её завершения.
        """
        if 'get_shardmap_task_id' not in self.ctx:
            resource_type = self._get_shardmap_resource_type()
            subtask_ctx = {
                ShardmapSvnUrlParameter.name: self.ctx[
                    ShardmapSvnUrlParameter.name],
                ShardmapResourceType.name: resource_type.__name__}
            get_shardmap_task = self.create_subtask(
                task_type=self._get_shardmap_task_type(),
                description=u'',
                input_parameters=subtask_ctx,
                priority=("SERVICE", "HIGH"),
                important=True,
                execution_space=256)
            self.ctx['get_shardmap_task_id'] = get_shardmap_task.id
            self.wait_all_tasks_stop_executing([get_shardmap_task.id])

        if ShardmapResource.name not in self.ctx:
            get_shardmap_task = channel.sandbox.get_task(
                self.ctx['get_shardmap_task_id'])
            shardmap_resource_list = apihelpers.list_task_resources(
                get_shardmap_task.id, self._get_shardmap_resource_type())
            if len(shardmap_resource_list) != 0:
                # get timestamp from deployment shardmap
                self.ctx['deployment_timestamp'] = Sandbox(
                    host=self.API_HOST,
                    port=self.API_PORT).get_resource_attribute(
                        shardmap_resource_list[0].id, u'shardmap_timestamp'
                )
                self.ctx['deployment_shardmapname'] = Sandbox(
                    host=self.API_HOST,
                    port=self.API_PORT).get_resource_attribute(
                    shardmap_resource_list[0].id, u'shardmap_name'
                )
                self.ctx['deployment_version'] = Sandbox(
                    host=self.API_HOST,
                    port=self.API_PORT).get_resource_attribute(
                    shardmap_resource_list[0].id, u'shardmap_version'
                )
                return shardmap_resource_list[0].id
            else:
                self.ctx['result'] = 'There is no new shardmap to deploy'
                logging.debug(
                    "There is no new shardmap")

    def _get_production_timestamp(self, nanny_client):
        """
            Получаем атрибут таймстемпа для шардмапа
        """

        self._get_active_snapshot(nanny_client)

        active_runtime_attrs = self._get_nanny_attrs_by_snapshot(
            nanny_client, self.ctx['production_snapshot'])
        service_shardmap_dump = active_runtime_attrs[
            u'content'][u'resources'][u'sandbox_bsc_shard']
        if 'SANDBOX_SHARDMAP' in service_shardmap_dump[u'chosen_type']:
            self.ctx['production_task_id'] = service_shardmap_dump[
                u'sandbox_shardmap'][u'task_id']
            self.ctx['production_resource_id'] = apihelpers.get_task_resource_id(
                self.ctx['production_task_id'],
                self._get_shardmap_resource_type())
            production_timestamp = Sandbox(
                host=self.API_HOST,
                port=self.API_PORT).get_resource_attribute(
                self.ctx['production_resource_id'],
                u'shardmap_timestamp')

            if not production_timestamp:
                raise SandboxTaskFailureError(
                    "Couldn't get sandbox id with production shardmap for service %s" %
                    (self.ctx[NannyServiceNameParameter.name]))
            logging.debug(
                "Get sandbox id with production shardmap for %s service: %s" %
                (self.ctx[
                    NannyServiceNameParameter.name],
                    self.ctx['production_resource_id']))
            return production_timestamp
        else:
            raise SandboxTaskFailureError(
                "Selected service doesn't use shardmaps")

    def _compare_timestamps(self):
        logging.debug(
            "Compare production shardmap timestamp %s with new one %s" %
            (self.ctx['production_timestamp'], self.ctx['deployment_timestamp']))
        # compare production timestamp with new one
        if int(
                self.ctx['production_timestamp']) < int(
                self.ctx['deployment_timestamp']):
            return True
        else:
            logging.debug(
                "New shardmap %s is older than in production: %s" %
                (self.ctx['deployment_timestamp'],
                 self.ctx['production_timestamp']))
            return False

    def _run_nanny_deploy(
            self,
            wait_before_deploy=False,
            wait_deploy=True,
            auto_deploy=True,
            service_name=None):
        """
            Запуск автодеплоя
        """
        if service_name is None:
            service_name = self.ctx[NannyServiceNameParameter.name]
        if 'deploy_shardmap_task_id' not in self.ctx:
            subtask_ctx = {
                NannyServiceNameParameter.name: service_name,
                ShardmapResource.name: self.ctx[
                    ShardmapResource.name],
                NannyWaitBeforeDeployParameter.name: wait_before_deploy,
                NannyWaitDeployParameter.name: wait_deploy,
                NannyAutoDeployParameter.name: auto_deploy
            }
            deploy_shardmap_task = self.create_subtask(
                task_type=DeployShardmapTask.DeployNannyShardmap.type,
                description=u'',
                input_parameters=subtask_ctx)
            self.ctx[
                'deploy_shardmap_task_id'] = deploy_shardmap_task.id
            self.wait_all_tasks_stop_executing([deploy_shardmap_task.id])
        else:
            deploy_shardmap_task = channel.sandbox.get_task(
                self.ctx['deploy_shardmap_task_id'])
            return deploy_shardmap_task

    def _run_nanny_deploy_neat(
            self,
            service_name,
            wait_before_deploy=False,
            wait_deploy=True,
            auto_deploy=True):
        """
            Запуск автодеплоя
        """
        subtask_ctx = {
            NannyServiceNameParameter.name: service_name,
            ShardmapResource.name: self.ctx[
                ShardmapResource.name],
            NannyWaitBeforeDeployParameter.name: wait_before_deploy,
            NannyWaitDeployParameter.name: wait_deploy,
            NannyAutoDeployParameter.name: auto_deploy
        }
        deploy_shardmap_task = self.create_subtask(
            task_type=DeployShardmapTask.DeployNannyShardmap.type,
            description=u'',
            input_parameters=subtask_ctx)
        return deploy_shardmap_task.id

    def _copy_resource(self, resource_id):
        resource = channel.sandbox.get_resource(resource_id)
        src_path = self.sync_resource(resource.id)
        dst_path = resource.file_name
        copy_path(src_path, dst_path)

        attributes = copy.copy(resource.attributes)
        attributes.pop('released', None)
        created_resource = self.create_resource(
            resource.description, dst_path, str(
                resource.type), arch=resource.arch, attributes=attributes)
        self.mark_resource_ready(created_resource)
        return created_resource.id

    def _update_its(self, payload):
        """
            Запускаем таск обновления ручек в ITS
        """
        subtasks = []
        if 'its_subtasks' not in self.ctx:
            for ruchka, value in payload.iteritems():
                subtask_ctx = {
                    ItsRuchkaNameParameter.name: ruchka,
                    ItsValueNameParameter.name: value,
                }
                its_task = self.create_subtask(
                    task_type=SetItsValueTask.SetItsValueTask.type,
                    description=u'',
                    input_parameters=subtask_ctx)
                subtasks.append(its_task.id)
            self.ctx['its_subtasks'] = subtasks
            # wait until ITS tasks completed
            self.wait_all_tasks_completed(self.ctx['its_subtasks'])

    def _get_thumbs_shardmap(self, nanny_client):
        """
            Получаем шардмап тумбнейлов
        """
        thumb_current_state = nanny_client.get_service_current_state(
            self.ctx[NannyThumbServiceNameParameter.name])
        thumb_service_status = thumb_current_state[u'content'][u'summary'][u'value']
        if thumb_service_status == u'ONLINE':
            for thumb_snapshot in thumb_current_state[u'content'][u'active_snapshots']:
                if thumb_snapshot[u'state'] in (u'ACTIVE'):
                    thumb_runtime_attrs = self._get_nanny_attrs_by_snapshot(nanny_client, thumb_snapshot[u'snapshot_id'])
                    return self._get_shardmap_attr(thumb_runtime_attrs, u'shardmap_timestamp')
        elif thumb_service_status in (u'UPDATING', u'PREPARING'):
            thumb_production_snapshot_id = thumb_current_state[u'content'][u'rollback_snapshot'][u'snapshot_id']
            for thumb_snapshot in thumb_current_state[u'content'][u'active_snapshots']:
                if thumb_snapshot[u'snapshot_id'] == thumb_production_snapshot_id:
                    thumb_runtime_attrs = self._get_nanny_attrs_by_snapshot(nanny_client, thumb_snapshot[u'snapshot_id'])
                    return self._get_shardmap_attr(thumb_runtime_attrs, u'shardmap_timestamp')

    def _get_shardmap_to_deploy(self, nanny_client, current_state):
        """
            Получаем шардмап для деплоя: если есть ресурс новее в sandbox'е, то от туда; если нет то пробуем из svn
        """
        for snapshot in current_state[u'content'][u'active_snapshots']:
            if snapshot[u'state'] in (u'ACTIVE'):
                # get runtime attr for production snapshot
                runtime_attrs = self._get_nanny_attrs_by_snapshot(nanny_client, snapshot[u'snapshot_id'])
                # get shardmap resource id from runtime attrs
                production_timestamp = self._get_shardmap_attr(runtime_attrs, u'shardmap_timestamp')
                # try to find newer shardmaps in sandbox
                shardmap_from_sandbox = self._get_deployment_shardmap_by_sandbox(production_timestamp)
                if shardmap_from_sandbox:
                    deployment_shardmap = shardmap_from_sandbox
                else:
                    # get shardmap to deploy from svn
                    deployment_shardmap = self._get_deployment_shardmap_by_svn()
                    if deployment_shardmap:
                        if self.ctx['deployment_timestamp'] <= production_timestamp:
                            return None
                return deployment_shardmap
