import copy
import hashlib
import json
import logging
import os
import shutil

from sandbox import common
from sandbox import sdk2
from sandbox.projects import resource_types
from sandbox.projects.common.arcadia import sdk as arcadiasdk
import sandbox.common.types.task as ctt

import cmd
import evlog
import patch
import ymake


# Used in AutocheckCMakeListsUnreachability and AutocheckDismissedOwners
class YmakeWrapper(object):
    BOOT_YA = 'ya'

    def __init__(self, srcRoot, logName, subcommands, args=None, buildProjects=None, use_autocheck_defines=False):
        self.srcRoot = srcRoot
        self.wrapper_path = os.path.join(srcRoot, self.BOOT_YA)
        self.cmdCtx = cmd.CmdCtx(logName, srcRoot)
        self.subcommands = subcommands
        self.args = args or []
        self.targets = buildProjects or []

    def ConstructCmd(self):
        sdk2.paths.add_executable_permissions_for_path(self.wrapper_path)
        return [self.wrapper_path] + self.subcommands + self.args + self.targets

    def Copy(self):
        return copy.deepcopy(self)


class YMakeCache(sdk2.Resource):
    """ Contains ymake.cache and ymake.json.cache """

    # common attributes
    releasable = True
    any_arch = True

    # custom attributes
    parameters_hash = sdk2.parameters.String('Caches separation attribute', required=True)
    cache_based = sdk2.parameters.Bool('Previous cache was used to create this one', required=True, default=False)
    arcadia_url = sdk2.parameters.String('Arcadia URL used to create the cache', required=True)


class YMakeDumps(sdk2.Resource):
    """ Contains results of ymake run """

    releasable = False
    any_arch = True


class YMakeBase(sdk2.Task):
    """ Run devtools/ymake/bin/ymake binary """

    arcadia_context = None      # Arcadia SDK mount point context manager

    YMAKE_INTERNAL_CACHE = 'ymake.cache'
    YMAKE_COMMANDS_CACHE = 'ymake.json.cache'

    class Requirements(sdk2.Task.Requirements):
        cores = 1
        ram = 20000

        class Caches(sdk2.Requirements.Caches):
            pass  # means that task do not use any shared caches

        tasks_resource = None

    class Parameters(sdk2.Task.Parameters):
        # common parameters
        kill_timeout = 4200
        max_restarts = 1

        # custom parameters

        arcadia_url = sdk2.parameters.ArcadiaUrl('Arcadia to run ymake on', required=False)

        use_arcadia_api = sdk2.parameters.Bool('Use arcadia-api fuse, not used', default=True)

        use_arc_instead_of_aapi = sdk2.parameters.Bool('Use arc fuse instead of aapi', default=False)

        ymake_bin = sdk2.parameters.Resource('YMake binary', resource_type=resource_types.YMAKE_BINARY, required=False)

        cache_resource = sdk2.parameters.Resource('Result of a previous run', resource_type=YMakeCache, required=False)

        use_cache = sdk2.parameters.Bool('Use cache')

        use_json_cache = sdk2.parameters.Bool('Use JSON cache')

        use_patch_mode = sdk2.parameters.Bool('Use ymake patch mode')
        with use_patch_mode.value[True]:
            use_arc_changelist = sdk2.parameters.Bool('Use arc for patch generating')
            fail_if_patch_unavailable = sdk2.parameters.Bool('Fail if patch is unavailable')

        gen_conf_opts = sdk2.parameters.List('Options for "ya dump conf"', value_type=sdk2.parameters.String, default=[])

        dumps_names = sdk2.parameters.List('Dumps names', value_type=sdk2.parameters.String, default=['graph.json', 'java.dart', 'test.dart', 'makefiles.dart'])

        ymake_opts = sdk2.parameters.List('Options for ymake', value_type=sdk2.parameters.String, default=['--xs', '-k', '-J', 'graph.json', '--xg'])

        targets = sdk2.parameters.List('Build targets', value_type=sdk2.parameters.String, default=[])

        hash_salt = sdk2.parameters.String('Task type (for cache hash calculation)', required=True, default='Manual run')

        dump_metrics = sdk2.parameters.Bool('Dump metrics from ymake run in Context', default=True)

    class Context(sdk2.Task.Context):
        parameters_hash = None
        metrics = None

    def prepare_source_root(self, url):
        if not arcadiasdk.fuse_available():
            raise common.errors.TaskFailure('Fuse unavailable')
        self.arcadia_context = arcadiasdk.mount_arc_path(
            url, use_arc_instead_of_aapi=self.Parameters.use_arc_instead_of_aapi, fetch_all=False
        )

    def load_caches(self, ymake_exec_info):
        use_cache = self.Parameters.use_cache
        use_json_cache = self.Parameters.use_json_cache
        if use_cache or use_json_cache:
            if not self.Parameters.cache_resource:
                logging.warning('No cache resource found, ignore "use_cache" and "use_json_cache"')
                return None
            if self.Parameters.cache_resource.parameters_hash != self.calculate_hash():
                logging.warning('Cache resource is incompatible with options, ignore "use_cache" and "use_json_cache"')
                return None
            ymake_exec_info.prepare_build_root()
            res = str(sdk2.ResourceData(self.Parameters.cache_resource).path)
            if use_cache:
                ymake_exec_info.load_cache(self.YMAKE_INTERNAL_CACHE, res)
            if use_json_cache:
                ymake_exec_info.load_cache(self.YMAKE_COMMANDS_CACHE, res)

    def save_caches(self, ymake_exec_info):
        resource = YMakeCache(
            task=self,
            description="Cache from YMakeBase task #{}".format(str(self.id)),
            path="cache",
            parameters_hash=self.calculate_hash(),
            cache_based=self.Parameters.use_cache or self.Parameters.use_json_cache,
            arcadia_url=self.Parameters.arcadia_url
        )

        data = sdk2.ResourceData(resource)
        data.path.mkdir(0o755, parents=True, exist_ok=True)
        resource_dir = str(data.path)
        ymake_exec_info.save_cache(self.YMAKE_INTERNAL_CACHE, resource_dir)
        ymake_exec_info.save_cache(self.YMAKE_COMMANDS_CACHE, resource_dir)
        self.Context.caches_resource_id = resource.id
        data.ready()

    def save_dumps(self, dumps):
        resource = YMakeDumps(
            task=self,
            description="Dumps from YMakeBase task #{}".format(str(self.id)),
            path="dumps"
        )

        data = sdk2.ResourceData(resource)
        data.path.mkdir(0o755, parents=True, exist_ok=True)
        resource_dir = str(data.path)

        for dump in dumps:
            if os.path.isfile(dump):
                shutil.move(dump, os.path.join(resource_dir, os.path.basename(dump)))

        self.Context.dumps_resource_id = resource.id
        data.ready()

    def load_ymake_binary(self, resource):
        return str(sdk2.ResourceData(resource).path)

    def calculate_hash(self):
        if self.Context.parameters_hash:
            return self.Context.parameters_hash

        md5 = hashlib.md5()
        items = (self.Parameters.ymake_opts,
                 self.Parameters.gen_conf_opts,
                 self.Parameters.hash_salt,
                 self.Parameters.targets)
        for item in items:
            md5.update(str(item))
        self.Context.parameters_hash = md5.hexdigest()
        return self.Context.parameters_hash

    def gen_conf(self, arcadia_dir):
        opts = self.Parameters.gen_conf_opts
        opts = filter(bool, opts)
        opts.append('-DJSON_CACHE_IS_ATTACHED=no')
        ymake_wrapper = YmakeWrapper(arcadia_dir, 'genconf', ['dump', 'conf'], opts)
        _, conf, _ = cmd.RunCmd(self, ymake_wrapper.ConstructCmd(), ymake_wrapper.cmdCtx)
        data = open(conf, 'r').read()
        data = data.replace(arcadia_dir, 'ARCADIA_DIR')
        open(conf, 'w').write(data)
        return conf

    def dump_metrics(self, ymake_err):
        metrics = evlog.YmakeEvLogConverter.extract_metrics(ymake_err)
        self.Context.metrics = json.dumps(metrics)

    def gen_patch(self, arcadia):
        old_arcadia = self.Parameters.cache_resource.arcadia_url
        new_arcadia = self.Parameters.arcadia_url
        old_arcadia = sdk2.vcs.svn.Arcadia.parse_url(old_arcadia)
        new_arcadia = sdk2.vcs.svn.Arcadia.parse_url(new_arcadia)
        if self.Parameters.use_arc_changelist:
            if str(self.Parameters.arcadia_url).startswith('arcadia-arc'):
                arc_repo = arcadia
            else:
                arc_repo = self.path('arc_repo')
                arc_repo.mkdir(0o755, parents=True, exist_ok=True)
                ctx = cmd.CmdCtx(logName='init_arc', workDir=str(arc_repo), outputExt='.out', errOutputExt='.err.out', wait=True)
                cmd.RunCmd(self, patch.gen_arc_init_command(), ctx)
            ctx = cmd.CmdCtx(logName='patch', workDir=str(arc_repo), outputExt='.out', errOutputExt='.err', wait=True)
            cmd.RunCmd(self, patch.gen_arc_diff_cmd(old_arcadia, new_arcadia), ctx)
            patch_path = str(self.log_path('patch.cl'))
            shutil.move(str(self.log_path('patch.out.log')), patch_path)
            return patch_path
        else:
            ctx = cmd.CmdCtx(logName='gen_patch', workDir=arcadia, outputExt='.out', errOutputExt='.err.out', wait=True)
        _, result, _ = cmd.RunCmd(self, patch.gen_svn_diff_cmd(old_arcadia, new_arcadia), ctx)
        patch_path = str(self.log_path('patch'))
        return patch.gen_zipatch(result, arcadia, patch_path)

    def on_create(self):
        self.Requirements.tasks_resource = sdk2.service_resources.SandboxTasksBinary.find(
            attrs={
                'release': ctt.ReleaseStatus.STABLE,
                'target': 'sandbox/projects/common/YMakeBase'
            }).first()

    def on_execute(self):
        if self.Parameters.use_patch_mode and not self.Parameters.use_cache:
            raise common.errors.TaskFailure('Incompatible options "use_patch_mode" and !"use_cache"')
        self.prepare_source_root(self.Parameters.arcadia_url)
        if self.Parameters.ymake_bin:
            ymake_bin = self.load_ymake_binary(self.Parameters.ymake_bin)
        else:
            ymake_bin = None
        targets = filter(bool, self.Parameters.targets)
        opts = filter(bool, self.Parameters.ymake_opts)
        opts = [str(self.path(x)) if x in self.Parameters.dumps_names else x for x in opts]
        if self.Parameters.dump_metrics:
            opts += ['-g', '--events', 'AO']

        ymake_exec_info = ymake.YMakeExecInfo(log_name='ymake', args=opts, build_root=str(self.path('build_root')),
                                              path=ymake_bin, targets=targets)
        self.load_caches(ymake_exec_info)

        with self.arcadia_context as arcadia:
            ymake_exec_info.conf = self.gen_conf(arcadia)
            if self.Parameters.use_patch_mode:
                try:
                    ymake_exec_info.patch = self.gen_patch(arcadia)
                except Exception as ex:
                    logging.warning('Unable to load path: ' + str(ex))
                    if self.Parameters.fail_if_patch_unavailable:
                        raise common.errors.TaskFailure('Path is unavailable')
            ymake_exec_info.cmd_context.workDir = ymake_exec_info.source_root = arcadia
            _, out, err = ymake_exec_info.execute(self)
            if self.Parameters.dump_metrics:
                self.dump_metrics(err)
            dumps = filter(bool, self.Parameters.dumps_names)
            self.save_dumps([out, err] + [str(self.path(x)) for x in dumps])

        self.save_caches(ymake_exec_info)
