# coding: utf-8

import itertools
import logging
import six

from sandbox import common
import sandbox.common.types.task as ctt

from sandbox.sandboxsdk import errors
from sandbox.sandboxsdk import channel as sdk_channel

from sandbox.projects import resource_types


channel = sdk_channel.channel


def list_task_resources(
    task_id,
    resource_type=None,
    arch=None,
    get_all_resources=False,
    limit=None,
    offset=None,
    **kwargs
):
    """
    List resources for specific task include hidden tasks

    :param task_id: task id
    :param resource_type: filter by resource type
    :param arch: filter by arch
    :param get_all_resources: omit broken resources
    :param limit count limit (no limit by default)
    :param offset cut off first `offset` resources

    :return: list of SandboxResource's (or empty list)
    """
    kwargs.update({
        'task_id': task_id,
        'arch': arch,
        'resource_type': resource_type,
        'hidden': True,
        'limit': limit,
        'offset': offset,
    })
    if not get_all_resources:
        kwargs['omit_failed'] = True
    task_resources = channel.rest.list_resources(**kwargs)
    return task_resources if task_resources else []


def list_task_dependenciens(
    task_id,
    resource_type=None,
    limit=10,
    offset=None,
    **kwargs
):
    kwargs.update({
        'dependant': task_id,
        'type': str(resource_type),
        'hidden': True,
        'limit': limit,
        'offset': offset,
    })

    rest = common.rest.Client()
    result = rest.resource.read(**kwargs)

    return [channel.sandbox.get_resource(r['id']) for r in result.get('items', [])]


def get_last_resource(resource_type, params=None):
    """
    Get last resource selected type

    :param resource_type: selected resource type
    :param params: dict with additional params& see sandboxapi.list_resources for it

    :return: SandboxResource instance (or None, if nothing found)
    """
    resource_type = str(resource_type)
    if not params:
        params = {}
    if 'status' not in params:
        params['status'] = 'READY'
    if 'arch' not in params:
        params['arch'] = common.utils.get_sysparams()['arch']
    if 'omit_failed' not in params:
        params['omit_failed'] = False
    if 'hidden' not in params:
        params['hidden'] = True
    if 'limit' not in params:
        params['limit'] = 1

    return next(iter(channel.rest.list_resources(resource_type=resource_type, **params) or []), None)


def get_last_resource_with_attrs(resource_type, attrs, all_attrs=False, params=None):
    """
    Get last resource selected type with specified attributes

    :param resource_type: selected resource type
    :param attrs: attributes dict (if value equal None - value ignored)
    :param all_attrs: search for all attrs or any attrs
    :param params: aditional params& see sandboxapi.list_resources for it

    :return: SandboxResource instance
    """
    if not params:
        params = {}
    params['status'] = 'READY'
    params['limit'] = 1
    #
    resources = channel.rest.list_resources(
        resource_type=resource_type,
        all_attrs=(attrs if all_attrs else None),
        any_attrs=(attrs if not all_attrs else None),
        **params
    )
    return resources[0] if resources else None


def get_resources_with_attribute(
    resource_type,
    attribute_name=None, attribute_value=None,
    limit=10, offset=None,
    status=None, arch=None, owner=None
):
    """
    Get resources list of selected type with specified attributes

    :param resource_type: selected resource type
    :param attribute_name: attribute name or None
    :param attribute_value: attribute value or None
    :param limit: limit list with
    :param offset: offset list with
    :param status: resource status
    :param arch: resource arch or platform
    :param owner: resource owner

    :return: list of SandboxResource's
    """
    check_res(resource_type)
    if status is None:
        status = 'READY'
    return channel.rest.list_resources(
        resource_type=resource_type, limit=limit, status=status, offset=offset,
        attribute_name=attribute_name, attribute_value=attribute_value,
        arch=arch, owner=owner
    )


def get_last_resource_with_attribute(
    resource_type,
    attribute_name=None,
    attribute_value=None,
    status=None,
    arch=None,
    owner=None,
):
    """
    Get last resource selected type with specified attribute

    :param resource_type: selected resource type
    :param attribute_name: attribute name or None
    :param attribute_value: attribute value or None
    :param status: resource status
    :param arch: resource arch or platform
    :param owner: resource owner

    :return: last (by time) resource object
    """
    resources_list = get_resources_with_attribute(
        resource_type=resource_type, attribute_name=attribute_name,
        attribute_value=attribute_value, status=status, arch=arch,
        owner=owner, limit=1
    )
    if resources_list:
        return resources_list[0]


def get_last_released_resource(
    resource_type, release_status=ctt.ReleaseStatus.STABLE, arch=None, task_type=None, order_by='-id',
):
    """
    !!! DEPRECATED !!! USE rest api method for resources with released=stable attribute !!!
    Get latest released resource of given type. By default, only STABLE releases
    are taken into account (the most common case used). If task_type specified - check 10 last releases to have
    specified task type and return last of them.

    :param resource_type: resource type (string or AbstractResource successor)
    :param release_status release status (STABLE by default)
    :param arch: resource arch

    :return: SandboxResource instance or None if nothing was found
    """
    check_res(resource_type)
    arch = arch if arch else common.utils.get_sysparams()['arch']
    params = {
        "resource_type": resource_type,
        "arch": arch,
        "release_status": release_status,
        "order_by": order_by,
        "limit": 10,  # fix strange behavior of list_releases method
    }
    logging.debug("Look for release with params: %s", params)
    stable_releases = channel.sandbox.list_releases(**params)
    logging.debug("Last 10 stable releases: %s", stable_releases)
    if task_type is None:
        resources = list(itertools.chain.from_iterable(sr.resources for sr in stable_releases))
        return resources[0] if resources else None
    else:
        for res in stable_releases:
            task_id = res.task_id
            if channel.sandbox.get_task(task_id).type == task_type:
                return res


def match_attributes(r, attrs, all_attrs):
    matcher = all if all_attrs else any
    return matcher(a in r and r[a] == val for a, val in six.iteritems(attrs))


def get_last_released_resource_with_attr(
    resource_type, attrs, release_status=ctt.ReleaseStatus.STABLE, all_attrs=False, arch=None
):
    """
    Get latest released resource of given type. By default, only STABLE releases
    are taken into account (the most common case used).

    :param resource_type: resource type (string or AbstractResource successor)
    :param release_status release status (STABLE by default)
    :param arch: resource arch

    :return: SandboxResource instance or None if nothing was found
    """
    check_res(resource_type)
    arch = arch if arch else common.utils.get_sysparams()['arch']

    offset = 0
    page_count = 20
    while True:
        stable_releases = channel.sandbox.list_releases(
            resource_type=resource_type,
            arch=arch,
            limit=page_count,
            offset=offset,
            release_status=release_status,
        )

        if not stable_releases:
            return None

        appropriate = next(
            (
                x for x in itertools.chain.from_iterable(stable_release.resources for stable_release in stable_releases)
                if match_attributes(x.attributes, attrs, all_attrs)
            ),
            None
        )

        if appropriate:
            return appropriate

        offset += page_count

    return None


def get_task_resource_id(task_id, resource_type, arch=None, **kwargs):
    """
    Возвращает идентификатор ресурса указанной задачи
    Дополнительно можно уточнять архитектуру ресурса

    Если ресурс не найден - кидается исключение
    """

    resources = list_task_resources(
        task_id,
        resource_type=resource_type,
        arch=arch,
        **kwargs
    )
    if resources:
        return resources[0].id

    raise errors.SandboxTaskFailureError(
        'Cannot find resource {} in task {} (arch={})'.format(
            resource_type, task_id, arch
        )
    )


def check_res(resource_type):
    if not isinstance(resource_type, (six.string_types, resource_types.AbstractResource)):
        errors.SandboxTaskFailureError("Wrong type of resource: {}".format(type(resource_type)))
