import copy
import json
import logging
import os
import six
import requests
from requests.adapters import HTTPAdapter
from requests.exceptions import ConnectionError, ReadTimeout
import socket
import random
import string
import shutil
import stat

from sandbox.common.fs import make_folder
from sandbox.sandboxsdk.channel import channel
from sandbox.sandboxsdk.parameters import ResourceSelector
from sandbox.sandboxsdk.process import run_process
from sandbox.sandboxsdk.svn import Arcadia
from sandbox.sandboxsdk.paths import copy_path


from sandbox.projects import resource_types
from sandbox.projects.common.profiling import gperftools as profiler
from sandbox.projects.common.utils import sync_resource
from sandbox.projects.common.utils import wait_searcher_start, wait_searcher_stop


def has_error(lines):
    if not lines:
        return False
    for line in lines:
        if line.find("[ERROR]") != -1:
            return True
    return False


def quantile(histogram, perc=None):
    import numpy

    if perc is None:
        perc = [0.5, 0.75, 0.9, 0.95, 0.99, 0.995, 0.999, 0.9999]

    bin_counts = [x for idx, x in histogram]
    cumulative_sum = numpy.cumsum(bin_counts)
    maximum_sum = cumulative_sum[-1]

    res = []
    for p in perc:
        p_value = maximum_sum * p

        for idx, i in enumerate(cumulative_sum):
            if i > p_value:
                if idx == 0:
                    res.append([p, histogram[idx][0]])
                else:
                    res.append([p, histogram[idx - 1][0]])
                break

    return res


def remove_readonly(func, path, _):
    os.chmod(path, stat.S_IWRITE)
    func(path)


def json_dump(j, fp):
    return json.dump(j, fp, indent=4, sort_keys=True, separators=(',', ': '))


BASE_BENCHMARK_CONFIG = {
    "app_host": {
        "type": "app_host",
        "options": {
            "config": {
                "port": 13300,
                "threads": 1,
                "executor_threads": 2,
                "total_quota": 1000,
                "group_quotas": {},
                "engine": {
                    "async": 1
                },
                "conf_dir": "graphs/",
                "log": "app_host.event.log",
                "update_from_fs_config": {}
            },
            "graphs": {
                "shoot.json": {
                    "graph": {
                        "RESPONSE": [
                            "INIT",
                            "SOURCE_1"
                        ],
                        "SOURCE_1": [
                            "INIT"
                        ]
                    },
                    "name": "shoot",
                    "sources": {
                        "SOURCE_1": {
                            "backend_config": {
                                "backend_descrs": [
                                    {
                                        "ip": "127.0.0.1",
                                        "host": "localhost",
                                        "protocol": "post",
                                        "port": 13400,
                                        "weight": 1.0,
                                    }
                                ],
                            },
                            "use_encryption": False,
                            "use_grpc": False,
                            "timeout": 500
                        }
                    }
                }
            }
        }
    },
    "perftest": {
        "type": "perftest_servant",
        "options": {
            "port": 13400,
            "threads": 1,
            "response_type": "perftest-response",
            "trash_prob": 0.0,
            "resp_time_distribution": "0,0,0",
            "resp_size_distribution": "1024,0,0"
        }
    }
}


class BinaryProvider(object):
    def __init__(self):
        self.binary = "binary"
        self.folder = "binary"
        self.ports = {"main": "80"}
        self.protocol_for_backends = "main"
        self.ctx = None
        self.options = {}
        self.cmd_tpl = "{binary} -c {config}"
        self.use_profiler = True
        self.use_gperftools = True
        self.process = None
        self.handle = "_golovan"
        self.metrics = None
        self.human_name = None
        self.check_port_url = "admin?action=ping"
        self.check_port_reply = "pong"
        self.save_unistat = True

        self.resources = {}
        self.files = {}

    @property
    def port(self):
        return self.ports.get("main")

    def prepare(self):
        pass

    def _stop(self):
        pass

    def _log_prefix(self):
        return "{}".format(self.binary.split('/')[-1])

    # here we are mapping resources key to self.key when we download resources
    def _download_resources(self):
        for resource_name, resource in self.resources.iteritems():
            resource_id = resource.get('id')
            resource_type = resource.get('type')
            path = sync_resource(resource_id)

            if resource_type == "file":
                make_folder(os.path.join(self.ctx['abs_path'], self.folder))
                cmd = 'tar zxvf {path} --strip-components=1 -C {folder}'.format(path=path,
                                                                                folder=os.path.join(
                                                                                    self.ctx['abs_path'],
                                                                                    self.folder))
                run_process(cmd.split(), shell=True, check=True, log_prefix=self._log_prefix() + "_tar")

                file_name = getattr(self, resource_name)
                old_path = os.path.join(self.ctx['abs_path'], self.folder, file_name)
                if resource_name == "binary":
                    file_name = self.human_name
                new_path = os.path.join(self.ctx['abs_path'], self.folder, '{}-{}'.format(file_name, self.port))

                logging.debug("{} {} {} {}".format(self.folder, old_path, new_path, getattr(self, resource_name)))

                copy_path(old_path, new_path)
                # st = os.stat(new_path)
                # os.chmod(new_path, st.st_mode | stat.S_IEXEC)

                # self.binary = new_path
                setattr(self, resource_name, new_path)

                logging.debug(getattr(self, resource_name))

            elif resource_type == "directory":
                new_path = os.path.join(self.ctx['abs_path'], "{}-{}".format(resource_name, self.port))
                if os.path.isdir(new_path):
                    logging.debug("Removing {}".format(new_path))
                    shutil.rmtree(new_path, onerror=remove_readonly)

                # copy from atom_candidates dir to rerankd/candidates
                copy_path(path, new_path)

                # self.candidates = new_path
                setattr(self, resource_name, new_path)

                logging.debug(getattr(self, resource_name))
                logging.debug(new_path)

            elif resource_type == "unpacked_file":
                new_folder = os.path.join(self.ctx["abs_path"], self.folder)
                make_folder(new_folder)
                copy_path(path, new_folder)

    def alive(self):
        if not self.process:
            return False

        return all([self.process.poll() is None, self.check_port()])

    def check_port(self):
        base_url = "http://localhost:{port}/{url}".format(port=self.port, url=self.check_port_url)
        s = requests.Session()
        s.mount(base_url, HTTPAdapter(max_retries=5))
        r = s.get(base_url, timeout=5)

        result = str(self.check_port_reply) in str(r.text)

        logging.debug("Checking port {url} instance: {answer}, result: {result}".format(url=base_url, answer=r.text,
                                                                                        result=result))
        return result

    def _get_run_cmd(self):
        pass

    def start(self):
        cmd = self._get_run_cmd()
        environment = dict(os.environ)
        os.umask(0o000)
        if self.use_profiler:
            if self.use_gperftools:
                env_upd = profiler.get_profiler_environment(
                    use_gperftools=self.use_gperftools,
                    executable_path=self.binary,
                    session_name=str(self.port)
                )
            else:
                env_upd = profiler.get_profiler_environment()
            environment.update(env_upd)

        logging.info(
            'Starting {} on port {} with command `{}`'.format(
                self.binary,
                self.port,
                cmd))
        if not self.alive():
            logging.debug(cmd.split())
            self.process = run_process(
                cmd.split(),
                outputs_to_one_file=False,
                wait=False,
                log_prefix=self._log_prefix(),
                environment=environment)
            logging.info(
                '===waiting for {} to start {}:{}'.format(
                    self._log_prefix(),
                    socket.gethostname(),
                    self.port)
            )
            wait_searcher_start(
                'localhost',
                self.port,
                subproc_list=[self.process],
                timeout=5 * 60)
            if self.check_port():
                logging.info("Evertything started on port {}".format(self.port))
        else:
            logging.warn('{} still alive'.format(self._log_prefix()))

    def _stop_action(self):
        pass

    def stop(self):
        logging.info('stopping {}'.format(self._log_prefix()))
        if not self.alive():
            raise Exception('{} is dead'.format(self._log_prefix()))
        else:
            try:
                self._stop_action()
                self.process.terminate()
            except ConnectionError:
                pass
            try:
                wait_searcher_stop(
                    socket.gethostname(),
                    self.port,
                    timeout=30)
            except ConnectionError:
                pass
            except ReadTimeout:
                pass
            self.process = None
            if self.use_profiler:
                profiler.read_profile_data(
                    self.binary,
                    None,
                    str(self.port),
                    use_gperftools=self.use_gperftools)

    def _save_unistat(self):
        if self.save_unistat and self.alive():
            path = "unistat.{app}.json".format(app=self._log_prefix())
            req = requests.get('http://localhost:{port}/{handle}'.format(port=self.port, handle=self.handle))

            self._save_unistat_metrics(req)

            with open(path, "w") as fp:
                fp.write(req.text)
                fp.close()

            resource = channel.task.create_resource('{} unistat'.format(self._log_prefix()),
                                                    path,
                                                    resource_types.UNISTAT_RESPONSE)

            return resource
        return False

    def _save_eventlog(self, path):
        resource = channel.task.create_resource(
            '{} eventlog'.format(self._log_prefix()),
            path,
            resource_types.EVENTLOG_DUMP)

        return resource

    def _save_log(self, path):
        resource = channel.task.create_resource(
            '{} {}'.format(self._log_prefix(), path),
            path,
            resource_types.EVENTLOG_DUMP)
        return resource

    def _save_unistat_metrics(self, req):
        res = {}
        if self.metrics:
            for metric_name, metric_value in req.json():
                if metric_name in self.metrics:
                    if metric_name.endswith('_dhhh'):
                        res[metric_name] = quantile(metric_value)
                    else:
                        res[metric_name] = metric_value

            path = "metrics.{app}.json".format(app=self._log_prefix())

            with open(path, "w") as fp:
                json_dump(res, fp)

            channel.task.create_resource('{} stat'.format(self._log_prefix()),
                                         path,
                                         resource_types.BENCHMARK_METRICS,
                                         attributes={"app": self.human_name})

    def save(self):
        pass

    def __enter__(self):
        self.start()
        return self

    def __exit__(self, exit_type, value, traceback):
        self.stop()


class AppHostProvider(BinaryProvider):
    def __init__(self):
        super(AppHostProvider, self).__init__()
        self.cmd_tpl = '{binary} --config {config}'
        self.binary = 'app_host'
        self.folder = 'app_host'
        self.binary_name = 'app_host'
        self.metrics = ['SELF-SS-Requests_dmmm', 'SELF-SS-Successes_dmmm', 'SELF-SS-Failures_dmmm',
                        'SELF-SS-AverageResponseTimeMcs_ammv', 'SELF-SS-ResponseTimesMcsHistogram_dhhh',
                        'SELF-SS-ResponseSizeHistogram_dhhh']
        self.response_time_hist_intervals = [
            {
                "left": 0,
                "right": 5000,
                "precision": 10
            },
            {
                "left": 5000,
                "right": 10000,
                "precision": 100
            },
            {
                "left": 10000,
                "right": 20000,
                "precision": 200
            },
            {
                "left": 20000,
                "right": 50000,
                "precision": 500
            },
            {
                "left": 50000,
                "right": 1000000,
                "precision": 500
            }
        ]
        self.response_size_hist_intervals = [
            {
                "left": 0,
                "right": 5000,
                "precision": 10
            },
            {
                "left": 5000,
                "right": 10000,
                "precision": 100
            },
            {
                "left": 10000,
                "right": 20000,
                "precision": 200
            },
            {
                "left": 20000,
                "right": 50000,
                "precision": 500
            },
            {
                "left": 50000,
                "right": 1000000,
                "precision": 500
            }
        ]

    def _stop_action(self):
        for _ in six.moves.range(0, 3):
            try:
                requests.get(
                    'http://localhost:{}/admin?action=shutdown'.format(
                        self.port),
                    timeout=15)
            except ReadTimeout:
                pass

    def _config_path(self):
        return os.path.join(self.ctx['abs_path'], '{}.json'.format(self._log_prefix()))

    def _get_run_cmd(self):
        return self.cmd_tpl.format(binary=self.binary, config=self._config_path())

    def prepare(self):
        self._download_resources()

        self.options.get('config')['port'] = self.port
        graphs_path = os.path.join(self.ctx['abs_path'], self.options.get('config').get('conf_dir'))
        logs_path = os.path.join(self.ctx['abs_path'], self.options.get('config').get('log'))
        self.options.get('config')['conf_dir'] = graphs_path
        self.options.get('config')['response_time_hist_intervals'] = self.response_time_hist_intervals
        self.options.get('config')['response_size_hist_intervals'] = self.response_size_hist_intervals
        self.options.get('config')['log'] = logs_path
        self.options.get('config')['default_load_control_config'] = {}
        with open(self._config_path(), 'w') as fh:
            json_dump(self.options.get('config'), fh)

        # write graph configs
        make_folder(graphs_path)

        def _get_backend(port, handle=None):
            srv = dict()
            srv['ip'] = '127.0.0.1'
            srv['host'] = 'localhost'
            srv['protocol'] = 'post'
            srv['port'] = port
            srv['weight'] = 1
            if handle:
                srv['path'] = handle
            return srv

        for graph_name, graph_content in self.options.get('graphs').iteritems():
            ports = []
            for i in ['perftest_servant', 'rerankd', 'src_setup', 'custom']:
                res = self.ctx.get('app_ports').get(i)
                if res:
                    ports = res
                    break

            logging.debug(graph_content)
            logging.debug(ports)
            if "SOURCE_1" in graph_content.get('sources').keys():
                graph_content.get('sources').get("SOURCE_1")["backend_config"] = {}
                graph_content.get('sources').get("SOURCE_1")["backend_config"]['backend_descrs'] = []
                for app_name, app_proto, app_port in self.ctx.get('app_ports').get('perftest_servant'):
                    if app_proto == self.protocol_for_backends:
                        graph_content.get('sources').get("SOURCE_1")["backend_config"]['backend_descrs'].append(
                            _get_backend(app_port))
            else:
                for app_name, app_proto, app_port in ports:
                    if app_name in graph_content.get('sources').keys() and app_proto == self.protocol_for_backends:
                        path = None
                        if graph_content.get('sources').get(app_name).get('path'):
                            path = graph_content.get('sources').get(app_name).get('path')
                        logging.debug(
                            "app_name {} app_port {} get_backend {} path {}".format(app_name, app_port,
                                                                                    _get_backend(app_port, path),
                                                                                    path))
                        graph_content.get('sources').get(app_name)['backend_config'] = {}
                        graph_content.get('sources').get(app_name)['backend_config']['backend_descrs'] = []
                        graph_content.get('sources').get(app_name)['backend_config']['backend_descrs'].append(
                            _get_backend(app_port, path))

            graph_path = os.path.join(graphs_path, graph_name)

            logging.debug('creating graph config `{}` with content `{}`'.format(graph_path, graph_content))

            with open(graph_path, 'w') as fh:
                json_dump(graph_content, fh)

    def save(self):
        self._save_eventlog('{}-{}'.format(self.options.get('config').get('log'), self.port))
        self._save_unistat()


class PerfTestServantProvider(BinaryProvider):
    def __init__(self):
        super(PerfTestServantProvider, self).__init__()
        self.binary = 'perftest_servant'
        self.folder = 'perftest_servant'
        self.responses = None
        self.cmd_tpl = '{binary} -p {port} -P {grpc_port} -T {threads}' \
                       ' -t {response_type}' \
                       ' --trash-prob {trash_prob}' \
                       ' --structured-trash-prob {structured_trash_prob}' \
                       ' --consistency-check-aware-trash-prob {consistency_check_aware_trash_prob}' \
                       ' --break-some-bytes-prob {break_some_bytes_prob} --broken-byte-count {broken_byte_count}' \
                       ' --resp-time-distribution {resp_time_distribution}' \
                       ' --resp-size-distribution {resp_size_distribution}' \
                       ' {custom_options}'

    def _get_run_cmd(self):
        custom_options = ""
        custom_options += self.options.get('custom_options', "")

        return self.cmd_tpl.format(
            binary=self.binary,
            grpc_port=self.ports.get("grpc"),
            port=self.port,
            response_type=self.options.get('response_type', 'perftest-response'),
            threads=self.options.get('threads', 16),
            trash_prob=self.options.get('trash_prob', 0),
            structured_trash_prob=self.options.get('structured_trash_prob', 0),
            consistency_check_aware_trash_prob=self.options.get('consistency_check_aware_trash_prob', 0),
            break_some_bytes_prob=self.options.get('break_some_bytes_prob', 0),
            broken_byte_count=self.options.get('broken_byte_count', 1),
            resp_time_distribution=self.options.get('resp_time_distribution', '0,0,0'),
            resp_size_distribution=self.options.get('resp_size_distribution', '8192,0,0'),
            responses=self.responses,
            resp_time_samples=self.options.get('resp_time_samples', False),
            custom_options=custom_options
        )

    def prepare(self):
        self._download_resources()

        if self.options.get('response_type') == 'perftest-response' and self.responses:
            self.cmd_tpl = "%s --resps-path {responses}" % self.cmd_tpl

        if self.options.get('resp_time_samples', False):
            self.cmd_tpl = "%s --resp-time-samples {resp_time_samples}" % self.cmd_tpl

    def save(self):
        self._save_unistat()


class HttpAdapterProvider(BinaryProvider):
    def __init__(self):
        super(HttpAdapterProvider, self).__init__()
        self.binary = 'http_adapter'
        self.config = "http_adapter-{}.json".format(self.port)
        self.folder = 'http_adapter'
        self.cmd_tpl = "{binary} --config {main_config} -p {port}"
        self.new_config = ''
        self.metrics = ['SELF-SS-Requests_dmmm', 'SELF-SS-Errors_dmmm"', 'SELF-SS-Successes_dmmm',
                        "SELF-SS-ResponseTimesMcsHistogram_dhhh", "SELF-SS-ResponseSizeHistogram_dhhh"]
        self.response_time_hist_intervals = [
            {
                "left": 0,
                "right": 5000,
                "precision": 10
            },
            {
                "left": 5000,
                "right": 10000,
                "precision": 100
            },
            {
                "left": 10000,
                "right": 20000,
                "precision": 200
            },
            {
                "left": 20000,
                "right": 50000,
                "precision": 500
            },
            {
                "left": 50000,
                "right": 1000000,
                "precision": 500
            }
        ]
        self.response_size_hist_intervals = [
            {
                "left": 0,
                "right": 5000,
                "precision": 10
            },
            {
                "left": 5000,
                "right": 10000,
                "precision": 100
            },
            {
                "left": 10000,
                "right": 20000,
                "precision": 200
            },
            {
                "left": 20000,
                "right": 50000,
                "precision": 500
            },
            {
                "left": 50000,
                "right": 1000000,
                "precision": 500
            }
        ]

    def _get_run_cmd(self):
        return self.cmd_tpl.format(binary=self.binary,
                                   port=self.port,
                                   main_config=self.config)

    def _get_main_config(self, app_ports):
        compression = self.options.get("enable_compression", True)
        config = {
            "access_log": "{}/current-access_log-http_adapter".format(self.ctx["abs_path"]),
            "access_log_from_report_proxy": "{}/current-access_log_from_report_proxy-http_adapter".format(self.ctx["abs_path"]),
            "request_log": "{}/current-request_log-http_adapter".format(self.ctx["abs_path"]),
            "backend_config": {
                "grpc_port_offset": 1,
                "backends": [
                ],
                "timeout": 10000
            },
            "http_server_options": {
                "enable_compression": compression
            },
            "response_time_hist_intervals": self.response_time_hist_intervals,
            "response_size_hist_intervals": self.response_size_hist_intervals,
            "log": "{}/current-eventlog-http_adapter".format(self.ctx["abs_path"]),
            "logging_config": {
                "log_full_response": False
            },
            "protocol_options": {
                "post/ConnectTimeout": "25ms"
            },
            "routing_config_path": "{}/routing_config-{}.json".format(self.ctx["abs_path"], self.port)
        }

        for port in app_ports:
            config.get("backend_config")["backends"].append("post://localhost:{}".format(port))

        return config

    @staticmethod
    def _get_routing_config():
        return {
            "path2graph": {
                "/request": "shoot"
            }
        }

    def _create_new_config(self):
        app_to_shoot = self.options.get("app_to_shoot", "app_host")
        apps = self.ctx.get('app_ports').get(app_to_shoot, [["app_host", "main", 9999]])
        app_ports = []
        for app_name, app_proto, app_port in apps:
            if app_proto == self.protocol_for_backends:
                app_ports.append(app_port)

        self.config = os.path.join(self.ctx['abs_path'], "http_adapter-{}.json".format(self.port))
        self.routing_config = os.path.join(self.ctx['abs_path'], "routing_config-{}.json".format(self.port))

        with open(self.config, "w") as fp:
            json_dump(self._get_main_config(app_ports), fp)

        with open(self.routing_config, "w") as fp:
            json_dump(self._get_routing_config(), fp)

    def prepare(self):
        self._download_resources()
        self._create_new_config()

    def save(self):
        self._save_unistat()
        self._save_eventlog("current-eventlog-http_adapter-{}".format(self.port))
        self._save_log("current-access_log-http_adapter-{}".format(self.port))


class CustomProvider(BinaryProvider):
    def __init__(self):
        super(CustomProvider, self).__init__()
        self.binary = 'custom'
        self.config = None
        self.folder = 'custom'
        self.cmd_tpl = "{binary} {port} {path}"
        self.new_config = ''

    def _get_run_cmd(self):
        return self.cmd_tpl.format(binary=self.binary,
                                   port=self.port,
                                   path=os.path.join(self.ctx["abs_path"], self.folder))

    def _stop_action(self):
        for _ in six.moves.range(0, 3):
            try:
                requests.get(
                    'http://localhost:{}/admin?action=shutdown'.format(
                        self.port),
                    timeout=5)
            except:
                pass

            try:
                requests.get(
                    'http://localhost:{}/admin?action=shutdown'.format(
                        self.ports.get("http")),
                    timeout=5)
            except:
                pass

    def prepare(self):
        self._download_resources()
        self.metrics = self.options.get('metrics', {})

    def save(self):
        self._save_unistat()


class SportProxyProvider(BinaryProvider):
    def __init__(self):
        super(SportProxyProvider, self).__init__()
        self.binary = "sport_proxy"
        self.config = None
        self.folder = "sport_proxy_folder"
        self.command_template = " ".join(["ListenPort={http_port}",
                                          "EventsData={folder}/sport_proxy.data",
                                          "GeoDB={folder}/geodb.data",
                                          "SearchAppDids={folder}/search_app_dids.data",
                                          "UseCache=true",
                                          "{folder}/sport_proxy",
                                          "{folder}/sport_proxy.cfg"])

    def _get_run_cmd(self):
        return self.command_template.format(http_port=self.ports.get("http"),
                                            folder=os.path.join(self.ctx["abs_path"], self.folder))

    def _stop_action(self):
        requests.get(
            "http://localhost:{}/admin?action=shutdown".format(
                self.ports.get("main")),
            timeout=5)
        requests.get(
            "http://localhost:{}/admin?action=shutdown".format(
                self.ports.get("http")),
            timeout=5)

    def prepare(self):
        self._download_resources()
        self.metrics = self.options.get("metrics", {})

    def save(self):
        self._save_unistat()


class Benchmark(object):
    desc = 'Abstract Benchmark'

    concurrency = 4
    benchmark_config_id = None
    dolbilo_plan_id = None
    app_host_bundle_id = None
    app_host_test_servants_id = None
    benchmark_config = copy.deepcopy(BASE_BENCHMARK_CONFIG)

    executor_mode = 'finger'
    fuckup_mode_max_simultaneous_requests = 1
    total_sessions = 1
    requests_limit = 100000
    app_to_shoot = 'app_host'
    ctx = None
    new_benchmark_config_id = None
    store_unistat = True
    use_grpc = False
    use_secure = False
    resources = dict()

    def __repr__(self):
        return self.desc

    def create_task(self, parent_task):
        self.setup(parent_task)

        logging.info('serialize {}'.format(json.dumps(self.ctx)))

        self.copy_context(parent_task)
        logging.debug("create_subtask ctx = {}".format(self.ctx))
        task = parent_task.create_subtask(task_type='BENCHMARK_APP_HOST', description=self.desc,
                                          input_parameters=self.ctx)
        return task

    def setup(self, parent_task):
        pass

    def add_resources(self, cfg):
        keys = [self.resources.get(key).get('type') for key in self.resources.keys()]
        if keys:
            logging.debug("keys: {}".format(keys))
        else:
            logging.error("Empty resources")

        for app, config in cfg.iteritems():
            app_type = config.get('type')
            logging.debug(app)
            logging.debug(app_type)
            if app_type in keys:
                cfg.get(app).setdefault("resources", {})
                logging.debug("resources: {}".format(self.resources))
                res = self.resources.get(app_type)
                logging.debug("res {}".format(res))
                res.pop('type', None)
                cfg.get(app)['resources'] = res
            else:
                logging.debug("No such resources for {app_type}".format(app_type=app_type))

        return cfg

    def copy_context(self, parent_task):
        ctx = copy.deepcopy(parent_task.ctx)
        logging.debug("self.new_benchmark_config_id {}".format(self.new_benchmark_config_id.id))
        ctx['APP_HOST_BENCHMARK_CONFIG'] = self.new_benchmark_config_id.id
        ctx['APP_HOST_BENCHMARK_PLAN'] = self.dolbilo_plan_id

        ctx['app_to_shoot'] = self.app_to_shoot

        ctx['executor_mode'] = self.executor_mode
        ctx['fuckup_mode_max_simultaneous_requests'] = self.fuckup_mode_max_simultaneous_requests
        ctx['total_sessions'] = self.total_sessions
        ctx['requests_limit'] = self.requests_limit

        ctx['store_unistat'] = self.store_unistat
        ctx['notify_via'] = None
        ctx['port_to_shoot'] = 'main'
        if self.use_secure:
            ctx['port_to_shoot'] = 'nehs'
        if self.use_grpc:
            ctx['port_to_shoot'] = 'grpc'
        if self.use_grpc and self.use_secure:
            ctx['port_to_shoot'] = 'grpcs'

        self.ctx = ctx

    @staticmethod
    def clean_template(cfg):
        cfg.pop('perftest', None)
        return cfg

    def create_config_resource(self, cfg, parent_task):
        path = 'benchmark.{}.json'.format(generate_random_name())

        with open(path, 'w') as fh:
            json_dump(cfg, fh)

        self.new_benchmark_config_id = parent_task.create_resource(
            self.desc,
            path,
            resource_types.APP_HOST_BENCHMARK_CONFIG,
            attributes={"type": "benchmark"})

        parent_task.mark_resource_ready(self.new_benchmark_config_id)

    def prepare_plan(self, parent_task):
        pass


def parse_release_version(url):
    parts = url.split("/")
    if parts[-1].startswith("arcadia") and parts[-3] == "releases":
        return "v%s" % parts[-2]

    parsed_url = Arcadia.parse_url(url)
    assert parsed_url.revision
    return "r%s" % parsed_url.revision


class SymbolsDumper(ResourceSelector):
    name = "breakpad_symbols_dumper_resource_id"
    resource_type = "BREAKPAD_SYMBOLS_DUMPER"
    description = "dump_syms"
    required = True


def gen_symdump(task, package_dir, build_dir, target, dump_file_name=None):
    dumper = task.sync_resource(task.ctx[SymbolsDumper.name])
    st = os.stat(dumper)
    os.chmod(dumper, st.st_mode | stat.S_IEXEC)
    with open(os.path.join(package_dir,
                           "%s.sym" % target[target.rfind("/") + 1:] if dump_file_name is None else dump_file_name),
              "w") as out:
        run_process(
            "%s %s" % (dumper, os.path.join(build_dir, target)), stdout=out
        )


def generate_random_name():
    return ''.join(random.choice(string.ascii_lowercase + string.digits) for _ in range(8))
