import six

from collections import OrderedDict

from sandbox import common
from sandbox.sandboxsdk import parameters

from sandbox.projects import resource_types
from sandbox.projects.app_host import resources as ah_resources
from sandbox.projects.common.dolbilka import resources as dolbilka_resources

VERTICALS = [
    'ALICE',
    'ATOM',
    'COMMON',
    'IMGS',
    'MAIL',
    'MAILCORP',
    'NEWS',
    'SHARED',
    'VIDEO',
    'VIDEO_HOSTING',
    'WEB',
    'WEATHER',
    'DISTRICT',
    'MUSIC',
    'GOODS',
]

ALL_BLIST = ('ALL_apphost', 'ALL')
MIN_BLIST = ('MINIMAL_apphost', 'MINIMAL')
COMPRESSING_BLIST = ('apphost_compressing', 'Compressing')
STREAMING_BLIST = ('apphost_streaming', 'Streaming')
SUBHOST_BLIST = ('apphost_subhost', 'Subhost')
FUZZING_BLIST = ('apphost_fuzzing', 'Fuzzing')
SIZE_BLIST = ('apphost_size', 'Size')
DELAY_BLIST = ('apphost_delay', 'Delay')
BREADTH_BLIST = ('apphost_breadth', 'Breadth')
BREADTH_WITH_DELAY_BLIST = ('apphost_breadth_delay', 'Breadth with delay')
BALANCING_CONT_BLIST = ('apphost_balancing_cont', 'Balancing Contention')
BALANCING_EFFICIENCY_BLIST = ('apphost_balancing_eff', 'Balancing Efficiency')
REQUEST_FRACTION_BLIST = ('apphost_request_fraction', 'Request Fraction')

HTTP_ADAPTER_BLIST = ('ALL_http_adapter', 'http_adapter')


ALL_BENCHMARKS = [
    ALL_BLIST, MIN_BLIST,
    COMPRESSING_BLIST, FUZZING_BLIST, SIZE_BLIST, DELAY_BLIST, BREADTH_BLIST, BREADTH_WITH_DELAY_BLIST,
    BALANCING_CONT_BLIST, BALANCING_EFFICIENCY_BLIST, REQUEST_FRACTION_BLIST,
    STREAMING_BLIST, SUBHOST_BLIST, HTTP_ADAPTER_BLIST
]

ALL_APPHOST = [
    COMPRESSING_BLIST, FUZZING_BLIST, SIZE_BLIST, DELAY_BLIST, BREADTH_BLIST, BREADTH_WITH_DELAY_BLIST,
    BALANCING_CONT_BLIST, BALANCING_EFFICIENCY_BLIST, REQUEST_FRACTION_BLIST,
    STREAMING_BLIST, SUBHOST_BLIST
]


MINIMAL_APPHOST = [FUZZING_BLIST, SIZE_BLIST, DELAY_BLIST]


def get_last_stable_resource(resource_type, release_status=None, raise_on_lookup_error=True):
    import sdk2
    from sandbox.sandboxsdk.errors import SandboxTaskFailureError
    try:
        releases = sdk2.task.Task.server.release.read(
            resource_type=resource_type,
            type="stable",
            limit=1
        )["items"]
        task_id = releases[0]["task_id"]
        resources = sdk2.task.Task.server.resource.read(
            task_id=task_id,
            type=resource_type,
            release_status=release_status,
            limit=1
        )["items"]
        return resources[0]["id"]
    except LookupError:
        if raise_on_lookup_error:
            raise SandboxTaskFailureError(
                "%s not found, either define a specific version or release smth" % resource_type.name)
        return None


class LastReleasedStableResource(parameters.ResourceSelector):
    """ Resource selector parameter defaults to latest stable released resource of given type. """

    @common.utils.classproperty
    def default_value(cls):
        resource_types = cls._get_resource_types()
        if len(resource_types) != 1:
            return None
        return get_last_stable_resource(resource_types[0], raise_on_lookup_error=False)

    @classmethod
    def get_resource_from_ctx(cls, ctx):
        resource = ctx.get(cls.name)
        if not resource:
            resource = get_last_stable_resource(cls.resource_type, release_status="stable")
        return resource


class ChooseVertical(parameters.SandboxBoolGroupParameter):
    name = 'choose_vertical'
    description = 'Choose vertical to build'
    choices = tuple([(v, v) for v in VERTICALS])

    default_value = ' '.join(VERTICALS)


def get_verticals(ctx):
    return list(set(ctx.get(ChooseVertical.name).split(' ')))


class GraphGenerator(LastReleasedStableResource):
    name = 'graph_generator'
    resource_type = resource_types.APP_HOST_GRAPH_GENERATOR_EXECUTABLE
    description = 'graph_generator resource'
    do_not_copy = True


class SmokeTest(LastReleasedStableResource):
    name = 'smoke_test'
    resource_type = resource_types.APP_HOST_SMOKE_TEST_EXECUTABLE
    description = 'smoke_test resource'
    do_not_copy = True


class PVA(LastReleasedStableResource):
    name = 'pva'
    resource_type = ah_resources.APP_HOST_TOOL_PVA_EXECUTABLE
    description = 'pva resource'
    do_not_copy = True


class LastReleasedAppHostBundle(LastReleasedStableResource):
    name = 'APP_HOST_BUNDLE'
    resource_type = resource_types.APP_HOST_BUNDLE
    description = 'APP_HOST_BUNDLE resource'
    do_not_copy = True
    required = True


def parameters_to_dict(params, group):
    ordered = OrderedDict()

    for parameter in params:
        if not group:
            parameter.group = None
        ordered[parameter.__name__] = parameter

    return ordered


def get_benchmark_keys(bench_list):
    return [k for k, v in bench_list]


class AppHostBundle(LastReleasedStableResource):
    name = 'APP_HOST_BUNDLE'
    description = 'app_host'
    resource_type = resource_types.APP_HOST_BUNDLE


class HttpAdapterBundle(LastReleasedStableResource):
    name = 'APP_HOST_HTTP_ADAPTER_BUNDLE_COMMON'
    description = 'http_adapter'
    resource_type = resource_types.APP_HOST_HTTP_ADAPTER_BUNDLE_COMMON


class AppHostTestServants(parameters.ResourceSelector):
    name = 'APP_HOST_TEST_SERVANTS'
    description = 'perftest_servant'
    resource_type = resource_types.APP_HOST_TEST_SERVANTS


class BaseBenchmarkPlan(parameters.ResourceSelector):
    name = 'BASE_APP_HOST_BENCHMARK_PLAN'
    description = 'base dolbilo plan'
    resource_type = resource_types.APP_HOST_BENCHMARK_PLAN
    attrs = {"app": "base"}


class DplannerExecutable(parameters.ResourceSelector):
    name = 'DPLANNER_EXECUTABLE'
    description = 'dplanner'
    resource_type = dolbilka_resources.DPLANNER_EXECUTABLE


PER_APP_SETTINGS = dict()
PER_APP_SETTINGS["app_host"] = ALL_APPHOST + [ALL_BLIST] + [HTTP_ADAPTER_BLIST] + [MIN_BLIST]
PER_APP_SETTINGS["perftest_servant"] = PER_APP_SETTINGS["app_host"]
PER_APP_SETTINGS["http_adapter"] = [HTTP_ADAPTER_BLIST]
PER_APP_SETTINGS["dplanner"] = [HTTP_ADAPTER_BLIST]

APP_TO_RESOURCE_MAPPING = {
    "app_host": AppHostBundle,
    "dplanner": DplannerExecutable,
    "http_adapter": HttpAdapterBundle,
    "perftest_servant": AppHostTestServants
}

SUB_FIELDS = {}
for app, benchmarks in six.iteritems(PER_APP_SETTINGS):
    for benchmark in benchmarks:
        SUB_FIELDS.setdefault(benchmark[0], [])
        SUB_FIELDS[benchmark[0]].append(APP_TO_RESOURCE_MAPPING.get(app).name)
