# -*- coding: utf-8 -*-

import os
import json

import requests
import logging

from sandbox.projects.common import decorators
from sandbox.projects.common import error_handlers as eh

AQUA_URL = "http://aqua.yandex-team.ru/"


class AquaLauncher(object):
    def __init__(self, pack_name):
        self._pack_name = pack_name
        self.session = requests.Session()
        self.session.headers["Content-Type"] = "application/json"
        self._pack_id = self._get_pack_id()

    @staticmethod
    def _get_url(*args):
        return os.path.join(AQUA_URL, "aqua-api", "services", *args)

    @decorators.retries(3, delay=10)
    def _get_pack_id(self):
        all_packs = self.session.get(self._get_url("pack", "all", "simple"))
        all_packs.raise_for_status()
        filtered_by_name = [pack for pack in all_packs.json() if pack['name'] == self._pack_name]
        if not filtered_by_name:
            eh.fail("No packs found by name {}".format(self._pack_name))
        logging.info("%s packs found", len(filtered_by_name))
        return filtered_by_name[0]['id']

    def launch_template(self, pack_params):
        test_info = self._update_info(pack_params)
        r = self.session.put(self._get_url("launch", "pack"), data=json.dumps(test_info))
        r.raise_for_status()
        launch_id = r.json()['id']
        return launch_id

    def _update_info(self, pack_params):
        test_info = self.get_info("pack", self._pack_id)
        for key, value in pack_params.iteritems():
            filtered_props = [prop for prop in test_info["properties"] if prop['key'] == key]
            if filtered_props:
                filtered_props[0]["value"] = value
            else:
                test_info["properties"].append({
                    "key": key,
                    "value": value,
                    "id": None,
                    "title": ""
                })
        return test_info

    @decorators.retries(3, delay=10)
    def get_info(self, item, item_id):
        r = self.session.get(self._get_url(item, str(item_id)))
        r.raise_for_status()
        return r.json()


def get_result_link(launch_id):
    return "{}report/{}/custom-report-plugin/index.html".format(AQUA_URL, launch_id)


def get_allure_result_link(launch_id):
    return "{}report/{}/allure-maven-plugin/index.html".format(AQUA_URL, launch_id)
