# -*- coding: utf-8 -*-
"""
    Module with methods for build arcadia with yatool

    NOTE: Please STOP COMMIT NEW CODE IN THIS MODULE.

    Large libraries are too hard to refactor, but we NEED TO
    extract sdk2/binary-buildable part from Arcadia SDK.

    With any contra-arguments please discuss with mvel@
    and Release Machine team or your code will be melted down.
"""
import copy
import datetime
import errno
import glob
import json
import logging
import multiprocessing
import os
import platform
import random
import signal
import six
import shlex
import shutil
import string
import subprocess
import sys
import tarfile
import tempfile
import time
import re

from six.moves import _thread as thread
from six.moves import urllib_parse as urlparse

from collections import deque
from collections import Counter

import kernel.util.functional

from sandbox import common
import sandbox.common.types.misc as ctm
import sandbox.common.types.client as ctc
import sandbox.common.types.statistics as ctst

from sandbox.sandboxsdk import channel
from sandbox.sandboxsdk import process
from sandbox.sandboxsdk import errors
from sandbox.sandboxsdk import svn
from sandbox.sandboxsdk import paths
import sandbox.sandboxsdk.environments as senv

from sandbox import sdk2
from sandbox.sdk2 import statistics

from sandbox.projects import resource_types
from sandbox.projects.common import context_managers
from sandbox.projects.common import error_handlers as eh
from sandbox.projects.common import utils2
from sandbox.projects.common import file_utils as fu
from sandbox.projects.common.vcs import aapi
from sandbox.projects.common.vcs import arc
from sandbox.projects.common.vcs import util as vcs_util
import sandbox.projects.common.constants as consts

from sandbox.projects.common.arcadia import sdk_tested_parts as stp
from sandbox.projects.common.arcadia import resources


# according to devtools/ya/core/error.py
INFRASTRUCTURE_ERROR_EXIT_CODE = 12
TEST_FAILED_EXIT_CODE = 10
BUILD_ERROR_EXIT_CODES = [10, 11]
ARCADIA_TESTS_DATA = 'arcadia_tests_data'

YA_TIME_TO_KILL = 60
DEFAULT_DIST_PRIORITY = -100000000

# A list of owners allowed to redefine DistBuild priorities -- reserved for precommit checks only!
PRIVILEGED_TASK_TYPES = {
    ("SANDBOX", "BUILD_SANDBOX_TASKS"),
    ("YATOOL", "YT_HEATER"),
    ("YA_YT_CACHE", "YT_HEATER"),
}

tempdir = None
_tmp_build_dir_lock = thread.allocate_lock()


def _is_local_installation():
    return common.config.Registry().common.installation == ctm.Installation.LOCAL


def _get_custom_fetcher():
    return None if _is_local_installation() else str(channel.channel.task.synchrophazotron)


# XXX: pay attention: this function use sdk2
def save_vault(task, vault_key, filename, postprocess=lambda x: x):
    path = str(task.path(filename))
    with open(path, 'w') as f:
        f.write(postprocess(sdk2.task.Vault.data(vault_key)))
    logging.info('Saved auth (%s) to: %s', vault_key, path)
    return path


def wait_any(processes, timeout, timeout_sleep=1, need_terminate_all=True):
    try:
        return stp.wait_any(processes, timeout, timeout_sleep, need_terminate_all,
                            kill_subtree=True, terminate_timeout=40, kill_signal=getattr(signal, 'SIGKILL', None))
    except stp.SubprocessTimeoutError as er:
        raise errors.SandboxSubprocessTimeoutError(er.message)


def add_link_to_report_file(report_file, resource_type, ttl=4, resource_subpath=None, description=None):
    logging.debug(
        'Add link to report %s with type %s, ttl %s, subpath %s, description %s',
        report_file, resource_type, ttl, resource_subpath, description
    )
    is_file_exists = report_file and os.path.exists(report_file) and os.path.isfile(report_file)
    is_directory_not_empty = report_file and os.path.exists(report_file) and os.path.isdir(report_file) and os.listdir(report_file)
    if is_file_exists or is_directory_not_empty:
        file_desc = os.path.basename(report_file)
        resource_desc = "{} ({})".format(description, file_desc) if description else file_desc

        output_html_res = channel.channel.task.create_resource(
            description=resource_desc,
            resource_path=report_file,
            resource_type=resource_type,
            attributes={
                'ttl': ttl,
            }
        )
        info_prefix = "See {}: ".format(description) if description else "See "
        channel.channel.task.mark_resource_ready(output_html_res.id)
        channel.channel.task.set_info(
            info_prefix + utils2.resource_redirect_link(output_html_res.id, file_desc, resource_subpath),
            do_escape=False,
        )
    else:
        logging.debug(
            'Do not create link to report %s with type %s, ttl %s, subpath %s, description %s',
            report_file, resource_type, ttl, resource_subpath, description
        )


def get_unique_resource_filename(name):
    name_and_extension = name.rsplit(".", 1)
    if channel.channel.task.agentr.iteration:
        name_and_extension[0] = name_and_extension[0] + '_' + str(channel.channel.task.agentr.iteration)
    name = '.'.join(name_and_extension)

    return paths.get_unique_file_name(channel.channel.task.abs_path(""), name)


class AutoLinks(object):
    def __init__(self):
        self._files = []

    def add(self, name, resource_type, description=None, ttl=4):
        fname = get_unique_resource_filename(name)
        logging.debug('Add auto link %s', (fname, resource_type, description))
        self._files.append((fname, resource_type, description, ttl))

        return fname

    def finalize(self):
        for name, resource_type, description, ttl in self._files:
            add_link_to_report_file(name, resource_type, description=description, ttl=ttl)


def parse_flags(flags):
    return stp.parse_flags(flags)


def parse_test_params(test_params):
    return stp.parse_test_params(test_params)


def validate_platform_flags(flags):
    return stp.validate_platform_flags(flags)


def update_field(field, key, value, fname=None, sync=False, is_sdk2=False):
    if is_sdk2:
        if getattr(sdk2.Task.current.Context, field) is ctm.NotExists:
            setattr(sdk2.Task.current.Context, field, {})
        getattr(sdk2.Task.current.Context, field)[key] = value
        if sync:
            sdk2.Task.current.Context.save()
            json.dump(
                getattr(sdk2.Task.current.Context, field),
                open(str(sdk2.Task.current.path(fname or field + '.json')), 'wt'),
                indent=2
            )
    else:
        if field not in channel.channel.task.ctx:
            channel.channel.task.ctx[field] = {}
        channel.channel.task.ctx[field][key] = value
        if sync:
            channel.channel.sandbox.set_task_context_value(
                channel.channel.task.id, field, channel.channel.task.ctx[field]
            )
            json.dump(
                channel.channel.task.ctx[field],
                open(channel.channel.task.abs_path(fname or field + '.json'), 'wt'),
                indent=2
            )


def stage_started(name, t=None, sync=False, is_sdk2=False):
    stage(name + '_started', t=t, sync=sync, is_sdk2=is_sdk2)


def stage_finished(name, t=None, sync=False, is_sdk2=False):
    stage(name + '_finished', t=t, sync=sync, is_sdk2=is_sdk2)


def stage(name, t=None, sync=False, is_sdk2=False):
    stages_time = t or time.time()
    update_field('__STAGES', name, stages_time, 'stages.json', sync=sync, is_sdk2=is_sdk2)
    # human_readable_time = time.strftime("%Y-%m-%d %H:%M:%S", time.localtime(stages_time))
    # channel.channel.task.set_info('{}: {}'.format(name, human_readable_time), do_escape=False)


def load_stages(stages_path, is_sdk2=False):
    if os.path.exists(stages_path):
        stages = json.load(open(stages_path))
        for k, v in sorted(six.iteritems(stages), key=lambda x: x[1]):
            stage(k, v, sync=False, is_sdk2=is_sdk2)


def _get_temp_build_dir():
    global tempdir
    if tempdir is None:
        _tmp_build_dir_lock.acquire()
        try:
            if tempdir is None:
                tempdir = tempfile.mkdtemp()
                # Make tempdir accessible for skynet services like sharing, etc.
                os.chmod(tempdir, 0o775)
        finally:
            _tmp_build_dir_lock.release()
    return tempdir


def _get_ya_tool(arcadia_root, frepkage_root=None):
    if frepkage_root:
        return os.path.join(frepkage_root, 'ya-bin')

    return os.path.join(arcadia_root, 'ya')


def _run_ya_handler(
    arcadia_root, use_ya_dev, base_handler,
    log_prefix=None, work_dir=None, stdout=None, stderr=None, timeout=180 * 60, **kwargs
):
    """
        Run ya base_handler using 'ya - < json_data'

        :param arcadia_root: arcadia root (source root)
        :param use_ya_dev: use ya-dev or ya
        :param base_handler: handler name
        :param **kwargs: handler arguments
    """

    with statistics.measure_time('ya_{}'.format(base_handler)):
        ya_process = _create_ya_process(
            arcadia_root, use_ya_dev, base_handler, log_prefix, work_dir, stdout, stderr, **kwargs
        )
        process.check_process_timeout(ya_process, timeout, timeout_sleep=1)
    _check_ya_return_code(ya_process)


def _create_ya_process(
    arcadia_root, use_ya_dev, base_handler,
    log_prefix=None, work_dir=None, stdout=None, stderr=None, **kwargs
):
    cmd = [
        _python(),
        _get_ya_tool(arcadia_root),
    ]

    if kwargs.get("verbose", True):
        cmd.append("-v")

    if kwargs.get("no_report", False):
        cmd.append("--no-report")

    cmd.append("-")

    args = {
        'handler': base_handler,
        'args': kwargs
    }

    do_clear_build = kwargs.get("args", {}).get("clear_build", False) or kwargs.get("clear_build", False)
    parent_build_cache_dir = _get_temp_build_dir() if do_clear_build else _get_parent_build_cache_dir()
    cache_dir = _get_build_cache_dir(parent_build_cache_dir=parent_build_cache_dir)
    tools_cache_dir = os.path.join(parent_build_cache_dir, 'tools')

    env = os.environ.copy()
    env.update({
        'YA_CACHE_DIR': cache_dir,
        'YA_CACHE_DIR_TOOLS': tools_cache_dir,
        'YA_ARCADIA_TESTS_DATA': os.path.join(cache_dir, ARCADIA_TESTS_DATA),
        'YA_USE_SVN_REPOSITORY_FOR_DIST_BUILD': '1',
        'YA_DOWNLOAD_ARTIFACTS': '1',
        'YA_CACHE_NAMESPACE': 'SB',
    })

    if kwargs.get('log_file'):
        env['YA_LOG_FILE'] = kwargs['log_file']

    if kwargs.get('evlog_file'):
        env['YA_EVLOG_FILE'] = kwargs['evlog_file']

    if use_ya_dev:
        env['YA_DEV'] = 'true'

    logging.info("Running 'ya -' with  env %s and stdin:\n%s", env, json.dumps(args))

    ya_process = process.run_process(
        cmd,
        stdin=subprocess.PIPE,
        log_prefix=log_prefix or 'run_ya_' + base_handler,
        wait=False,
        work_dir=work_dir,
        outputs_to_one_file=False,
        stdout=stdout,
        stderr=stderr,
        environment=env,
        time_to_kill=YA_TIME_TO_KILL
    )
    ya_process.stdin.write(json.dumps(args).encode())
    ya_process.stdin.close()
    return ya_process


def _check_ya_return_code(p, check_rc=True, failed_tests_cause_error=True):
    if p.returncode == INFRASTRUCTURE_ERROR_EXIT_CODE:
        raise common.errors.TemporaryError('Ya exited with special exit code ({})'.format(p.returncode))
    if failed_tests_cause_error:
        acceptable_rc = [0]
    else:
        acceptable_rc = [0, TEST_FAILED_EXIT_CODE]
    logging.debug(
        'Process "%s" finished with exit code %s, check_rc = %s, acceptable = %s',
        p.saved_cmd, p.returncode, check_rc, acceptable_rc
    )
    # Always report an issue if ya is killed by signal (returncode < 0).
    # This situation is abnormal and always should be reported, but on Windows - see DEVTOOLS-9470 for details.
    if (check_rc and p.returncode not in acceptable_rc) or (p.returncode < 0 and not common.platform.on_windows()):
        process.check_process_return_code(p)


@kernel.util.functional.memoized
def _need_drop_cache(build_dir):
    return os.path.exists(build_dir) and random.randint(1, 100) == 1


def _is_on_multislot():
    return ctc.Tag.MULTISLOT in common.config.Registry().client.tags


def _get_parent_build_cache_dir():
    if _is_on_multislot():
        logging.debug("On multislot, not using build cache")
        return _get_temp_build_dir()
    else:
        return senv.SandboxEnvironment.build_cache_dir


def _get_build_cache_dir(build_system=consts.YMAKE_BUILD_SYSTEM, parent_build_cache_dir=None):
    if parent_build_cache_dir is None:
        parent_build_cache_dir = _get_parent_build_cache_dir()
    build_dir = os.path.join(
        parent_build_cache_dir,
        # see DEVTOOLSSUPPORT-9011
        'yb' if build_system in (consts.YMAKE_BUILD_SYSTEM, consts.YA_MAKE_FORCE_BUILD_SYSTEM) else 'db'
    )
    if _need_drop_cache(build_dir):
        suffix = ''.join(random.choice(string.ascii_uppercase + string.digits) for _ in range(5))
        tmp_dir = '{}.{}'.format(build_dir, suffix)
        try:
            shutil.move(build_dir, tmp_dir)
            logging.debug("Drop cache. Remove %s", build_dir)
        except Exception as e:
            logging.exception('Unable to move build directory %s because of %s', build_dir, repr(e))

    for tmp_yabuild in glob.glob('{}.*'.format(build_dir)):
        if os.path.exists(tmp_yabuild):
            try:
                def onerror(target_function, path, exc_info):
                    e_type, e_value, e_traceback = exc_info
                    if e_type == OSError and e_value.errno == errno.EACCES:
                        currpath = path
                        while not build_dir.startswith(currpath):
                            common.fs.chmod_for_path(currpath, "a+rwx")
                            currpath = os.path.dirname(currpath)

                        try:
                            target_function(path)
                        except OSError:
                            paths = []
                            while path and (not paths or path != paths[-1]):
                                paths.append(path)
                                path = os.path.dirname(path)
                            proc = process.run_process(
                                ['ls', '-ld'] + paths, check=False, log_prefix='ls-ld.dump.path.permissions'
                            )
                            with open(proc.stdout_path) as afile:
                                logging.error("Problem file stat: %s", afile.read().strip("\n "))
                            raise

                shutil.rmtree(tmp_yabuild, onerror=onerror)
            except Exception as e:
                logging.exception('Unable to clean temp build directory %s because of %s', tmp_yabuild, repr(e))

    return build_dir


def _logging_directory_recursive_stat(path):
    logging.error("Print content of the directory %s:", path)
    for root, dirs, files in os.walk(path):
        for d in dirs:
            dpath = os.path.join(root, d)
            logging.error("directory %s, permissions %s", dpath, bin(os.stat(dpath).st_mode))
        for f in files:
            fpath = os.path.join(root, f)
            logging.error("file %s, permissions %s", fpath, bin(os.stat(fpath).st_mode))


def _clear_ya_make_build_artefacts():
    if _is_on_multislot():
        return
    build_dir = os.path.join(senv.SandboxEnvironment.build_cache_dir, 'yabuild')
    for subdir_name in ('build_root', 'conf', 'tmp_cache'):
        subdir = os.path.join(build_dir, subdir_name)
        if os.path.exists(subdir):
            try:
                shutil.rmtree(subdir, ignore_errors=True)
            except OSError:
                _logging_directory_recursive_stat(subdir)
                raise


def _get_dist_priority():
    try:
        return DEFAULT_DIST_PRIORITY - channel.channel.task.id
    except Exception:
        pass
    return DEFAULT_DIST_PRIORITY


class YaMakeParams(object):
    def __init__(
        self,
        source_root,
        build_dir,
        build_type_args,
        definition_args,
        abs_targets,
        clear_build=False,
        use_ya_dev=False,
        test=None,
        test_params=None,
        output_only_tests=False,
        test_filters=None,
        build_results_report_file=None,
        report_tests_only=False,
        report_config_path=None,
        keep_on=None,
        junit_report_path=None,
        host_platform=None,
        target_platform=None,
        target_platform_flags=None,
        coverage=None,
        coverage_prefix_filter=None,
        coverage_report_path=None,
        sanitize=None,
        lto=False,
        results_dir=None,
        build_system=consts.YMAKE_BUILD_SYSTEM,
        stages_path=None,
        force_build_depends=False,
        ignore_recurses=False,
        patch=None,
        semi_dist_ctx=None,
        test_size_filter=None,
        allure_report_path=None,
        test_threads=0,
        json_prefix=None,
        cache_test_results=False,
        tests_retries=1,
        canonize_tests=False,
        checkout=False,
        pgo_add=False,
        pgo_use=None,
        test_type_filter=None,
        fuzzing=None,
        fuzz_opts=None,
        sanitize_coverage=None,
        cache_namespace=None,
        sandbox_token=None,
        ssh_user=None,
        musl=False,
        resource_owner=None,
        streaming_report_url=None,
        streaming_report_id=None,
        add_result=None,
        sonar=False,
        sonar_options=None,
        sonar_project_filter=None,
        sonar_default_project_filter=False,
        java_coverage=False,
        yndexing_params=None,
        sandbox_uploaded_resource_ttl=None,
        build_threads=None,
        build_execution_time=None,
        cpp_coverage_type=None,
        upload_coverage=False,
        merge_coverage=False,
        save_links_for_files=None,
        javac_options=None,
        sanitizer_flags=None,
        python_coverage=False,
        go_coverage=False,
        nlg_coverage=False,
        dist_priority=None,
        coordinators_filter=None,
        make_context_on_distbuild=False,
        skip_test_console_report=False,
        thinlto=False,
        host_platform_flags=None,  # string 'key=value...' or dict
        frepkage_root=None,
        trace_ya_output=False,
        coverage_unified_agent=False,
        coverage_unified_agent_sid=None,
        coverage_unified_agent_strict=False,
        coverage_unified_agent_uids_file_path=None,
        coverage_unified_agent_failed_uids_file_path=None,
        heater_mode=False,
        retriable_displaced_build=False,
        distbuild_pool=None,
        ya_make_extra_parameters=None,
        stat_dir=None,
        fuzz_minimize=False,
    ):
        self.source_root = source_root
        self.build_dir = build_dir
        self.build_type_args = build_type_args
        self.definition_args = definition_args
        self.abs_targets = abs_targets
        self.clear_build = clear_build
        self.use_ya_dev = use_ya_dev
        self.test = test
        self.test_params = test_params
        self.output_only_tests = output_only_tests
        self.test_filters = test_filters
        self.canonize_tests = canonize_tests
        self.build_results_report_file = build_results_report_file
        self.report_tests_only = report_tests_only
        self.report_config_path = report_config_path
        self.keep_on = keep_on
        self.junit_report_path = junit_report_path
        self.host_platform = host_platform
        self.host_platform_flags = (
            stp.parse_flags(host_platform_flags, d_flags_only=False)
            if isinstance(host_platform_flags, six.string_types)
            else host_platform_flags
        )
        self.target_platform = target_platform
        self.target_platform_flags = target_platform_flags
        self.coverage = coverage
        self.coverage_prefix_filter = coverage_prefix_filter
        self.coverage_report_path = coverage_report_path
        self.sanitize = sanitize
        self.lto = lto
        self.results_dir = results_dir
        self.build_system = build_system
        self.stages_path = stages_path
        self.force_build_depends = force_build_depends
        self.ignore_recurses = ignore_recurses
        self.patch = patch
        self.semi_dist_ctx = semi_dist_ctx
        self.test_size_filter = test_size_filter
        self.allure_report_path = allure_report_path
        self.test_threads = test_threads
        self.json_prefix = json_prefix
        self.cache_test_results = cache_test_results
        self.tests_retries = tests_retries
        self.checkout = checkout
        self.pgo_add = pgo_add
        self.pgo_use = pgo_use
        self.test_type_filter = test_type_filter
        self.fuzzing = fuzzing
        self.fuzz_opts = fuzz_opts
        self.sanitize_coverage = sanitize_coverage
        self.cache_namespace = cache_namespace
        self.sandbox_token = sandbox_token
        self.ssh_user = ssh_user
        self.musl = musl
        self.resource_owner = resource_owner
        self.streaming_report_url = streaming_report_url
        self.streaming_report_id = streaming_report_id
        self.add_result = add_result
        self.sonar = sonar
        self.sonar_options = sonar_options
        self.sonar_project_filter = sonar_project_filter
        self.sonar_default_project_filter = sonar_default_project_filter
        self.java_coverage = java_coverage
        self.yndexing_params = yndexing_params
        self.sandbox_uploaded_resource_ttl = sandbox_uploaded_resource_ttl
        self.build_threads = build_threads
        self.build_execution_time = build_execution_time
        self.cpp_coverage_type = cpp_coverage_type
        self.upload_coverage = upload_coverage
        self.merge_coverage = merge_coverage
        self.save_links_for_files = save_links_for_files
        self.javac_options = javac_options
        self.sanitizer_flags = sanitizer_flags
        self.python_coverage = python_coverage
        self.go_coverage = go_coverage
        self.nlg_coverage = nlg_coverage
        self.coverage_unified_agent = coverage_unified_agent
        self.coverage_unified_agent_sid = coverage_unified_agent_sid
        self.coverage_unified_agent_strict = coverage_unified_agent_strict
        self.coverage_unified_agent_uids_file_path = coverage_unified_agent_uids_file_path
        self.coverage_unified_agent_failed_uids_file_path = coverage_unified_agent_failed_uids_file_path
        self.heater_mode = heater_mode
        try:
            self.dist_priority = int(dist_priority)
        except Exception:
            self.dist_priority = _get_dist_priority()
        self.coordinators_filter = coordinators_filter
        self.make_context_on_distbuild = make_context_on_distbuild
        self.skip_test_console_report = skip_test_console_report
        self.thinlto = thinlto
        self.frepkage_root = frepkage_root
        self.trace_ya_output = trace_ya_output
        self.retriable_displaced_build = retriable_displaced_build
        self.distbuild_pool = distbuild_pool
        self.ya_make_extra_parameters = ya_make_extra_parameters
        self.stat_dir = stat_dir
        self.fuzz_minimize = fuzz_minimize


def _get_ya_make_cmd(params):
    logs_dir = stp.make_unique_dir(channel.channel.task.log_path('{}-logs'.format(params.build_system)))
    ya_tool = _get_ya_tool(params.source_root, params.frepkage_root)
    build_threads = multiprocessing.cpu_count() if params.build_threads is None else params.build_threads
    channel.channel.task.ctx[statistics.TASK_STAT]['ya_build_threads'] = build_threads
    if params.frepkage_root:
        ya_make_cmd = [ya_tool]
    else:
        ya_make_cmd = [_python(), ya_tool]

    if params.trace_ya_output:
        ya_make_cmd += [
            'tool', 'optrace',
            '--follow-forks',
            '--human-readable',
            '--report-size', '2048',
            '--forward-all-signals',
        ] + ya_make_cmd

    ya_make_cmd += [
        'make',
        '-T',
        '-xx',  # temporary solution for fix possible problems with ymake graph update (for example LS_FOR_TESTS)
        '-j', str(build_threads),
        '--build-dir', params.build_dir,
        '--stat',
        '--no-src-links',
        '-G',
    ]
    semi_dist_ctx = params.semi_dist_ctx or {}
    if params.stages_path:
        ya_make_cmd += ['--stages', params.stages_path]
    if params.clear_build:
        ya_make_cmd += ['--rebuild']

    if (
        params.build_system in (consts.DISTBUILD_BUILD_SYSTEM, consts.DISTBUILD_FORCE_BUILD_SYSTEM) or
        (params.build_system == consts.SEMI_DISTBUILD_BUILD_SYSTEM and 'build_dir' not in semi_dist_ctx)
    ):
        dist_priority = params.dist_priority
        for entry in PRIVILEGED_TASK_TYPES:
            if not isinstance(entry, (tuple, list)) or len(entry) != 2:
                continue
            owner, task_type = entry
            if (
                (owner is None or owner == channel.channel.task.owner) and
                (task_type is None or task_type == channel.channel.task.type)
            ):
                break
        else:
            dist_priority = min(dist_priority, _get_dist_priority())

        ya_make_cmd += ['--dist', '--dist-priority', str(dist_priority)]

        head, dirty, revision = _extract_arc_head_and_state(params.source_root, True)
        arc_dist_url = None
        if revision is None and head is not None:
            if dirty:
                eh.check_failed("Can't run 'ya make' on distbuild from dirty repository with commit not from trunk.")
            else:
                arc_dist_url = _arc_hash_to_distbuild_url(head)
                logging.debug("Url '{}' will be used for distributed build with arcc.")
                ya_make_cmd += ['--use-arcc-in-distbuild']
                ya_make_cmd += ['--arc-url-as-working-copy-in-distbuild', arc_dist_url]

        if params.distbuild_pool:
            ya_make_cmd += ['--distbuild-pool', params.distbuild_pool]
        if params.patch:
            ya_make_cmd += ['--apply-patch', params.patch]
        if params.cache_namespace:
            ya_make_cmd += ['--cache-namespace', params.cache_namespace]
        if params.build_execution_time:
            ya_make_cmd += ['--build-time', str(params.build_execution_time)]
        if params.make_context_on_distbuild:
            if arc_dist_url:
                eh.check_failed("Option make_context_on_distbuild can't be used for arc repo with commit not from trunk ('{}').".format(arcadia_url))
            ya_make_cmd += ['--make-context-on-distbuild']
            parsed_url = svn.Arcadia.parse_url(channel.channel.task.ctx.get(consts.ARCADIA_URL_KEY))
            ya_make_cmd += ['--svn-url', _get_svn_url(parsed_url)]
            ya_make_cmd += ['--revision', parsed_url.revision]
        ya_make_cmd += ['--dump-distbuild-result', os.path.join(logs_dir, 'distbuild-result.json')]
        ya_make_cmd += ['--dump-distbuild-graph', os.path.join(logs_dir, 'distbuild-graph.json')]
    if params.force_build_depends:
        ya_make_cmd += ['--force-build-depends']
    if params.ignore_recurses:
        ya_make_cmd += ['--ignore-recurses']

    if params.test:
        ya_make_cmd += ['-t', '--run-all-tests', '--show-passed-tests']
        if params.allure_report_path:
            ya_make_cmd += ['--allure', params.allure_report_path]
        if params.test_params:
            for key, val in six.iteritems(params.test_params):
                ya_make_cmd += ['--test-param', '{}={}'.format(key, val)]
        if params.output_only_tests:
            ya_make_cmd += ['--output-only-tests']

    if params.keep_on:
        ya_make_cmd += ['--keep-going']

    if params.junit_report_path:
        ya_make_cmd += ['--junit', params.junit_report_path]

    if params.test_filters:
        for filt in params.test_filters:
            ya_make_cmd += ["--test-filter", filt]

    if params.canonize_tests:
        ya_make_cmd += ["--canonize-tests"]

    if params.test_size_filter:
        for size in params.test_size_filter:
            ya_make_cmd += ["--test-size", size]

    if params.test_type_filter:
        for fname in params.test_type_filter:
            ya_make_cmd += ['--test-type', fname]

    if params.host_platform:
        ya_make_cmd += ['--host-platform', params.host_platform]

    if params.host_platform_flags:
        for (k, v) in six.iteritems(params.host_platform_flags):
            ya_make_cmd += ['--host-platform-flag', '{}={}'.format(k, v)]

    if params.target_platform:
        ya_make_cmd += ['--target-platform', params.target_platform]

    if params.target_platform_flags:
        ya_make_cmd += shlex.split(params.target_platform_flags)

    if params.coverage:
        ya_make_cmd += ['--coverage']
    if params.java_coverage:
        ya_make_cmd += ['--java-coverage']
    if params.coverage_prefix_filter:
        ya_make_cmd += ['--coverage-prefix-filter', params.coverage_prefix_filter]
    if params.coverage_report_path:
        ya_make_cmd += ['--coverage-report-path', params.coverage_report_path]

    if params.sanitize:
        ya_make_cmd += ['--sanitize', params.sanitize]
    for f in params.sanitizer_flags or []:
        ya_make_cmd += ['--sanitizer-flag', f]
    if params.sanitize_coverage:
        ya_make_cmd += ['--sanitize-coverage', params.sanitize_coverage]

    if params.lto:
        ya_make_cmd += ['--lto']

    if params.thinlto:
        ya_make_cmd += ['--thinlto']

    if params.pgo_add:
        ya_make_cmd += ['--pgo-add']

    if params.pgo_use:
        ya_make_cmd += ['--pgo-use', params.pgo_use]

    if params.build_results_report_file:
        ya_make_cmd += ["--build-results-report", params.build_results_report_file]

    if params.report_tests_only:
        ya_make_cmd += ["--build-results-report-tests-only"]

    if params.report_config_path:
        ya_make_cmd += ["--report-config", params.report_config_path]

    if params.results_dir and not params.heater_mode:
        stp.mkdirp(params.results_dir)
        ya_make_cmd += ['--results-root', params.results_dir]

    if params.test_threads:
        ya_make_cmd += ['--test-threads', str(params.test_threads)]

    if params.json_prefix:
        ya_make_cmd += ['--json-prefix', params.json_prefix]

    if params.cache_test_results:
        ya_make_cmd += ['--cache-tests']

    if params.tests_retries > 1:
        ya_make_cmd += ['--tests-retries', str(params.tests_retries)]

    if params.checkout:
        ya_make_cmd += ['--checkout']

    if params.fuzzing:
        ya_make_cmd += ['--fuzzing']

    if params.fuzz_opts:
        ya_make_cmd += ['--fuzz-opts', ' '.join(params.fuzz_opts)]

    if params.fuzz_minimize:
        ya_make_cmd += ['--fuzz-minimize']

    if params.sandbox_token:
        ya_make_cmd += ['--token', params.sandbox_token]

    if params.ssh_user:
        ya_make_cmd += ['--user', params.ssh_user]

    if params.musl:
        ya_make_cmd += ['--musl']

    if params.resource_owner:
        ya_make_cmd += ['--owner', params.resource_owner]

    if params.streaming_report_url:
        ya_make_cmd += ['--streaming-report-url', params.streaming_report_url]
        ya_make_cmd += ['--streaming-task-id', str(channel.channel.task.id)]

    if params.streaming_report_id:
        ya_make_cmd += ['--streaming-report-id', params.streaming_report_id]

    if params.add_result:
        options = params.add_result
        if not isinstance(options, (list, tuple)):
            options = [options]
        for opt in options:
            ya_make_cmd += ['--add-result', opt]

    if params.sonar:
        ya_make_cmd += ['--sonar']
        for sonar_option in (params.sonar_options or []):
            ya_make_cmd += ['--sonar-property', sonar_option]
        if params.sonar_default_project_filter:
            ya_make_cmd += ['--sonar-default-project-filter']
        elif params.sonar_project_filter:
            ya_make_cmd += ['--sonar-project-filter', params.sonar_project_filter]

    if params.yndexing_params:
        ya_make_cmd += ['--yndexing']
        if params.yndexing_params.get('yt_root'):
            ya_make_cmd += ['--yt-root', params.yndexing_params['yt_root']]
        if params.yndexing_params.get('yt_cluster'):
            ya_make_cmd += ['--yt-cluster', params.yndexing_params['yt_cluster']]
        if params.yndexing_params.get('java_yndexing'):
            ya_make_cmd += ['--java-yndexing']
        if params.yndexing_params.get('py3_yndexing'):
            ya_make_cmd += ['--py3-yndexing']
        if not params.yndexing_params.get('py_yndexing'):
            ya_make_cmd += ['--no-py-yndexing']

    if params.sandbox_uploaded_resource_ttl:
        ya_make_cmd += ['--ttl', str(params.sandbox_uploaded_resource_ttl)]

    if params.cpp_coverage_type == 'clang':
        ya_make_cmd += ['--clang-coverage']

    if params.python_coverage:
        ya_make_cmd += ['--python-coverage']

    if params.go_coverage:
        ya_make_cmd += ['--go-coverage']

    if params.nlg_coverage:
        ya_make_cmd += ['--nlg-coverage']

    if params.upload_coverage:
        ya_make_cmd += ['--upload-coverage']
        if params.coverage_unified_agent:
            if not params.coverage_unified_agent_sid:
                raise AssertionError('Upload coverage with unified agent must be used with sid')
            ya_make_cmd += ['--upload-coverage-with-ua', '--coverage-ua-upload-sid', params.coverage_unified_agent_sid]
            if params.coverage_unified_agent_uids_file_path:
                ya_make_cmd += [
                    '--coverage-save-uploaded-reports-uids-path',
                    params.coverage_unified_agent_uids_file_path,
                ]
            if params.coverage_unified_agent_failed_uids_file_path:
                ya_make_cmd += [
                    '--coverage-save-failed-reports-uids-path',
                    params.coverage_unified_agent_failed_uids_file_path
                ]
            if params.coverage_unified_agent_strict:
                ya_make_cmd += ['--upload-coverage-with-ua-only']

    if params.merge_coverage:
        ya_make_cmd += ['--merge-coverage']

    for filename in params.save_links_for_files or []:
        ya_make_cmd += ['--save-links-for', filename]

    for javac_option in (params.javac_options or []):
        ya_make_cmd += ['--javac-opts', javac_option]

    if params.coordinators_filter:
        ya_make_cmd += ['--coordinators-filter', params.coordinators_filter]

    if params.skip_test_console_report:
        ya_make_cmd += ['--skip-test-console-report']

    if params.frepkage_root:
        ya_make_cmd += ['--custom-context', os.path.join(params.frepkage_root, 'build_context.json')]

    if params.stat_dir:
        ya_make_cmd += ['--stat-dir', params.stat_dir]

    if params.ya_make_extra_parameters:
        for item in params.ya_make_extra_parameters:
            vals = re.split(r"=", item, 1)
            k = vals[0]
            v = None
            if len(vals) == 2:
                v = vals[1]
            ya_make_cmd += [k] if v is None or v == "" else [k, v]

    return list(six.moves.map(
        lambda e: six.ensure_text(six.text_type(e), encoding='utf-8'),
        ya_make_cmd + params.build_type_args + params.definition_args + params.abs_targets
    ))


def _get_ya_dump_cmd(source_root, definition_args, abs_targets, use_ya_dev=False):
    ya_tool = _get_ya_tool(source_root)
    ya_dump_cmd = [
        _python(),
        ya_tool,
        '-v',
        'dump',
        'dep-graph',
    ]

    # TODO: Support build_type_args in ya dump:
    # in ya:
    # r1967089 | nslus | 2015-11-02 17:12:36 +0300 (Mon, 02 Nov 2015) | 2 lines
    return ya_dump_cmd + definition_args + abs_targets


def _dump_graph_for_debug(source_root, definition_args, abs_targets, use_ya_dev):
    try:
        ya_dump_cmd = _get_ya_dump_cmd(source_root, definition_args, abs_targets, use_ya_dev)
        with statistics.measure_time('ya_dump'):
            ya_dump_process = process.run_process(
                ya_dump_cmd,
                log_prefix='ya_dump',
                outputs_to_one_file=False,
                wait=False,
            )
            process.check_process_timeout(ya_dump_process, 600, timeout_sleep=1)  # 10m
        logging.debug('Ya process %s finished with exit code %s', ya_dump_cmd, ya_dump_process.returncode)
        _check_ya_return_code(ya_dump_process)
    except Exception as ex:
        logging.exception("Failed to run ya dump: %s", ex)


class YtStoreParams(object):
    def __init__(
        self,
        store,
        token,
        proxy=None,
        dir=None,
        put=False,
        codec=None,
        replace_result=False,
        replace_result_add_objects=False,
        replace_result_rm_binaries=False,
        max_cache_size=None,
        store_exclusive=False,
        threads=None,
        refresh_on_read=False,
    ):
        self.store = store if token else False
        self.token_path = None
        if token:
            self.token_path = tempfile.NamedTemporaryFile().name
            fu.write_file(self.token_path, token)
        self.proxy = proxy
        self.dir = dir
        self.put = put
        self.codec = codec
        self.replace_result = replace_result
        self.replace_result_add_objects = replace_result_add_objects
        self.replace_result_rm_binaries = replace_result_rm_binaries
        self.max_cache_size = max_cache_size
        self.store_exclusive = store_exclusive
        self.threads = threads
        self.refresh_on_read = refresh_on_read

    def report_status(self, task):
        if self.store:
            task.set_info('YT store successfully enabled')
        else:
            if not self.token_path:
                task.set_info('ERROR: YT store token is missing. Cannot use YT store, your builds will be slow. Read more about how to add YT Token here: https://docs.yandex-team.ru/ya-make/usage/ya_make/yt_store')


def _get_environment_vars(
        source_root,
        build_dir,
        build_resource_id=None,
        test=None,
        test_log_level=None,
        test_tag=None,
        results_dir=None,
        output_html=None,
        log_file=None,
        evlog_file=None,
        local_ymake_path=None,
        disable_test_timeout=False,
        arcadia_tests_data=None,
        coverage_exclude_regexp='',
        jvm_args=None,
        test_failure_code=None,
        no_src_changes=True,
        custom_fetcher=False,
        collect_test_cores=True,
        download_artifacts=True,
        coverage_yt_token_path=None,
        graph_timestamp=None,
        drop_graph_result_before_tests=None,
        checkout=None,
        report_config_path=None,
        strip_skipped_test_deps=False,
        fast_clang_coverage_merge=False,
        new_dist_mode=False,
        yt_store_params=None,
        valid_port_range=None,
        keep_alive_all_streams=False,
        frepkage_root=None,
        frepkage_target_uid=None,
        run_tagged_tests_on_sandbox=False,
        run_tagged_tests_on_yt=False,
        merge_split_tests=True,
        vanilla_execute_yt_token_path=None,
        ram_drive_path=None,
        heater_mode=False,
        retriable_displaced_build=False,
        dir_outputs=False,
        disable_flake8_migrations=False,
        ya_token=None,
):

    env_vars = {
        'YA_CACHE_DIR': build_dir,
        'YA_CACHE_NAMESPACE': 'SB',
        'YA_NEW_RUNNER3': '1',
        'YA_NEW_STORE3': '1',
        'YA_STRIP_SKIPPED_TEST_DEPS': str(int(bool(strip_skipped_test_deps))),
        'YA_USE_SVN_REPOSITORY_FOR_DIST_BUILD': '1',
        'YA_LOCAL_EXECUTOR': '1',
    }

    for name in ['SVN_SSH']:
        if os.environ.get(name) is not None:
            env_vars[name] = os.environ.get(name)

    if download_artifacts:
        env_vars['YA_DOWNLOAD_ARTIFACTS'] = '1'

    if custom_fetcher:
        custom_fetcher_value = _get_custom_fetcher()
        if custom_fetcher_value is not None:
            env_vars['YA_CUSTOM_FETCHER'] = custom_fetcher_value

    if build_resource_id:
        env_vars['YA_SANDBOX_BUILD_OUTPUT_ID'] = str(build_resource_id)

    # don't leave ATD undefined for backward-compatibility
    env_vars['YA_ARCADIA_TESTS_DATA'] = arcadia_tests_data or os.path.join(build_dir, ARCADIA_TESTS_DATA)

    if test:
        # temp hack
        atd_path = os.path.abspath(os.path.join(source_root, '..', ARCADIA_TESTS_DATA))
        if os.path.islink(atd_path):
            os.remove(atd_path)

        if test_log_level:
            env_vars['YA_TEST_LOG_LEVEL'] = test_log_level
        if test_tag:
            env_vars['YA_TEST_TAG'] = test_tag
        if disable_test_timeout:
            env_vars['YA_TEST_DISABLE_TIMEOUT'] = 'yes'

    if results_dir and not heater_mode:
        stp.mkdirp(results_dir)
        env_vars['YA_GENERATE_BIN_DIR'] = os.path.join(results_dir, 'bin')
        env_vars['YA_GENERATE_LIB_DIR'] = os.path.join(results_dir, 'lib')
        env_vars['YA_BUILD_RESULTS_REPORT'] = 'results.json'

    if heater_mode:
        env_vars['YA_YT_WT'] = '0'
        env_vars['YA_EAGER_EXECUTION'] = '1'

    if output_html:
        env_vars['YA_HTML_DISPLAY'] = output_html

    if log_file:
        env_vars['YA_LOG_FILE'] = log_file

    if evlog_file:
        env_vars['YA_EVLOG_FILE'] = evlog_file

    if local_ymake_path:
        env_vars['LOCAL_CONF_PATH'] = local_ymake_path
    else:
        env_vars['DO_NOT_USE_LOCAL_CONF'] = 'yes'

    if coverage_exclude_regexp:
        env_vars['YA_COVERAGE_EXCLUDE_REGEXP'] = coverage_exclude_regexp

    if jvm_args:
        env_vars["YA_JVM_ARGS"] = jvm_args

    if test_failure_code is not None:
        env_vars["YA_TEST_FAILURE_CODE"] = str(test_failure_code)

    if no_src_changes:
        env_vars["YA_NO_SRC_CHANGES"] = 'yes'

    if not collect_test_cores:
        env_vars["YA_TEST_COLLECT_CORES"] = 'no'

    if coverage_yt_token_path:
        env_vars["YA_COVERAGE_YT_TOKEN_PATH"] = coverage_yt_token_path

    if graph_timestamp:
        env_vars["YA_GRAPH_TIMESTAMP"] = str(graph_timestamp)

    if drop_graph_result_before_tests:
        env_vars["YA_DROP_GRAPH_RESULT_BEFORE_TESTS"] = '1'

    if checkout and report_config_path:
        env_vars["YA_CHECKOUT_EXTRA_PATH"] = ":".join([os.path.dirname(report_config_path)])

    if fast_clang_coverage_merge:
        env_vars['YA_FAST_CLANG_COVERAGE_MERGE'] = '1'

    if new_dist_mode:
        env_vars['YA_NEW_DIST'] = '1'

    if valid_port_range:
        env_vars['VALID_PORT_RANGE'] = valid_port_range

    if keep_alive_all_streams:
        env_vars['YA_KEEP_ALIVE_ALL_STREAMS'] = '1'

    if yt_store_params and (yt_store_params.store or yt_store_params.store_exclusive):
        env_vars['YA_YT_STORE4'] = '1'

        assert yt_store_params.token_path, 'Looks like task failed to obtain YT token - no token path was provided'
        env_vars['YA_YT_TOKEN_PATH'] = yt_store_params.token_path
        if yt_store_params.proxy:
            env_vars['YA_YT_PROXY'] = yt_store_params.proxy
        if yt_store_params.dir:
            env_vars['YA_YT_DIR'] = yt_store_params.dir
        if yt_store_params.put:
            env_vars['YA_YT_PUT'] = '1'
        if yt_store_params.codec:
            env_vars['YA_YT_STORE_CODEC'] = yt_store_params.codec
        if yt_store_params.replace_result:
            env_vars['YA_YT_REPLACE_RESULT'] = '1'
        if yt_store_params.replace_result_add_objects:
            env_vars['YA_YT_REPLACE_RESULT_ADD_OBJECTS'] = '1'
        if yt_store_params.replace_result_rm_binaries:
            env_vars['YA_YT_REPLACE_RESULT_RM_BINARIES'] = '1'
        if yt_store_params.max_cache_size:
            env_vars['YA_YT_MAX_STORAGE_SIZE'] = str(yt_store_params.max_cache_size)
        if yt_store_params.threads:
            env_vars['YA_YT_STORE_THREADS'] = str(yt_store_params.threads)
        else:
            env_vars['YA_YT_STORE_THREADS'] = str(max(multiprocessing.cpu_count() // 4, 2))
        if yt_store_params.store_exclusive:
            env_vars['YA_YT_STORE_EXCLUSIVE'] = '1'
        if yt_store_params.refresh_on_read:
            env_vars['YA_YT_STORE_REFRESH_ON_READ'] = '1'

    if frepkage_root:
        env_vars['YA_NO_RESPAWN'] = 'yes'
        env_vars['YA_FREPKAGE_ROOT'] = frepkage_root

    if frepkage_target_uid:
        env_vars['YA_FREPKAGE_TARGET_UID'] = frepkage_target_uid

    if vanilla_execute_yt_token_path:
        env_vars["YA_VANILLA_EXECUTE_YT_TOKEN_PATH"] = vanilla_execute_yt_token_path

    if run_tagged_tests_on_sandbox:
        env_vars['YA_RUN_TAGGED_TESTS_ON_SANDBOX'] = '1'
        # Setup sandbox_run_test node's env
        token_path = tempfile.NamedTemporaryFile().name
        fu.write_file(token_path, common.rest.Client._external_auth.task_token)
        env_vars['YA_SANDBOX_TASK_SESSION_TOKEN_PATH'] = token_path
        env_vars['YA_SANDBOX_TASK_SESSION_PARENT_ID'] = str(channel.channel.task.id)
        env_vars['YA_SANDBOX_TASK_SESSION_OWNER'] = str(channel.channel.task.owner)

    if run_tagged_tests_on_yt:
        env_vars['YA_RUN_TAGGED_TESTS_ON_YT'] = '1'
    else:
        env_vars['YA_RUN_TAGGED_TESTS_ON_YT'] = '0'

    env_vars['YA_MERGE_SPLIT_TESTS'] = '1' if merge_split_tests else '0'
    env_vars['YA_TEST_DISABLE_FLAKE8_MIGRATIONS'] = '1' if disable_flake8_migrations else '0'

    if ram_drive_path:
        env_vars['YA_RAM_DRIVE_PATH'] = str(ram_drive_path)

    env_vars['YA_DIR_OUTPUTS'] = '1' if dir_outputs else '0'

    if retriable_displaced_build:
        env_vars['YA_RETRIABLE_DISPLACED_BUILD'] = '1'

    if ya_token:
        env_vars['YA_TOKEN'] = ya_token

    return env_vars


def _ya_make(
        auto_links,
        source_root,
        targets,
        clear_build,
        build_type='release',
        build_system=consts.YMAKE_BUILD_SYSTEM,
        local_ymake_path=None,
        strip_binaries=False,
        test=False,
        test_filters=None,
        test_params=None,
        output_only_tests=False,
        test_log_level=None,
        test_tag=None,
        def_flags=None,
        results_dir=None,
        keep_on=None,
        check_rc=True,
        host_platform=None,
        target_platform=None,
        target_platform_flags=None,
        timeout=10800,
        use_ya_dev=False,
        junit_report_path=None,
        build_resource_id=None,
        coverage=None,
        coverage_prefix_filter=None,
        coverage_exclude_regexp='',
        coverage_report_path=None,
        sanitize=None,
        lto=False,
        stdout=None,
        stderr=None,
        disable_flake8_migrations=False,
        build_results_report_file=None,
        report_tests_only=False,
        report_config_path=None,
        disable_test_timeout=False,
        force_build_depends=False,
        ignore_recurses=False,
        semi_dist_ctx=None,
        patch=None,
        test_size_filter=None,
        allure_report_path=None,
        test_threads=0,
        json_prefix=None,
        arcadia_tests_data=None,
        cache_test_results=False,
        tests_retries=1,
        checkout=False,
        pgo_add=False,
        pgo_use=None,
        pgo_merge_timeout=600,
        test_type_filter=None,
        fuzzing=None,
        fuzz_opts=None,
        sanitize_coverage=None,
        jvm_args=None,
        test_failure_code=None,
        cache_namespace=None,
        sandbox_token=None,
        ssh_user=None,
        musl=False,
        no_src_changes=True,
        resource_owner=None,
        streaming_report_url=None,
        streaming_report_id=None,
        add_result=None,
        collect_test_cores=True,
        sonar=False,
        sonar_options=None,
        sonar_project_filter=None,
        sonar_default_project_filter=False,
        java_coverage=False,
        yndexing_params=None,
        sandbox_uploaded_resource_ttl=None,
        download_artifacts=True,
        build_execution_time=None,
        coverage_yt_token_path=None,
        cpp_coverage_type=None,
        upload_coverage=False,
        merge_coverage=False,
        graph_timestamp=None,
        drop_graph_result_before_tests=False,
        save_links_for_files=None,
        javac_options=None,
        env_vars=None,
        sanitizer_flags=None,
        strip_skipped_test_deps=False,
        python_coverage=False,
        go_coverage=False,
        nlg_coverage=False,
        dist_priority=None,
        coordinators_filter=None,
        make_context_on_distbuild=False,
        separate_result_dirs=False,
        yt_store_params=None,
        build_threads=None,
        fast_clang_coverage_merge=False,
        build_output_html_ttl=4,
        new_dist_mode=False,
        skip_test_console_report=False,
        thinlto=False,
        keep_alive_all_streams=False,
        build_dir=None,
        canonize_tests=False,
        host_platform_flags=None,
        frepkage_root=None,
        frepkage_target_uid=None,
        run_tagged_tests_on_sandbox=False,
        run_tagged_tests_on_yt=False,
        merge_split_tests=True,
        vanilla_execute_yt_token_path=None,
        trace_ya_output=False,
        ram_drive_path=None,
        coverage_unified_agent=False,
        coverage_unified_agent_sid=None,
        coverage_unified_agent_strict=False,
        coverage_unified_agent_uids_file_path=None,
        coverage_unified_agent_failed_uids_file_path=None,
        failed_tests_cause_error=True,
        dump_graph_for_debug=True,
        heater_mode=False,
        retriable_displaced_build=False,
        dir_outputs=False,
        ya_make_extra_parameters=None,
        distbuild_pool=None,
        fuzz_minimize=False,
):
    logging.debug("_ya_make(%s)", locals())

    def make_absolute_target_path(p):
        return source_root if p == 'ALL' else os.path.join(source_root, p)

    logging.debug('Run ya make, env is %s', os.environ)
    semi_dist_ctx = {} if semi_dist_ctx is None else semi_dist_ctx
    logging.debug("Semi-distbuild context: {}(addr:{})".format(semi_dist_ctx, id(semi_dist_ctx)))
    # XXX
    if build_system == consts.SEMI_DISTBUILD_BUILD_SYSTEM and semi_dist_ctx.get('build_dir'):
        build_dir = semi_dist_ctx.get('build_dir')
    elif build_dir is None:
        build_dir = _get_temp_build_dir() if clear_build else _get_build_cache_dir()
    else:
        logging.info("You are using %s as build cache directory -- do it on your own risk", build_dir)
    logging.debug('gettempdir() = %s', tempfile.gettempdir())
    logging.debug('Setup build dir to %s, clear build param is %s', build_dir, clear_build)
    abs_targets = [make_absolute_target_path(x) for x in targets]

    for target in abs_targets:
        if os.path.islink(target) or not os.path.isdir(target):
            # raise ValueError('Target {} is not a directory'.format(target))
            logging.warning('Target %s is not a directory', target)

    build_type_args = [
        '--build', build_type,
    ]
    definition_args = []
    if def_flags:
        definition_args += ['-D{}={}'.format(k, v) for k, v in six.iteritems(def_flags)]
    # TODO: add --strip flag in ya make and use it here
    if strip_binaries:
        definition_args += ['-DSTRIP=yes']

    if env_vars is None:
        env_vars = {}

    relative_report_config_path = report_config_path
    report_config_path = os.path.join(source_root, relative_report_config_path) if relative_report_config_path else None

    # XXX: remove as soon as all fat tests finish
    if report_config_path is not None and not os.path.exists(report_config_path):
        logging.warn('Report config path %s does not exist, reset it', report_config_path)
        report_config_path = None

    # Local results dir and results dir must be on the same FS for future hardlinking files between them.
    local_results_dir = tempfile.mkdtemp(dir=os.path.abspath(os.path.join(results_dir, os.pardir))) \
        if separate_result_dirs else results_dir
    logging.debug("Local build results dir: %s", local_results_dir)

    env_vars.update(_get_environment_vars(
        source_root,
        build_dir,
        build_resource_id,
        test,
        test_log_level,
        test_tag,
        local_results_dir,
        auto_links.add(
            'output.html',
            resource_types.BUILD_OUTPUT_HTML,
            "Human-readable colored build errors",
            ttl=build_output_html_ttl
        ),
        auto_links.add('log.txt', resource_types.BUILD_LOGS),
        auto_links.add('evlog.json', resource_types.BUILD_LOGS),
        local_ymake_path,
        disable_test_timeout,
        arcadia_tests_data,
        coverage_exclude_regexp,
        jvm_args,
        test_failure_code,
        no_src_changes,
        custom_fetcher=True,
        collect_test_cores=collect_test_cores,
        download_artifacts=download_artifacts,
        coverage_yt_token_path=coverage_yt_token_path,
        graph_timestamp=graph_timestamp,
        drop_graph_result_before_tests=drop_graph_result_before_tests,
        checkout=checkout,
        report_config_path=relative_report_config_path,
        strip_skipped_test_deps=strip_skipped_test_deps,
        fast_clang_coverage_merge=fast_clang_coverage_merge,
        new_dist_mode=new_dist_mode,
        yt_store_params=yt_store_params,
        # https://wiki.yandex-team.ru/sandbox/cookbook/#kakpoluchitdiapazongarantirovannosvobodnyxportov
        valid_port_range="15000:25000",
        keep_alive_all_streams=keep_alive_all_streams,
        frepkage_root=frepkage_root,
        frepkage_target_uid=frepkage_target_uid,
        run_tagged_tests_on_sandbox=run_tagged_tests_on_sandbox,
        run_tagged_tests_on_yt=run_tagged_tests_on_yt,
        merge_split_tests=merge_split_tests,
        vanilla_execute_yt_token_path=vanilla_execute_yt_token_path,
        ram_drive_path=ram_drive_path,
        heater_mode=heater_mode,
        retriable_displaced_build=retriable_displaced_build,
        dir_outputs=dir_outputs,
        disable_flake8_migrations=disable_flake8_migrations,
    ))
    logging.info('Add environment variables: {}'.format(str(env_vars)))
    if pgo_use:
        merged_profile = ''.join(
            random.choice(string.ascii_lowercase + string.digits) for _ in six.moves.range(10)
        ) + '.profile'
        profiles_dir = os.path.abspath('pgo-profiles')
        paths.make_folder(profiles_dir)
        merged_profile = os.path.join(profiles_dir, merged_profile)
        _do_pgo_merge(
            source_root, merged_profile, pgo_use,
            use_ya_dev=use_ya_dev, stderr=stderr, stdout=stdout, timeout=pgo_merge_timeout,
        )
        pgo_use = merged_profile
    if not sonar:
        sonar_options = None
        sonar_project_filter = None
        sonar_default_project_filter = None

    if coverage_unified_agent_uids_file_path:
        stp.mkdirp(os.path.dirname(coverage_unified_agent_uids_file_path))
    if coverage_unified_agent_failed_uids_file_path:
        stp.mkdirp(os.path.dirname(coverage_unified_agent_failed_uids_file_path))

    ya_make_params = YaMakeParams(
        source_root, build_dir, build_type_args, definition_args, abs_targets, clear_build,
        use_ya_dev, test, test_params, output_only_tests, test_filters,
        build_results_report_file, report_tests_only, report_config_path, keep_on, junit_report_path,
        host_platform, target_platform, target_platform_flags, coverage, coverage_prefix_filter, coverage_report_path,
        sanitize, lto, local_results_dir,
        checkout=checkout,
        force_build_depends=force_build_depends,
        ignore_recurses=ignore_recurses,
        patch=patch,
        test_size_filter=test_size_filter,
        test_threads=test_threads,
        json_prefix=json_prefix,
        cache_test_results=cache_test_results,
        tests_retries=tests_retries,
        canonize_tests=canonize_tests,
        pgo_add=pgo_add,
        pgo_use=pgo_use,
        test_type_filter=test_type_filter,
        sanitize_coverage=sanitize_coverage,
        cache_namespace=cache_namespace,
        ssh_user=ssh_user,
        musl=musl,
        resource_owner=resource_owner,
        streaming_report_url=streaming_report_url,
        streaming_report_id=streaming_report_id,
        add_result=add_result,
        sonar=sonar,
        sonar_options=sonar_options,
        sonar_project_filter=sonar_project_filter,
        sonar_default_project_filter=sonar_default_project_filter,
        java_coverage=java_coverage,
        yndexing_params=yndexing_params,
        sandbox_uploaded_resource_ttl=sandbox_uploaded_resource_ttl,
        fuzzing=fuzzing,
        fuzz_opts=fuzz_opts,
        sandbox_token=sandbox_token,
        build_execution_time=build_execution_time,
        cpp_coverage_type=cpp_coverage_type,
        upload_coverage=upload_coverage,
        merge_coverage=merge_coverage,
        save_links_for_files=save_links_for_files,
        javac_options=javac_options,
        sanitizer_flags=sanitizer_flags,
        python_coverage=python_coverage,
        go_coverage=go_coverage,
        nlg_coverage=nlg_coverage,
        dist_priority=dist_priority,
        coordinators_filter=coordinators_filter,
        make_context_on_distbuild=make_context_on_distbuild,
        build_threads=build_threads,
        skip_test_console_report=skip_test_console_report,
        thinlto=thinlto,
        host_platform_flags=host_platform_flags,
        frepkage_root=frepkage_root,
        trace_ya_output=trace_ya_output,
        coverage_unified_agent=coverage_unified_agent,
        coverage_unified_agent_sid=coverage_unified_agent_sid,
        coverage_unified_agent_strict=coverage_unified_agent_strict,
        coverage_unified_agent_uids_file_path=coverage_unified_agent_uids_file_path,
        coverage_unified_agent_failed_uids_file_path=coverage_unified_agent_failed_uids_file_path,
        heater_mode=heater_mode,
        distbuild_pool=distbuild_pool,
        ya_make_extra_parameters=ya_make_extra_parameters,
        stat_dir=auto_links.add(
            'build_statistics',
            resources.BUILD_STATISTICS,
            "build statistics directory dumped with --stat-dir",
            ttl=build_output_html_ttl,
        ),
        fuzz_minimize=fuzz_minimize,
    )

    ya_make_process = None
    ya_make_processes = []
    if checkout:
        os.environ['YA_MULTIPLEX_SSH'] = 'NO'
        env_vars_checkout = {}
        custom_fetcher_value = _get_custom_fetcher()
        if custom_fetcher_value is not None:
            env_vars_checkout['YA_CUSTOM_FETCHER'] = custom_fetcher_value

        checkout_params = copy.deepcopy(ya_make_params)
        checkout_params.build_threads = 0
        checkout_params.test_size_filter = test_size_filter
        checkout_params.checkout = True
        checkout_params.force_build_depends = True
        checkout_params.streaming_report_id = None
        checkout_params.streaming_report_url = None
        checkout_params.keep_on = True
        with process.CustomOsEnviron(env_vars_checkout):
            checkout_process = process.run_process(
                _get_ya_make_cmd(checkout_params),
                log_prefix='ya_make_checkout',
                outputs_to_one_file=False,
                wait=True,
                check=False,
                stdout=stdout,
                stderr=stderr,
            )

        _check_ya_return_code(checkout_process)

        if build_threads == 0:
            return 0

    if build_system not in (consts.DISTBUILD_FORCE_BUILD_SYSTEM, consts.DISTBUILD_BUILD_SYSTEM):
        with process.CustomOsEnviron(env_vars):
            if not frepkage_root and dump_graph_for_debug:
                _dump_graph_for_debug(source_root, definition_args, abs_targets, use_ya_dev)

            ya_make_local_params = copy.deepcopy(ya_make_params)
            ya_make_local_params.stages_path = channel.channel.task.log_path('stages.json')
            ya_make_local_params.allure_report_path = allure_report_path
            ya_make_local_params.test_size_filter = test_size_filter

            ya_make_process = process.run_process(
                _get_ya_make_cmd(ya_make_local_params),
                log_prefix='ya_make',
                outputs_to_one_file=False,
                wait=False,
                stdout=stdout,
                stderr=stderr,
                time_to_kill=YA_TIME_TO_KILL
            )
            ya_make_process.stages_path = ya_make_local_params.stages_path
            ya_make_process.build_dir = build_dir
            ya_make_process.results_dir = local_results_dir
            ya_make_processes += [ya_make_process]

    processes_started_time = time.time()
    is_semi_distbuild_build_phase = (
        build_system == consts.SEMI_DISTBUILD_BUILD_SYSTEM and 'build_dir' not in semi_dist_ctx
    )
    if (
        build_system in (
            consts.DISTBUILD_BUILD_SYSTEM,
            consts.DISTBUILD_FORCE_BUILD_SYSTEM
        )
        or is_semi_distbuild_build_phase
    ):
        dist_build_dir = _get_build_cache_dir(build_system=build_system)
        dist_results_dir = tempfile.mkdtemp(dir=os.path.abspath(os.path.join(results_dir, os.pardir))) if separate_result_dirs else results_dir
        logging.debug("Dist build results dir: %s", dist_results_dir)
        dist_env_vars = _get_environment_vars(
            source_root,
            dist_build_dir,
            build_resource_id,
            test,
            test_log_level,
            test_tag,
            dist_results_dir,
            auto_links.add('dist_output.html', resource_types.BUILD_OUTPUT_HTML, ttl=build_output_html_ttl),
            auto_links.add('dist_log.txt', resource_types.BUILD_LOGS),
            auto_links.add('dist_evlog.json', resource_types.BUILD_LOGS),
            local_ymake_path,
            disable_test_timeout,
            arcadia_tests_data,
            coverage_exclude_regexp,
            jvm_args,
            custom_fetcher=True,
            download_artifacts=download_artifacts,
            coverage_yt_token_path=coverage_yt_token_path,
            graph_timestamp=graph_timestamp,
            drop_graph_result_before_tests=drop_graph_result_before_tests,
            strip_skipped_test_deps=strip_skipped_test_deps,
            fast_clang_coverage_merge=fast_clang_coverage_merge,
            new_dist_mode=new_dist_mode,
            keep_alive_all_streams=keep_alive_all_streams,
            yt_store_params=yt_store_params,
            merge_split_tests=merge_split_tests,
            vanilla_execute_yt_token_path=vanilla_execute_yt_token_path,
            heater_mode=heater_mode,
            retriable_displaced_build=retriable_displaced_build,
            ya_token=(env_vars or {}).get('YA_TOKEN', None)
        )
        with process.CustomOsEnviron(dist_env_vars):
            ya_make_dist_params = copy.deepcopy(ya_make_params)
            ya_make_dist_params.build_dir = dist_build_dir
            ya_make_dist_params.results_dir = dist_results_dir
            ya_make_dist_params.build_system = build_system
            ya_make_dist_params.stages_path = channel.channel.task.log_path('dist_stages.json')
            # fallback 'timeout' not intended exactly for distbuild (it is sandbox YA_MAKE* task timeout, see input param 'ya_timeout')
            # and can contain too large (invalid for distbuild) value, so trim fallback to acceptable level (4 hour)
            ya_make_dist_params.build_execution_time = build_execution_time or (min(14400, timeout) if timeout is not None else None)

            ya_make_dist_process = process.run_process(
                _get_ya_make_cmd(ya_make_dist_params),
                log_prefix='ya_make_dist',
                outputs_to_one_file=False,
                wait=False,
                stdout=stdout,
                stderr=stderr,
                time_to_kill=YA_TIME_TO_KILL
            )
            ya_make_dist_process.stages_path = ya_make_dist_params.stages_path
            ya_make_dist_process.build_dir = dist_build_dir
            ya_make_dist_process.results_dir = dist_results_dir
            ya_make_processes += [ya_make_dist_process]

    assert len(ya_make_processes) > 0

    if len(ya_make_processes) == 1:
        logging.debug("Waiting process for %s(%s) secs", timeout, type(timeout))
    else:
        logging.debug("Waiting for any process (%d) for %s(%s) secs", len(ya_make_processes), timeout, type(timeout))

    finished_process = wait_any(ya_make_processes, timeout, timeout_sleep=1)
    first_process_finished_time = time.time()

    if len(ya_make_processes) > 1:
        logging.debug("First finished process's build_dir: %s", finished_process.build_dir)
        channel.channel.task.ctx['first_finished_build_system'] = (
            consts.YMAKE_BUILD_SYSTEM if finished_process == ya_make_process else build_system
        )
        channel.channel.task.ctx['first_finished_build_system_build_dir'] = finished_process.build_dir

    if separate_result_dirs:
        logging.debug("Hardlinking results from %s to %s", finished_process.results_dir, results_dir)
        _hardlink_tree(finished_process.results_dir, results_dir)

    _check_ya_return_code(finished_process, check_rc, failed_tests_cause_error)

    if build_system == consts.SEMI_DISTBUILD_BUILD_SYSTEM:
        semi_dist_ctx['build_dir'] = finished_process.build_dir

    load_stages(finished_process.stages_path)

    _clear_ya_make_build_artefacts()
    return finished_process.returncode


def do_create_zipatch(source_dir, out_path, timeout=300):
    ya_tool = _get_ya_tool(source_dir)
    ya_zp_cmd = [
        _python(),
        ya_tool,
        'check',
        '--create-zipatch',
        out_path
    ]

    ya_pr_process = process.run_process(
        ya_zp_cmd,
        work_dir=source_dir,
        log_prefix='ya_check_zp',
        outputs_to_one_file=False,
        wait=False,
        time_to_kill=YA_TIME_TO_KILL,
    )

    finished_process = wait_any([ya_pr_process], timeout, timeout_sleep=1)
    if os.path.exists(out_path):
        logging.info("zipatch has been successfully created")
    else:
        logging.error("failed to create zipatch")

    return finished_process.returncode


def post_statistics(snowden_key, start_time, end_time, build_system, report=None):
    delta_time = end_time - start_time
    report_msg = {
        'start_time': str(datetime.datetime.fromtimestamp(start_time)),
        'end_time': str(datetime.datetime.fromtimestamp(end_time)),
        'delta_seconds': str(delta_time),
        'delta': str(datetime.timedelta(seconds=delta_time)),
        'build_system': build_system,
    }
    if report:
        report_msg.update(report)

    statistics.update_stat_field(snowden_key, statistics.make_updater(delta_time), statistics.make_default_stat())


def log_disk_space_usage(msg):
    if platform.system() == "Windows":
        logging.warning("There is no df in windows, can't log free space")
        return True

    out = "unknown"
    try:
        with statistics.measure_time('arcadiasdk_df'):
            out = subprocess.check_output(
                ['df', '-h'],
                stderr=subprocess.STDOUT
            )
    except subprocess.CalledProcessError as e:
        logging.error("Failed to get disk usage space (df rc:{}): {}".format(e.returncode, e.output))

    logging.debug("Disk space usage %s:\n%s", msg, out)


@kernel.util.functional.memoized
def _is_fresh_arcadia_from_trunk(source_root):
    try:
        arcadia_url = None
        revision_date = None
        try:
            if os.path.exists(source_root):
                if not os.path.exists(os.path.join(source_root, '.svn')):
                    # so don't even try
                    logging.warning('No .svn folder in source root %s', source_root)
                    return None, None
                info = svn.Arcadia.info(source_root)
                logging.debug('svn info for %s: %s', source_root, info)
                arcadia_url = info['url']
                if 'date' in info:
                    revision_date = info['date']
        except svn.SvnError as ex:
            logging.exception("Failed to get svn info from %s: %s", source_root, ex)

        if revision_date is None:
            is_fresh_revision = None
        else:
            revision_date = datetime.datetime.strptime(revision_date, '%Y-%m-%dT%H:%M:%S.%fZ')
            is_fresh_revision = datetime.datetime.now() - revision_date < datetime.timedelta(days=1)

        if arcadia_url is None:
            is_arcadia_from_trunk = None
        else:
            is_arcadia_from_trunk = 'arc/trunk/arcadia' in arcadia_url

        return is_fresh_revision, is_arcadia_from_trunk
    except Exception as e:
        logging.exception("_is_fresh_arcadia_from_trunk failed: %s", e)

    return None, None


def _is_fresh_arcadia(source_root):
    return _is_fresh_arcadia_from_trunk(source_root)[0]


def _is_arcadia_from_trunk(source_root):
    return _is_fresh_arcadia_from_trunk(source_root)[1]


def _check_path_is_arc_repo(path):
    code = subprocess.call([arc.Arc().binary_path, "root"], cwd=path)
    return not code


def _arc_hash_to_distbuild_url(hash):
    return "arc://arcadia#{}".format(hash)


def _extract_arc_head_and_state(source_root, try_get_revision=False):
    """
    Constructs a link passed to the distbuild from the current state of the Arc repository.
    """
    logging.debug(
        "Trying to extract head and state from repository '{}' (try_get_revision={})".format(source_root, try_get_revision)
    )
    if not _check_path_is_arc_repo(source_root):
        logging.error("'{}' is not an arc repository, so it's impossible to extract the necessary hash.".format(source_root))
        return None, None, None

    described = arc.Arc().describe(source_root, always=True, dirty=" dirty", exclude="*")[0].split()
    if len(described) > 1 and described[1] != "dirty":
        logging.error("'arc describe' return unexpected result ('{}'). Please contact with Arc team and share your logs.".format(described))
        return None, None, None

    revision = None
    if try_get_revision:
        show_data = arc.Arc().show(source_root, commit=described[0], as_dict=True)
        try:
            revision = show_data[0]["commits"][0]["revision"]
        except Exception as e:
            logging.error("Can't load information about arc HEAD from json: {}".format(e))
    head, dirty = described[0], len(described) == 2 and described[1] == "dirty"
    logging.debug("Extracted state from arc repository: head='{}', dirty='{}', revision='{}'".format(head, dirty, revision))
    return head, dirty, revision


def _check_arc_compatible_build_system(build_system, source_root):
    """ Checks that arc doesn't use distbuild for commit not from trunk
    """
    logging.info("Checking compatibility of build system and repository type")
    if _check_path_is_arc_repo(source_root):
        if build_system in (
            consts.DISTBUILD_BUILD_SYSTEM,
            consts.DISTBUILD_FORCE_BUILD_SYSTEM,
            consts.SEMI_DISTBUILD_BUILD_SYSTEM
        ):
            logging.info("Getting information about HEAD revision state of Arc repo")
            head, dirty, revision = _extract_arc_head_and_state(source_root, True)

            if head is None:
                eh.check_failed("Can't load information about state of Arc repo")

            if dirty and revision is None:
                if build_system in (consts.SEMI_DISTBUILD_BUILD_SYSTEM, consts.YMAKE_BUILD_SYSTEM):
                    logging.info(
                        "Build type was changed from {} to {} to avoid using distbuild "
                        "for dirty Arc repo with commit not from trunk".format(
                            consts.SEMI_DISTBUILD_BUILD_SYSTEM,
                            consts.YA_MAKE_FORCE_BUILD_SYSTEM
                        )
                    )
                    return consts.YA_MAKE_FORCE_BUILD_SYSTEM
                elif build_system in (consts.DISTBUILD_BUILD_SYSTEM, consts.DISTBUILD_FORCE_BUILD_SYSTEM):
                    eh.check_failed("Arc repository can't use distbuild for dirty repo with commit not from trunk")

    return build_system


def _change_build_system(build_system, source_root, patch):
    """
    Experiment: change YMAKE_BUILD_SYSTEM to SEMI_DISTBUILD_BUILD_SYSTEM for
        builds without patch from fresh trunk arcadia
    """
    try:
        logging.debug(
            'call _change_build_system with build_system=%s, source_root=%s, patch=%s',
            build_system, source_root, patch
        )
        if build_system != consts.YMAKE_BUILD_SYSTEM or patch:
            return build_system

        if _is_arcadia_from_trunk(source_root) and _is_fresh_arcadia(source_root):
            return consts.SEMI_DISTBUILD_BUILD_SYSTEM

    except Exception as e:
        logging.exception("Unable to change build_system: %s", e)

    return build_system


def do_build(
    build_system,
    source_root,
    targets,
    build_type=consts.RELEASE_BUILD_TYPE,
    test=False,
    def_flags=None,
    results_dir=None,
    target_platform=None,
    force_build_system=False,
    **kwargs
):
    """
    Build targets with specified build system. Accepts most options and flags known to ya make.
    All keyword arguments except `clear_build` have sane defaults and are only needed for build customization

    :param build_system: build system (distbuild/ya make/semi-distbuild),
        may be one of `projects.common.constants.*_BUILD_SYSTEM`
    :param source_root: path to arcadia root on filesystem
    :param targets: targets to build, the keyword ALL means the arcadia root itself
    :param build_type: build type (debug/release etc.),
        may be one of `projects.common.constants.*_BUILD_TYPE
    :param clear_build: build from scratch ignoring local build cache (-xx flag of ya make).
        Despite its absence in definition, MUST be set explicitly
    :param results_dir: where to put build results
    :param target_platform: platform to build for.
        See: https://wiki.yandex-team.ru/yatool/target-platform

    :param sandbox_token: OAuth token for interaction with Sandbox API
    :param ssh_user: authenticate as `ssh_user` if no token is passed
    :param build_dir: custom directory for build cache
    :param add_result: list of file suffixes to add to result directory
    :param checkout: update dependencies before build and checkout missing ones
    :param force_build_depends: build by DEPENDS anyway
    :param force_vcs_info_update: force VCS info update
    :param ignore_recurses: do not build by RECURSES
    :param keep_on: build as much as possible
    :param build_execution_time: impose timeout on build process
    :param build_threads: multi-threaded build (-j; defaults to ncpu)
    :param timeout: maximum time for build and tests execution

    :param test: run tests
    :param test_filters: only run tests that match this string.
        See: https://wiki.yandex-team.ru/yatool/test/#filter
    :param test_size_filter: only run tests of these sizes.
        Example: ["SMALL", "MEDIUM", "LARGE"]
    :param test_type_filter: only run tests of these types.
        Example: ["unittest", "jstest", "pytest"]
    :param test_failure_code: exit with custom code when tests fail
    :param test_params: a dictionary of parameters to pass to test via commandline
    :param test_log_level: one of  ("critical", "error", "warning", "info", "debug") values
    :param test_tag: only run tests with this tag
    :param disable_test_timeout: run tests without timeout guard
    :param test_threads: restriction on concurrent tests (no limit by default)
    :param cache_test_results: cache results between test runs
    :param tests_retries: execute tests multiple times
    :param canonize_tests: canonize tests output

    :param def_flags:
        build system definition flags ({'key': 'value'} ->
        -Dkey=value), or string '-Dkey1=val1 -Dkey2=val2'
    :param coverage: collect coverage information
    :param sanitize: sanitizer type (address, memory, thread, undefined, leak)
    :param lto: build with LTO
    :param pgo: create PGO profile
    :param strip_binaries: strip debug information off built binaries
    :param musl: build with musl-libc
    :param sonar: analyze code with sonar
    :param python_coverage: collect python coverage information
    :param go_coverage: collect go coverage information
    :param nlg_coverage: collect nlg coverage information
    :param java_coverage: collect Java coverage information
    :param javac_options: set common javac flags
    :param jvm_args: JVM launch options

    :param fuzzing: extend tests' corpus
    :param fuzz_opts: space-separated string of fuzzing options
    :param heater_mode: if true do not put results to bin/lib directories
    :param retriable_displaced_build: "Build is displaced by other" is retriable error
    :param use_prebuilt_tools: use prebuilt tools (None, 'default', 'no', yes')
    """

    logging.debug("do_build(%s)", locals())
    if not targets:
        raise Exception("Nothing to build: targets are empty")
    semi_dist_ctx = {}

    semi_dist_flags = copy.deepcopy(def_flags)
    if kwargs.get('force_vcs_info_update', False):
        if def_flags is None or isinstance(def_flags, six.string_types):
            def_flags = (def_flags or '') + ' -DFORCE_VCS_INFO_UPDATE=yes'
        else:
            def_flags['FORCE_VCS_INFO_UPDATE'] = 'yes'

    def call_do_build(test, disable_streaming_report, flags=def_flags, force_build_depends=False):
        params = copy.deepcopy(kwargs)
        if disable_streaming_report:
            params.pop(consts.STREAMING_REPORT_ID, None)
            params.pop(consts.STREAMING_REPORT_URL, None)
        if force_build_depends:
            params['force_build_depends'] = True
        return _do_build(
            changed_build_system, source_root, targets, build_type, test, flags, results_dir, target_platform,
            is_build_system_changed, semi_dist_ctx=semi_dist_ctx, **params
        )

    log_disk_space_usage("before building")
    kwargs['patch'] = kwargs.pop('patch', None) or channel.channel.task.ctx.get(consts.ARCADIA_PATCH_KEY)

    changed_build_system = build_system
    if not force_build_system and build_system != consts.SEMI_DISTBUILD_BUILD_SYSTEM:
        # this really works for SVN checkouts only
        changed_build_system = _change_build_system(build_system, source_root, kwargs['patch'])
    # and this one is for Arc
    changed_build_system = _check_arc_compatible_build_system(build_system, source_root)
    is_build_system_changed = changed_build_system != build_system
    if is_build_system_changed:
        logging.info('build_system changed from %s to %s', build_system, changed_build_system)

    if changed_build_system == consts.SEMI_DISTBUILD_BUILD_SYSTEM:
        # test_filters parameter should be used only with 'test' parameter
        if 'test_filters' in kwargs:
            test_filters = kwargs.pop('test_filters')
        else:
            test_filters = None

        if test:
            with statistics.measure_time('arcadiasdk_only_build_time'):
                call_do_build(test=False, disable_streaming_report=True, force_build_depends=True, flags=semi_dist_flags)
            if test_filters is not None:
                kwargs['test_filters'] = test_filters
            log_disk_space_usage("before testing")
            with statistics.measure_time('arcadiasdk_test_time'):
                returncode = call_do_build(test=True, disable_streaming_report=False)
        else:
            with statistics.measure_time('arcadiasdk_only_build_time'):
                returncode = call_do_build(test=False, disable_streaming_report=False)

    else:
        with statistics.measure_time('arcadiasdk_test_time' if test else 'arcadiasdk_only_build_time'):
            returncode = call_do_build(test, disable_streaming_report=False)
    log_disk_space_usage("after testing")
    return returncode


def _do_build(
    build_system, source_root, targets, build_type, test, def_flags, results_dir,
    target_platform, is_build_system_changed,
    **kwargs
):
    logging.debug("_do_build(%s)", locals())
    channel.channel.task.ctx['do_build_runs_count'] = 1 + channel.channel.task.ctx.get('do_build_runs_count', 0)
    do_build_runs_count = channel.channel.task.ctx.get('do_build_runs_count')
    keep_on = kwargs.pop(consts.KEEP_ON, None)
    check_rc = kwargs.pop('check_rc', True)
    host_platform = kwargs.pop('host_platform', None)
    host_platform_flags = kwargs.pop('host_platform_flags', None)
    target_platform_flags = kwargs.pop('target_platform_flags', None)
    validate_platform_flags(target_platform_flags)

    clear_build = kwargs.pop('clear_build', None)
    if clear_build is None:
        raise ValueError("clear_build should be defined")

    build_execution_time = kwargs.pop(consts.BUILD_EXECUTION_TIME, None)
    build_resource_id = kwargs.pop('build_resource_id', None)
    junit_report_path = kwargs.pop('junit_report_path', None)
    coverage = kwargs.pop('coverage', None)
    coverage_prefix_filter = kwargs.pop('coverage_prefix_filter', None)
    coverage_exclude_regexp = kwargs.pop('coverage_exclude_regexp', '')
    sanitize = kwargs.pop('sanitize', None)
    lto = kwargs.pop(consts.LTO, False)
    pgo_add = kwargs.pop(consts.PGO_ADD, False)
    pgo_use = kwargs.pop(consts.PGO_USE, None)
    test_filters = kwargs.pop('test_filters', None)
    canonize_tests = kwargs.pop(consts.CANONIZE_TESTS, False)
    test_size_filter = kwargs.pop('test_size_filter', None)
    test_type_filter = kwargs.pop('test_type_filter', None)
    test_params = kwargs.pop('test_params', {})
    output_only_tests = kwargs.pop(consts.OUTPUT_ONLY_TESTS, False)
    test_log_level = kwargs.pop('test_log_level', None)
    test_tag = kwargs.pop(consts.TEST_TAG, None)
    allure_report = kwargs.pop(consts.ALLURE_REPORT, None)
    allure_report_ttl = kwargs.pop(consts.ALLURE_REPORT_TTL, 4)
    disable_test_timeout = kwargs.pop(consts.DISABLE_TEST_TIMEOUT, False)
    strip_binaries = kwargs.pop('strip_binaries', False)
    timeout = kwargs.pop('timeout', 10800)
    ram_drive_path = kwargs.pop('ram_drive_path', None)
    force_build_depends = kwargs.pop('force_build_depends', False)
    ignore_recurses = kwargs.pop('ignore_recurses', False)
    semi_dist_ctx = kwargs.pop('semi_dist_ctx', {})
    test_threads = kwargs.pop('test_threads', 0)
    json_prefix = kwargs.pop('json_prefix', None)
    arcadia_tests_data = kwargs.pop('arcadia_tests_data', None)
    cache_test_results = kwargs.pop(consts.CACHE_TEST_RESULTS, False)
    tests_retries = kwargs.pop(consts.TESTS_RETRIES, False)
    checkout = kwargs.get('checkout', False)
    pgo_merge_timeout = kwargs.pop('pgo_merge_timeout', 600)
    fuzzing = kwargs.pop(consts.FUZZING, False)
    fuzz_opts = kwargs.pop(consts.FUZZ_OPTS, [])
    sanitize_coverage = kwargs.pop(consts.SANITIZE_COVERAGE, None)
    sanitizer_flags = kwargs.pop(consts.SANITIZER_FLAGS, None)
    jvm_args = kwargs.pop(consts.JVM_ARGS_KEY, None)
    test_failure_code = kwargs.pop(consts.TEST_FAILURE_CODE_KEY, None)
    cache_namespace = kwargs.pop('cache_namespace', None)
    sandbox_token = kwargs.pop('sandbox_token', None)
    ssh_user = kwargs.pop('ssh_user', None)
    musl = kwargs.pop('musl', False)
    no_src_changes = kwargs.pop(consts.NO_SRC_CHANGES, True)
    resource_owner = kwargs.pop(consts.RESOURCE_OWNER, False)
    streaming_report_url = kwargs.pop(consts.STREAMING_REPORT_URL, None)
    streaming_report_id = kwargs.pop(consts.STREAMING_REPORT_ID, None)
    add_result = kwargs.pop('add_result', None)
    collect_test_cores = kwargs.pop('collect_test_cores', True)
    sonar = kwargs.pop('sonar', False)
    sonar_options = kwargs.pop('sonar_options', None)
    sonar_project_filter = kwargs.pop('sonar_project_filter', None)
    sonar_default_project_filter = kwargs.pop('sonar_default_project_filter', None)
    java_coverage = kwargs.pop('java_coverage', False)
    sandbox_uploaded_resource_ttl = kwargs.pop(consts.SANDBOX_UPLOADED_RESOURCE_TTL, None)
    coverage_yt_token_path = kwargs.pop('coverage_yt_token_path', None)
    cpp_coverage_type = kwargs.pop('cpp_coverage_type', None)
    upload_coverage = kwargs.pop('upload_coverage', False)
    merge_coverage = kwargs.pop('merge_coverage', False)
    graph_timestamp = kwargs.pop('graph_timestamp', None)
    drop_graph_result_before_tests = kwargs.pop('drop_graph_result_before_tests', False)
    save_links_for_files = kwargs.pop('save_links_for_files', None)
    javac_options = kwargs.pop(consts.JAVAC_OPTIONS, None)
    strip_skipped_test_deps = kwargs.pop('strip_skipped_test_deps', False)
    python_coverage = kwargs.pop('python_coverage', False)
    go_coverage = kwargs.pop('go_coverage', False)
    nlg_coverage = kwargs.pop('nlg_coverage', False)
    dist_priority = kwargs.pop('dist_priority', None)
    coordinators_filter = kwargs.pop('coordinators_filter', None)
    make_context_on_distbuild = kwargs.pop('make_context_on_distbuild', False)
    separate_result_dirs = kwargs.pop('separate_result_dirs', False)
    yt_store_params = kwargs.pop('yt_store_params', None)
    build_threads = kwargs.pop('build_threads', None)
    fast_clang_coverage_merge = kwargs.pop('fast_clang_coverage_merge', False)
    build_output_html_ttl = kwargs.pop('build_output_html_ttl', 4)
    new_dist_mode = kwargs.pop('new_dist_mode', False)
    skip_test_console_report = kwargs.pop('skip_test_console_report', False)
    thinlto = kwargs.pop(consts.THINLTO, False)
    keep_alive_all_streams = kwargs.pop(consts.KEEP_ALIVE_ALL_STREAMS, False)
    build_dir = kwargs.pop("build_dir", None)
    frepkage_root = kwargs.pop("frepkage_root", None)
    frepkage_target_uid = kwargs.pop("frepkage_target_uid", None)
    run_tagged_tests_on_sandbox = kwargs.pop("run_tagged_tests_on_sandbox", False)
    run_tagged_tests_on_yt = kwargs.pop("run_tagged_tests_on_yt", False)
    merge_split_tests = kwargs.pop("merge_split_tests", True)
    vanilla_execute_yt_token_path = kwargs.pop('vanilla_execute_yt_token_path', None)
    trace_ya_output = kwargs.pop('trace_ya_output', False)
    coverage_unified_agent = kwargs.pop('coverage_unified_agent', False)
    coverage_unified_agent_sid = kwargs.pop('coverage_unified_agent_sid', '')
    coverage_unified_agent_strict = kwargs.pop('coverage_unified_agent_strict', False)
    coverage_unified_agent_uids_res_id = kwargs.pop('coverage_unified_agent_uids_res_id', None)
    coverage_unified_agent_uids_file_path = kwargs.pop('coverage_unified_agent_uids_file_path', None)
    coverage_unified_agent_failed_uids_file_path = kwargs.pop('coverage_unified_agent_failed_uids_file_path', None)
    failed_tests_cause_error = kwargs.pop('failed_tests_cause_error', True)
    auto_links = kwargs.pop('auto_links', AutoLinks())
    dump_graph_for_debug = kwargs.pop('dump_graph_for_debug', True)
    heater_mode = kwargs.pop('heater_mode', False)
    retriable_displaced_build = kwargs.pop('retriable_displaced_build', False)
    dir_outputs = kwargs.pop('dir_outputs', False)
    use_prebuilt_tools = kwargs.pop('use_prebuilt_tools', None)
    distbuild_pool = kwargs.pop('distbuild_pool', None)
    fuzz_minimize = kwargs.pop('fuzz_minimize', [])

    if results_dir and coverage_unified_agent_uids_file_path:
        coverage_unified_agent_uids_file_path = os.path.join(results_dir, coverage_unified_agent_uids_file_path)
    if results_dir and coverage_unified_agent_failed_uids_file_path:
        coverage_unified_agent_failed_uids_file_path = os.path.join(
            results_dir, coverage_unified_agent_failed_uids_file_path
        )

    if test_filters:
        test_filters = test_filters.split()
    if coverage_prefix_filter and not coverage:
        raise ValueError('"coverage_prefix_filter" parameter should be used with "coverage" parameter.')
    if sonar_options:
        sonar_options = sonar_options.split(';')
    if javac_options:
        javac_options = javac_options.split(';')
    if (
        test
        and run_tagged_tests_on_sandbox
        and build_system in [consts.DISTBUILD_BUILD_SYSTEM, consts.DISTBUILD_FORCE_BUILD_SYSTEM]
    ):
        raise AssertionError("run_tagged_tests_on_sandbox mode doesn't support distbuild run")

    use_dev_version = kwargs.pop('use_dev_version', False)
    disable_flake8_migrations = kwargs.pop('disable_flake8_migrations', False)
    build_results_report_file = kwargs.pop('build_results_report_file', None)
    report_tests_only = kwargs.pop('report_tests_only', False)
    report_config_path = channel.channel.task.ctx.get(consts.REPORT_CONFIG_PATH, None)

    local_ymake_path = check_local_ymake(source_root)

    patch = kwargs.pop('patch', None)

    download_artifacts = kwargs.pop('download_artifacts', True)

    yndexing_params = kwargs.pop('yndexing_params', {})

    stdout = kwargs.pop('stdout', None)
    stderr = kwargs.pop('stderr', None)
    report_config = {}
    if report_config_path:
        report_config_full_path = (
            report_config_path if os.path.isabs(report_config_path)
            else os.path.join(source_root, report_config_path)
        )
        if os.path.exists(report_config_full_path):
            with open(report_config_full_path) as fp:
                report_config = json.load(fp)
        else:
            logging.error('report_config_path does not exist %s', report_config_full_path)

    merged_def_flags = stp.merge_def_flags(report_config, 'autocheck_build_vars', def_flags)
    host_platform_flags = stp.merge_def_flags(
        report_config, 'host_platform_flags',
        stp.parse_flags(host_platform_flags, d_flags_only=False)
    )

    if use_prebuilt_tools and use_prebuilt_tools != 'default':
        for flags in (merged_def_flags, host_platform_flags):
            flags['USE_PREBUILT_TOOLS'] = use_prebuilt_tools

    if isinstance(test_params, six.string_types):
        test_params = stp.parse_test_params(test_params)
    if ram_drive_path:
        test_params["ram_drive_path"] = ram_drive_path

    if not target_platform:
        target_platform = None

    if target_platform_flags and target_platform:
        raise ValueError('"target_platform_flags" parameter should be used without "target_platform" parameter.')

    ya_make_extra_parameters = kwargs.pop(consts.YA_MAKE_EXTRA_PARAMETERS, None)

    logging.debug('Unused options: %s', kwargs)

    start_build_time = time.time()

    allure_report_path = None
    if allure_report:
        allure_report_path = channel.channel.task.abs_path(
            'allure_report{}'.format('' if do_build_runs_count == 1 else do_build_runs_count)
        )
    coverage_report_path = None
    if coverage and test:
        coverage_report_path = channel.channel.task.abs_path(
            'coverage.report{}'.format('' if do_build_runs_count == 1 else do_build_runs_count)
        )

    stat_data = {
        'test': test,
        'is_build_system_changed': is_build_system_changed,
        'without_patch': not patch,
        'is_fresh_arcadia': _is_fresh_arcadia(source_root),
        'is_arcadia_from_trunk': _is_arcadia_from_trunk(source_root),
    }

    if build_system not in (
        consts.YMAKE_BUILD_SYSTEM,
        consts.YA_MAKE_FORCE_BUILD_SYSTEM,
        consts.DISTBUILD_BUILD_SYSTEM,
        consts.DISTBUILD_FORCE_BUILD_SYSTEM,
        consts.SEMI_DISTBUILD_BUILD_SYSTEM
    ):
        raise NotImplementedError('Build system "{}" is not supported'.format(build_system))

    if distbuild_pool and not kwargs.get("env", {}).get('YA_TOKEN') and build_system in (
        consts.YMAKE_BUILD_SYSTEM,
        consts.DISTBUILD_BUILD_SYSTEM,
        consts.DISTBUILD_FORCE_BUILD_SYSTEM,
        consts.SEMI_DISTBUILD_BUILD_SYSTEM
    ):
        raise ValueError('distbuild pool can only be used with YA_TOKEN environment variable')

    try:
        returncode = _ya_make(
            auto_links, source_root, targets, clear_build, build_type, build_system, local_ymake_path, strip_binaries,
            test, test_filters, test_params, output_only_tests, test_log_level, test_tag,
            merged_def_flags, results_dir, keep_on, check_rc, host_platform, target_platform, target_platform_flags,
            timeout, use_dev_version, junit_report_path, build_resource_id, coverage, coverage_prefix_filter,
            coverage_exclude_regexp, coverage_report_path, sanitize, lto, stdout, stderr, disable_flake8_migrations,
            build_results_report_file, report_tests_only, report_config_path, disable_test_timeout, force_build_depends,
            ignore_recurses, semi_dist_ctx, patch, test_size_filter, allure_report_path, test_threads, json_prefix,
            arcadia_tests_data, cache_test_results, tests_retries, checkout, pgo_add, pgo_use, pgo_merge_timeout,
            test_type_filter, fuzzing, fuzz_opts, sanitize_coverage, jvm_args, test_failure_code, cache_namespace,
            sandbox_token, ssh_user, musl, no_src_changes, resource_owner,
            streaming_report_url=streaming_report_url,
            streaming_report_id=streaming_report_id,
            add_result=add_result, collect_test_cores=collect_test_cores,
            sonar=sonar, sonar_options=sonar_options,
            sonar_project_filter=sonar_project_filter, sonar_default_project_filter=sonar_default_project_filter,
            java_coverage=java_coverage, yndexing_params=yndexing_params,
            sandbox_uploaded_resource_ttl=sandbox_uploaded_resource_ttl,
            download_artifacts=download_artifacts,
            build_execution_time=build_execution_time,
            coverage_yt_token_path=coverage_yt_token_path,
            cpp_coverage_type=cpp_coverage_type,
            upload_coverage=upload_coverage,
            coverage_unified_agent=coverage_unified_agent,
            coverage_unified_agent_sid=coverage_unified_agent_sid,
            coverage_unified_agent_strict=coverage_unified_agent_strict,
            coverage_unified_agent_uids_file_path=coverage_unified_agent_uids_file_path,
            coverage_unified_agent_failed_uids_file_path=coverage_unified_agent_failed_uids_file_path,
            graph_timestamp=graph_timestamp,
            drop_graph_result_before_tests=drop_graph_result_before_tests,
            save_links_for_files=save_links_for_files,
            javac_options=javac_options,
            sanitizer_flags=sanitizer_flags,
            strip_skipped_test_deps=strip_skipped_test_deps,
            python_coverage=python_coverage,
            go_coverage=go_coverage,
            nlg_coverage=nlg_coverage,
            merge_coverage=merge_coverage,
            dist_priority=dist_priority,
            coordinators_filter=coordinators_filter,
            make_context_on_distbuild=make_context_on_distbuild,
            separate_result_dirs=separate_result_dirs,
            yt_store_params=yt_store_params,
            env_vars=kwargs.get("env"),
            build_threads=build_threads,
            fast_clang_coverage_merge=fast_clang_coverage_merge,
            build_output_html_ttl=build_output_html_ttl,
            new_dist_mode=new_dist_mode,
            skip_test_console_report=skip_test_console_report,
            thinlto=thinlto,
            keep_alive_all_streams=keep_alive_all_streams,
            build_dir=build_dir,
            canonize_tests=canonize_tests,
            host_platform_flags=host_platform_flags,
            frepkage_root=frepkage_root,
            frepkage_target_uid=frepkage_target_uid,
            run_tagged_tests_on_sandbox=run_tagged_tests_on_sandbox,
            run_tagged_tests_on_yt=run_tagged_tests_on_yt,
            merge_split_tests=merge_split_tests,
            vanilla_execute_yt_token_path=vanilla_execute_yt_token_path,
            trace_ya_output=trace_ya_output,
            ram_drive_path=ram_drive_path,
            failed_tests_cause_error=failed_tests_cause_error,
            dump_graph_for_debug=dump_graph_for_debug,
            heater_mode=heater_mode,
            retriable_displaced_build=retriable_displaced_build,
            dir_outputs=dir_outputs,
            ya_make_extra_parameters=ya_make_extra_parameters,
            distbuild_pool=distbuild_pool,
            fuzz_minimize=fuzz_minimize,
        )
    except common.errors.TemporaryError:
        stat_data['status'] = 'TemporaryError'
        post_statistics('arcadiasdk_build_test_time', start_build_time, time.time(), build_system, stat_data)
        raise
    except errors.SandboxSubprocessError:
        add_link_to_report_file(allure_report_path, resource_types.ALLURE_REPORT, allure_report_ttl, 'index.html')
        add_link_to_report_file(coverage_report_path, resource_types.COVERAGE_REPORT, resource_subpath='index.html')
        stat_data['status'] = 'SandboxSubprocessError'
        post_statistics('arcadiasdk_build_test_time', start_build_time, time.time(), build_system, stat_data)
        raise
    finally:
        auto_links.finalize()
    add_link_to_report_file(allure_report_path, resource_types.ALLURE_REPORT, allure_report_ttl, 'index.html')
    add_link_to_report_file(coverage_report_path, resource_types.COVERAGE_REPORT, resource_subpath='index.html')
    stat_data['status'] = 'success'
    post_statistics('arcadiasdk_build_test_time', start_build_time, time.time(), build_system, stat_data)
    return returncode


def dump_targets_deps(source_root, targets, def_flags=None, plain_mode=False):
    ya_tool = _get_ya_tool(source_root)
    get_deps_cmd_base = [
        _python(),
        ya_tool,
        '-v',
        'dump',
        'dir-graph',
    ]
    if plain_mode:
        get_deps_cmd_base += ['--plain']
    else:
        get_deps_cmd_base += ['--dump-deps']

    common_args = []
    if not _is_local_installation():
        common_args += ['-DSANDBOX_FETCHER={}'.format(channel.channel.task.synchrophazotron)]
    if def_flags:
        common_args += ['-D{}={}'.format(k, v) for k, v in six.iteritems(def_flags)]
    get_deps_cmd_base += common_args

    processes = []
    outs = []
    for target in targets:
        abs_target = os.path.join(source_root, target)
        deps_for_target = 'deps_for_{}'.format(os.path.basename(target))
        get_deps_cmd = get_deps_cmd_base + [abs_target]
        with statistics.measure_time('ya_dump_deps'):
            pr = process.run_process(get_deps_cmd, log_prefix=deps_for_target, outputs_to_one_file=False)
            outs.append(pr.stdout_path)
        processes.append(pr)

    for pr in processes:
        process.check_process_timeout(pr, 600, timeout_sleep=1)  # 10m
        _check_ya_return_code(pr)

    result = set()
    for file_path in outs:
        with open(file_path) as of:
            result |= set(json.load(of))
    return sorted(list(result))


def check_local_ymake(source_root):
    try:
        local_ymake_path = os.path.join(source_root, 'local.ymake')
        if os.path.exists(local_ymake_path):
            info = svn.Arcadia.info(local_ymake_path)
            exported_path = os.path.join(tempfile.mkdtemp(), 'local.ymake')
            svn.Arcadia.export(info['url'], exported_path, info['entry_revision'])
            return exported_path
    except svn.SvnError as ex:
        logging.exception("Failed to export local.ymake: %s", ex)
    return None


def do_package(
        source_root,
        packages,
        use_ya_dev=False,
        publish=False,
        sloppy_deb=False,
        artifactory=False,
        artifactory_password="",
        publish_to=None,
        key=None,
        changelog=None,
        build_threads=None,
        run_tests=False,
        run_long_tests=False,
        ignore_fail_tests=False,
        sandbox_task_id=None,
        debian=False,
        build_type='release',
        strip_binaries=False,
        full_strip_binaries=False,
        clear_build=False,
        be_verbose=False,
        arch_all=False,
        force_dupload=False,
        dupload_max_attempts=1,
        build_debian_scripts=False,
        convert=False,
        host_platform=None,
        target_platform=None,
        sanitize=None,
        checkout=False,
        html_display=None,
        custom_version=None,
        create_dbg=False,
        compress_archive=True,
        dump_build_targets_to=None,
        build_system=consts.YMAKE_BUILD_SYSTEM,
        distbuild_pool=None,
        build_only=False,
        debian_compression_level=None,
        debian_store_package=True,
        musl=False,
        debian_compression_type=None,
        lto=False,
        timeout=180 * 60,
        semi_clear_build=False,
        raw_package=False,
        raw_package_path=None,
        yt_store_params=None,
        thinlto=False,
        force_build_depends=False,
        force_vcs_info_update=False,
        ignore_recurses=False,
        build_docker=False,
        docker_image_repository=None,
        docker_save_image=False,
        docker_push_image=False,
        docker_registry=None,
        docker_build_network=None,
        docker_build_arg=None,
        run_medium_tests=False,
        debian_distribution=None,
        rpm=False,
        wheel=False,
        wheel_access_key_path=None,
        wheel_secret_key_path=None,
        overwrite_read_only_files=False,
        dump_inputs_to=None,
        ensure_package_published=False,
        aar=False,
        use_prebuilt_tools=None,
        wheel_python3=False,
        npm=False,
        publish_logs=True,
        create_build_output_resource=False,
        check_build_system=True,
        debian_arch=None,
        cache_tests=False,
        compression_filter=None,
        uc_codec=None,
        compression_level=None,
        test_threads=0,
        test_params=None,
        test_filters=None,
        test_size_filter=None,
        test_tag=None,
        test_type_filter=None,
        test_log_level=None,
        tests_retries=0,
        build_logs_ttl=4,
        build_output_ttl=4,
        patch_pkg_json_version=False,
        patch_pkg_json_version_files=None,
        direct_arcadia_mode=False,
        no_report=False,
):
    """
        Create packages using ya package

        :param source_root: source root where arcadia sits
        :param packages: packages to create, relative to arcadia
        :param use_ya_dev: use ya-dev or ya (deprecated!)
        :param publish: publish resulting packages or not
        :param sloppy_deb: if not publishing, build debian packages faster
        :param artifactory: publish to artifactory
        :param artifactory_password: artifactory user password
        :param publish_to: debian repository to publish to
        :param key: key to sign a debian package
        :param changelog: changelog message
        :param build_threads: number of threads to build targets in packages
        :param run_tests: run tests or not
        :param sandbox_task_id: use certain sandbox task id in package version
        :param debian: create debian package or tarball
        :param build_type: build type(e.g. release, debug...)
        :param strip_binaries: strip compiled binaries (by default only debug info, i.e. strip -g)
        :param full_strip_binaries: strip compiled binaries (strip all)
        :param clear_build: rebuild
        :param be_verbose: verbose
        :param arch_all: "Architecture: all" in debian
        :param force_dupload: dupload --force
        :param dupload_max_attempts: How many times try to run dupload if it fails
        :param build_debian_scripts: build debian scripts directory
        :param convert: convert old json formal on the fly
        :param host_platform: host platform to build on
        :param target_platform: target platform to build for
        :param sanitize: build with specific sanitizer
        :param checkout: run ya make with --checkout
        :param html_display: path to html output
        :param custom_version: custom package version
        :param create_dbg: create separate package with debug info
        :param compress_archive: gzip result tar archive
        :param dump_build_targets_to
        :param build_only: build package only
        :param debian_compression_level: debian compression level None or value from 0-9
        :param debian_store_package: store debian package
        :param musl: build with MUSL libs
        :param lto: build with LTO
        :param thinlto: build with ThinLTO (-flto=thin flag)
        :param force_build_depends: build tests' depends on distbuild if testing is requested
        :param force_vcs_info_update: force VCS info update
        :param ignore_recurses: don't follow RECURSEs (only works without testing)
        :param timeout: ya package command timeout
        :param semi_clear_build: run ya package with clear_build = False, but in a fresh build directory
        :param raw_package: used with "tar" package type to get package content without tarring
        :param yt_store_params: YT store parameters
        :param debian_distribution: Debian distribution name
        :param dump_inputs_to
        :param ensure_package_published: fail task if publishing was requested but failed
        :param check_build_system: check build system
        :param debian_arch: debian arch
        :param cache_tests: enable stable test's uid calculation and caching test results
        :param compression_filter: compression filter for tar (gzip/zstd)
        :param compression_level: compression level for filter
        :param uc_codec Codec name for uc compression
        :param test_threads Restriction on concurrent tests
        :param test_params Arbitrary parameters to be passed to tests (name=val)
        :param test_filters Run only tests that match <tests-filter> or chunks that match "[.*?] chunk"
        :param test_size_filter Run only specified set of tests
        :param test_tag Run tests that have specified tag
        :param test_type_filter Run only specified types of tests
        :param test_log_level Specifies logging level for output test logs
        :param tests_retries Run every test specified number of times
        :param build_logs_ttl Custom TTL for BUILD_LOGS resources
        :param build_output_ttl Custom TTL for BUILD_OUTPUT resources
        :param patch_pkg_json_version Set 'meta'->'version' in each file from <packages> to <custom_version>
        :param patch_pkg_json_version_files Set 'meta'->'version' only in given files when <patch_pkg_json_version> is True and <custom_version> is set
    """
    if distbuild_pool and not os.environ.get('YA_TOKEN') and not os.environ.get('YA_TOKEN_PATH') and build_system in (
        consts.YMAKE_BUILD_SYSTEM,
        consts.DISTBUILD_BUILD_SYSTEM,
        consts.DISTBUILD_FORCE_BUILD_SYSTEM,
        consts.SEMI_DISTBUILD_BUILD_SYSTEM
    ):
        raise ValueError('distbuild pool can only be used with YA_TOKEN or YA_TOKEN_PATH environment variable')

    if direct_arcadia_mode:
        arcadia_root = source_root
    else:
        arcadia_root = os.path.join(source_root, 'arcadia')

    patch = channel.channel.task.ctx.get(consts.ARCADIA_PATCH_KEY)
    changed_build_system = build_system
    if check_build_system:
        changed_build_system = _change_build_system(build_system, arcadia_root, patch)
        changed_build_system = _check_arc_compatible_build_system(build_system, source_root)
    if changed_build_system != build_system:
        logging.info('build_system changed from %s to %s', build_system, changed_build_system)
    changed_args = locals()
    # TODO: uncomment
    # changed_args['build_system'] = changed_build_system

    auto_links = AutoLinks()
    logs_ttl = channel.channel.task.ctx.get(consts.BUILD_LOGS_TTL, build_logs_ttl)

    def add_link_func(filename):
        return auto_links.add(filename, resource_types.BUILD_LOGS, ttl=logs_ttl)

    changed_args['evlog_file'] = add_link_func('package_evlog.json')
    changed_args['log_file'] = add_link_func('package_log.txt')
    changed_args['html_display'] = add_link_func('package_output.html')

    if create_build_output_resource:
        output_dir = get_unique_resource_filename('output_dir')
        output_resource = channel.channel.task.create_resource(
            description='Build output',
            resource_path=output_dir,
            resource_type=resource_types.BUILD_OUTPUT,
            attributes={
                'ttl': channel.channel.task.ctx.get(consts.BUILD_OUTPUT_TTL, build_output_ttl),
            }
        )
        changed_args['output_root'] = output_dir
        changed_args['build_results_resource_id'] = str(output_resource.id)
    else:
        output_resource = None

    if patch_pkg_json_version_files is None:
        patch_pkg_json_version_files = packages

    if patch_pkg_json_version and custom_version is not None and custom_version != '':
        for pkg_file in patch_pkg_json_version_files:
            _patch_pkg_json_version(package=os.path.join(arcadia_root, pkg_file),
                                    version=custom_version)

    try:
        args = _get_package_args(add_link_func=add_link_func, **changed_args)
        args["verbose"] = False
        logging.debug("do_package(%s)", args)
        start_package_time = time.time()
        _run_ya_handler(arcadia_root, use_ya_dev, 'package', timeout=timeout, **args)
        post_statistics(
            'arcadiasdk_package_time', start_package_time, time.time(), build_system,
            {'status': 'success', 'run_tests': run_tests}
        )
    finally:
        if publish_logs:
            auto_links.finalize()
        if output_resource:
            channel.channel.task.mark_resource_ready(output_resource.id)


def _patch_pkg_json_version(package, version):
    with open(package, 'r') as in_f:
        package_info = json.load(in_f)

    package_info['meta']['version'] = version

    with open(package, 'w') as out_f:
        json.dump(package_info, out_f, indent=4)
        out_f.write('\n')


def _get_package_args(**kwargs):
    source_root = kwargs["source_root"]
    publish = kwargs["publish"]
    sloppy_deb = kwargs["sloppy_deb"]
    artifactory = kwargs["artifactory"]
    artifactory_password = kwargs["artifactory_password"]
    password_path = None
    if artifactory_password:
        password_path = os.path.abspath("password_path")
        fu.write_file(password_path, artifactory_password.data()[artifactory_password.default_key])
    publish_to = kwargs["publish_to"]
    key = kwargs["key"]
    changelog = kwargs["changelog"]
    build_threads = kwargs.pop("build_threads")
    run_tests = kwargs["run_tests"]
    run_long_tests = kwargs["run_long_tests"]
    run_medium_tests = kwargs["run_medium_tests"]
    if run_medium_tests:
        run_tests = 2
    if run_long_tests:
        run_tests = 3
    ignore_fail_tests = kwargs["ignore_fail_tests"]
    sandbox_task_id = kwargs["sandbox_task_id"]
    debian = kwargs["debian"]
    docker = kwargs["build_docker"]
    rpm = kwargs["rpm"]
    aar = kwargs["aar"]
    wheel = kwargs["wheel"]
    npm = kwargs["npm"]
    build_type = kwargs["build_type"]
    strip_binaries = kwargs["strip_binaries"]
    full_strip_binaries = kwargs["full_strip_binaries"]
    clear_build = kwargs["clear_build"]
    be_verbose = kwargs["be_verbose"]
    arch_all = kwargs["arch_all"]
    force_dupload = kwargs["force_dupload"]
    dupload_max_attempts = kwargs["dupload_max_attempts"]
    build_debian_scripts = kwargs["build_debian_scripts"]
    convert = kwargs["convert"]
    host_platform = kwargs["host_platform"]
    target_platform = kwargs["target_platform"]
    sanitize = kwargs["sanitize"]
    checkout = kwargs["checkout"]
    html_display = kwargs["html_display"]
    custom_version = kwargs["custom_version"]
    create_dbg = kwargs["create_dbg"]
    compress_archive = kwargs["compress_archive"]
    dump_build_targets_to = kwargs["dump_build_targets_to"]
    build_system = kwargs["build_system"]
    build_only = kwargs["build_only"]
    packages = kwargs["packages"]
    debian_compression_level = kwargs["debian_compression_level"]
    debian_arch = kwargs.get("debian_arch", None)
    debian_store_package = kwargs["debian_store_package"]
    musl = kwargs["musl"]
    lto = kwargs.get("lto", False)
    thinlto = kwargs.get("thinlto", False)
    debian_compression_type = kwargs["debian_compression_type"]
    semi_clear_build = kwargs.get("semi_clear_build", False)
    force_build_depends = kwargs.get("force_build_depends", False)
    ignore_recurses = kwargs.get("ignore_recurses", False)
    raw_package = kwargs.get("raw_package", False)
    raw_package_path = kwargs.get("raw_package_path", None)
    yt_store_params = kwargs.pop("yt_store_params", None)
    overwrite_read_only_files = kwargs["overwrite_read_only_files"]
    dump_inputs_to = kwargs["dump_inputs_to"]
    ensure_package_published = kwargs['ensure_package_published']
    custom_fetcher_value = _get_custom_fetcher()
    use_prebuilt_tools = kwargs.get('use_prebuilt_tools', 'default')
    wheel_python3 = kwargs.get("wheel_python3", False)
    output_root = kwargs.get("output_root", None)
    build_results_resource_id = kwargs.get("build_results_resource_id", None)
    cache_tests = kwargs.get("cache_tests", False)
    compression_filter = kwargs.get("compression_filter", None)
    compression_level = kwargs.get("compression_level", None)
    uc_codec = kwargs.get("uc_codec", None)
    test_threads = kwargs.get("test_threads", 0)
    test_params = kwargs.get("test_params", None)
    test_filters = kwargs.get("test_filters", None)
    test_size_filter = kwargs.get("test_size_filter", None)
    test_tag = kwargs.get("test_tag", None)
    test_type_filter = kwargs.get("test_type_filter", None)
    test_log_level = kwargs.get("test_log_level", None)
    tests_retries = kwargs.get("tests_retries", 0)
    add_link_func = kwargs.pop('add_link_func')
    args = {
        'arch_all': arch_all,
        'artifactory': artifactory,
        'be_verbose': be_verbose,
        'build_debian_scripts': build_debian_scripts,
        'build_only': build_only,
        'build_type': build_type,
        'cache_tests': cache_tests,
        'checkout': checkout,
        'compress_archive': compress_archive,
        'create_dbg': create_dbg,
        'custom_source_root': source_root,
        'custom_version': custom_version,
        'debian_arch': debian_arch,
        'debian_compression_level': debian_compression_level,
        'debian_compression_type': debian_compression_type,
        'dump_graph': 'json',
        'dupload_max_attempts': dupload_max_attempts,
        'force_build_depends': force_build_depends,
        'force_dupload': force_dupload,
        'full_strip': full_strip_binaries,
        'html_display': html_display,
        'ignore_recurses': ignore_recurses,
        'lto': lto,
        'musl': musl,
        'packages': packages,
        'run_tests': run_tests,
        'sandbox_download_protocols': ['http', 'skynet', 'http_tgz'],
        'sanitize': sanitize,
        'sign': bool(key),
        'store_debian': debian_store_package,
        'strip': strip_binaries and not full_strip_binaries,
        'thinlto': thinlto,
    }

    if password_path is not None:
        args['artifactory_password_path'] = password_path

    if build_threads is not None:
        args['build_threads'] = build_threads

    if custom_fetcher_value is not None:
        args['custom_fetcher'] = custom_fetcher_value

    if kwargs.get("debian_distribution"):
        args['debian_distribution'] = kwargs["debian_distribution"]

    if yt_store_params and (yt_store_params.store or yt_store_params.store_exclusive):
        yt_store_args = {}
        yt_store_args["yt_store"] = True
        yt_store_args["yt_token"] = fu.read_file(yt_store_params.token_path)
        if yt_store_params.proxy:
            yt_store_args["yt_proxy"] = yt_store_params.proxy
        if yt_store_params.dir:
            yt_store_args["yt_dir"] = yt_store_params.dir
        if yt_store_params.put:
            yt_store_args["yt_readonly"] = False
        if yt_store_params.codec:
            yt_store_args["yt_store_codec"] = yt_store_params.codec
        if yt_store_params.threads:
            yt_store_args["yt_store_threads"] = yt_store_params.threads
        else:
            yt_store_args["yt_store_threads"] = max(multiprocessing.cpu_count() // 4, 2)
        if yt_store_params.store_exclusive:
            yt_store_args["yt_store_exclusive"] = True
        if yt_store_params.refresh_on_read:
            yt_store_args["yt_store_refresh_on_read"] = True

        kwargs.update(yt_store_args)
        args.update(yt_store_args)

    if use_prebuilt_tools and use_prebuilt_tools != 'default':
        def _set_flags(args, attr, key, value):
            flags = args.get(attr, {})
            flags[key] = value
            args[attr] = flags

        for flags in ('flags', 'host_platform_flags'):
            _set_flags(args, flags, 'USE_PREBUILT_TOOLS', use_prebuilt_tools)

    if 'log_file' in kwargs:
        args['log_file'] = kwargs.get('log_file')

    if 'evlog_file' in kwargs:
        args['evlog_file'] = kwargs.get('evlog_file')

    if not clear_build:
        if build_system == consts.SEMI_DISTBUILD_BUILD_SYSTEM:
            with statistics.measure_time('do_build_package'):
                args['custom_build_directory'] = _do_build_package(add_link_func=add_link_func, **kwargs)
        else:
            args['custom_build_directory'] = _get_build_cache_dir(build_system=build_system)
    else:
        if semi_clear_build:
            # use clean build directory for all packages, but ya package doesn't set clear build = True
            clear_build = False
            args['custom_build_directory'] = _get_temp_build_dir()

    force_vcs_info_update = kwargs.pop("force_vcs_info_update", False)
    if force_vcs_info_update:
        flags = args.get('flags', {})
        flags['FORCE_VCS_INFO_UPDATE'] = 'yes'
        args['flags'] = flags

    args['clear_build'] = clear_build

    if publish and publish_to:
        args['publish_to'] = publish_to
    if key:
        args['key'] = key

    if host_platform:
        args['host_platform'] = host_platform

    if target_platform:
        args['target_platforms'] = [{
            'platform_name': target_platform,
            'run_tests': None,
            'build_type': None,
            'flags': {}
        }]

    else:
        args['sloppy_deb'] = sloppy_deb

    if changelog:
        args['change_log'] = changelog

    if sandbox_task_id:
        args['sandbox_task_id'] = sandbox_task_id

    if debian and docker:
        raise Exception("Invalid packaging format: both debian and docker are set to True")

    if debian:
        args['format'] = 'debian'
    elif docker:
        args['format'] = 'docker'
        args['docker_registry'] = kwargs.get('docker_registry')
        args['docker_repository'] = kwargs.get('docker_image_repository')
        args['docker_push_image'] = kwargs.get('docker_push_image')
        args['docker_save_image'] = kwargs.get('docker_save_image')
        args['docker_build_network'] = kwargs.get('docker_build_network')
        args['docker_build_arg'] = kwargs.get('docker_build_arg') or {}
    elif rpm:
        args['format'] = 'rpm'
    elif aar:
        args['format'] = 'aar'
    elif wheel:
        args['format'] = 'wheel'
        args['wheel_access_key_path'] = kwargs.get("wheel_access_key_path")
        args['wheel_secret_key_path'] = kwargs.get("wheel_secret_key_path")
    elif npm:
        args['format'] = 'npm'
    else:
        args['format'] = 'tar'
        if compression_filter:
            args['compression_filter'] = compression_filter
        if compression_level:
            args['compression_level'] = compression_level
        if uc_codec:
            # In ya package parameter is called 'codec'
            args['codec'] = uc_codec

    if run_tests:
        args['run_long_tests'] = run_long_tests
        args['ignore_fail_tests'] = ignore_fail_tests

        if test_threads:
            args['test_threads'] = test_threads

        if test_params:
            args['test_params'] = test_params

        if test_filters:
            test_filters = [x.strip() for x in test_filters.split(";")]
            args['tests_filters'] = test_filters

        if test_size_filter:
            args['test_size_filter'] = test_size_filter

        if test_tag:
            args['test_tag'] = test_tag

        if test_type_filter:
            args['test_type_filter'] = test_type_filter

        if test_log_level:
            args['test_log_level'] = test_log_level

        if tests_retries:
            args['tests_retries'] = tests_retries

    if build_results_resource_id:
        args['output_root'] = output_root
        args['build_results_resource_id'] = build_results_resource_id

    if build_system in (consts.DISTBUILD_BUILD_SYSTEM, consts.DISTBUILD_FORCE_BUILD_SYSTEM):
        args['use_distbuild'] = True
        args["dist_priority"] = _get_dist_priority()

    args['convert'] = convert

    if dump_build_targets_to:
        args["dump_build_targets"] = dump_build_targets_to

    if dump_inputs_to:
        args["dump_inputs"] = dump_inputs_to

    if raw_package:
        args["raw_package"] = True

    if raw_package_path:
        args["raw_package_path"] = raw_package_path

    if overwrite_read_only_files:
        args["overwrite_read_only_files"] = True

    if ensure_package_published:
        args['ensure_package_published'] = True

    if wheel_python3:
        args['wheel_python3'] = True

    if kwargs.get('distbuild_pool'):
        args['distbuild_pool'] = kwargs['distbuild_pool']
        args['use_new_distbuild_client'] = True  # enable distbuild client with authentication to use distbuild pool

    if kwargs.get('no_report', False):
        args['no_report'] = True

    return args


def _do_build_package(**kwargs):
    if kwargs.get('add_link_func'):
        add_link_func = kwargs.pop('add_link_func')
    else:
        add_link_func = lambda x: None

    custom_fetcher_value = _get_custom_fetcher()
    arcadia_root = kwargs.pop("arcadia_root")
    use_ya_dev = kwargs.pop("use_ya_dev")
    if kwargs["run_tests"] != 0:
        kwargs["force_build_depends"] = True
    kwargs["build_only"] = True
    kwargs["run_tests"] = 0
    kwargs["run_long_tests"] = False
    kwargs["custom_source_root"] = kwargs.pop("source_root")
    kwargs["dump_graph"] = 'json'

    if kwargs.get("checkout"):
        checkout_build_args = copy.deepcopy(kwargs)
        checkout_build_args["use_distbuild"] = False
        checkout_build_args["build_threads"] = 0  # don't build, just checkout
        checkout_build_args["keep_on"] = True
        checkout_build_process = _create_ya_process(
            arcadia_root, use_ya_dev, "package", "checkout_package", work_dir=None, stdout=None, **checkout_build_args
        )
        checkout_build_process.wait()
        kwargs["checkout"] = False  # other processes should not perform checkout

    local_build_args = copy.deepcopy(kwargs)
    local_build_args['verbose'] = False
    local_build_args['log_file'] = add_link_func('build_package_locally.log')
    local_build_args['html_display'] = add_link_func('build_package_locally.html')
    local_build_args["use_distbuild"] = False
    local_build_args["custom_build_directory"] = _get_build_cache_dir()
    if custom_fetcher_value is not None:
        local_build_args['custom_fetcher'] = custom_fetcher_value
    local_build_process = _create_ya_process(
        arcadia_root, use_ya_dev, "package", "build_package_locally", work_dir=None, stdout=None, **local_build_args
    )
    local_build_process.build_dir = local_build_args["custom_build_directory"]
    local_build_process.description = "local build"

    distbuild_args = copy.deepcopy(kwargs)
    distbuild_args['verbose'] = False
    distbuild_args['log_file'] = add_link_func('build_package_distbuild.log')
    distbuild_args['html_display'] = add_link_func('build_package_distbuild.html')
    distbuild_args["use_distbuild"] = True
    distbuild_args["custom_build_directory"] = _get_build_cache_dir(build_system=consts.DISTBUILD_BUILD_SYSTEM)
    if custom_fetcher_value is not None:
        distbuild_args['custom_fetcher'] = custom_fetcher_value
    if "dist_priority" not in distbuild_args:
        distbuild_args["dist_priority"] = _get_dist_priority()
    if distbuild_args.get("distbuild_pool"):
        distbuild_args["use_new_distbuild_client"] = True
    dist_build_process = _create_ya_process(
        arcadia_root, use_ya_dev, "package", "build_package_distbuild", work_dir=None, stdout=None, **distbuild_args
    )
    dist_build_process.build_dir = distbuild_args["custom_build_directory"]
    dist_build_process.description = "distbuild"

    process = wait_any([local_build_process, dist_build_process], timeout=10800)
    logging.info("Finished building by '%s' with exit code %s", process.description, process.returncode)
    if process == dist_build_process and process.returncode != 0:
        logging.info("Waiting for local build")
        process = local_build_process
        process.wait()

    _check_ya_return_code(process)
    return process.build_dir


def do_bloat(arcadia_root, bin_root_dir, binaries, use_ya_dev=False):
    if not len(binaries):
        return

    bloat_common = channel.channel.task.abs_path('bloat_common')
    stp.mkdirp(bloat_common)
    ya_bloat_res = channel.channel.task.create_resource(
        description='ya tool bloat output',
        resource_path=bloat_common,
        resource_type=resource_types.YA_BLOAT_OUTPUT,
        attributes={
            'ttl': 4
        }
    )
    for binary in binaries:
        bin_abs_path = os.path.join(bin_root_dir, binary)
        bloat_args = [
            '--save-html',
            'bloat_{}'.format(os.path.basename(binary)),
            '--input',
            bin_abs_path,
            '--linker-map',
            bin_abs_path + '.map.lld',
        ]
        run_tool(arcadia_root, 'bloat', bloat_args, use_ya_dev=use_ya_dev, work_dir=bloat_common)

    channel.channel.task.mark_resource_ready(ya_bloat_res.id)


def _get_branch(parsed_url):
    if parsed_url.branch:
        return '/arc/branches/' + parsed_url.branch + '/arcadia'
    if parsed_url.tag:
        return '/arc/tags/' + parsed_url.tag + '/arcadia'
    return None


def _get_svn_url(parsed_url):
    if parsed_url.trunk:
        return '/arc/trunk/arcadia'
    return _get_branch(parsed_url)


def do_clone(arcadia_url, task, use_checkout=True):
    parsed_arcadia_url = svn.Arcadia.parse_url(arcadia_url)
    branch = _get_branch(parsed_arcadia_url)
    revision = parsed_arcadia_url.revision

    logging.info('Do clone for %s %s', branch, revision)
    ya = svn.Arcadia.export('svn+ssh://arcadia.yandex.ru/arc/trunk/arcadia/ya', 'ya')

    if use_checkout:
        fake_svn = str(task.arcaphazotron)
        logging.info('Using "%s" as svn executable', fake_svn)
        os.environ['SVN_EXECUTABLE'] = fake_svn

    cmd = [
        ya,
        '-v',
        'clone'
    ]

    if revision:
        cmd += [
            '-r',
            revision
        ]

    if branch:
        cmd += [
            '--branch',
            branch
        ]

    env_vars = {}
    custom_fetcher_value = _get_custom_fetcher()
    if custom_fetcher_value is not None:
        env_vars['YA_CUSTOM_FETCHER'] = custom_fetcher_value

    with statistics.measure_time('ya_clone'):
        with process.CustomOsEnviron(env_vars):
            process.run_process(cmd, log_prefix='ya_clone')

    return os.path.join(os.getcwd(), 'arcadia')


def do_dump(source_dir, dump_handler, use_dev_version=False, parse_json=False, **kwargs):
    custom_fetcher_value = _get_custom_fetcher()
    if custom_fetcher_value is not None and 'custom_fetcher' not in kwargs:
        kwargs['custom_fetcher'] = custom_fetcher_value

    handler_args = {
        "handler": dump_handler,
        "args": kwargs,
    }

    output = channel.channel.task.log_path("run_ya_dump_{}.out.txt".format(dump_handler))
    with open(output, "w") as afile:
        _run_ya_handler(source_dir, use_dev_version, "dump", stdout=afile, **handler_args)
    with open(output) as afile:
        if parse_json:
            return json.load(afile)
        else:
            return afile.read()


def dump_modules(source_dir, targets):
    """
    :param source_dir: path to arcadia folder
    :param targets: absolute paths to targets inside arcadia
    :return: stdout of "ya dump modules targets"
    """
    return do_dump(source_dir, "modules", build_targets=targets)


def do_check(source_dir, targets, autocommit=False, use_dev_version=False, parse_json=False, **kwargs):
    """
    Run 'ya check'
    :param source_dir: arcadia source root
    :param targets: list of targets to include into patch
    :param autocommit: adds --commit
    :param use_dev_version: use ya-dev
    :param parse_json: adds --json-output
    :param kwargs: other args to ya check
    :return: if parse_json=True returns JSON object with keys 'check_id', 'check_url', 'zipatch_url'
             if parse_json=False returns stderr of the command
    """
    kwargs['build_targets'] = targets
    kwargs['commit_on_success'] = autocommit
    if 'json_output' not in kwargs and parse_json:
        kwargs['json_output'] = True

    json_output = channel.channel.task.log_path("run_ya_check.out.txt")
    raw_output = channel.channel.task.log_path("run_ya_check.err.txt")
    with open(raw_output, "w") as ofile, open(json_output, "w") as jfile:
        _run_ya_handler(source_dir, use_dev_version, "check", stdout=jfile, stderr=ofile, work_dir=source_dir, **kwargs)

    if parse_json:
        with open(json_output) as jfile:
            return json.load(jfile)
    else:
        with open(raw_output, "w") as ofile:
            return ofile.read()


def get_required_atd_files(source_dir, targets, use_dev_version, clear_build=False):
    def get_atd_paths(path):
        args = {
            "build_targets": [os.path.join(source_dir, path)],
            "follow_dependencies": False,
            "clear_build": clear_build,
        }
        tests_info = do_dump(source_dir, "json-test-list", use_dev_version, parse_json=True, **args)

        paths = set()
        for entry in tests_info:
            for path in entry.get('dart_info', {}).get('TEST-DATA', []):
                if path.startswith(ARCADIA_TESTS_DATA):
                    paths.add(path)
        logging.debug("Found atd dependencies: %s", paths)

        return paths

    logging.info("Obtaining required arcadia_tests_data paths")
    atd_paths = set()
    for target in targets:
        atd_paths.update(get_atd_paths(target))

    return [p.encode("UTF-8") if isinstance(p, six.text_type) else p for p in atd_paths]


def update_atd(atd_paths):
    def remove_nested_dirs(paths):
        result = []
        top_dir = None
        paths = deque(sorted(paths))
        while paths:
            path = paths.popleft()
            if not top_dir or not path.startswith(top_dir):
                result.append(path)
                top_dir = path
        return result

    atd_paths = [
        path[len(ARCADIA_TESTS_DATA) + 1:] if path.startswith(ARCADIA_TESTS_DATA) else path
        for path in atd_paths
    ]
    atd_paths = remove_nested_dirs(atd_paths)
    logging.info("Updating arcadia_tests_data paths: %s", atd_paths)
    for path in atd_paths:
        svn.ArcadiaTestData.get_arcadia_test_data(channel.channel.task, svn.Arcadia.append(get_atd_url(), path))


def get_atd_url():
    return os.path.normpath(
        svn.Arcadia.append(channel.channel.task.ctx[consts.ARCADIA_URL_KEY], os.path.join("..", ARCADIA_TESTS_DATA))
    )


def get_atd_path():
    # test_data_location returns value is a tuple, with tests data root as first irun_testsndex
    return svn.ArcadiaTestData.test_data_location(svn.Arcadia.append(get_atd_url(), "null"))[0]


def fuse_available():
    return vcs_util.fuse_available()


class ArcadiaApiNoSuchUrl(aapi.ArcadiaApiCommandFailed):
    pass


def wait_aapi_url(url, timeout=600):
    aapi_path, commit_id = aapi.ArcadiaApi.extract_path_and_revision(url)

    if aapi_path is None:
        raise ArcadiaApiNoSuchUrl('Url {} is not present in arcadia-api'.format(url))

    if aapi.ArcadiaApi.is_hg_scheme(url):
        changeset = aapi.ArcadiaApi.hg_id(commit_id)
        return aapi.ArcadiaApi.has_hg_changeset(changeset) or aapi.ArcadiaApi.wait_hg_changeset(changeset, timeout)

    else:
        revision = commit_id

        if revision is None:
            revision = aapi.ArcadiaApi.svn_head()

        return aapi.ArcadiaApi.has_svn_revision(revision) or aapi.ArcadiaApi.wait_svn_revision(revision, timeout)


def is_arc_path(url):
    """
    Checks whether this url is a Arc VCS mount path
    """
    return urlparse.urlparse(url).scheme == svn.Arcadia.ARCADIA_ARC_SCHEME


def mount_arc_path(
    url,
    timeout=600,
    fallback=False,
    use_arc_instead_of_aapi=True,
    arc_oauth_token=None,
    fetch_all=False,
    minimize_mount_path=False,
):
    """
    Returns MountPoint context manager.

    Usage example:
    with mount_arc_path('arcadia:/arc/trunk/arcadia@3500000') as path:
        # Here path exists and /trunk/arcadia@3500000 is mounted to this path.
        # Path is modifiable.
        pass
    # Here path does not exist or empty, arcadia is no longer mounted to this path.

    Url examples
    ------------

    SVN:
        arcadia:/arc/trunk/arcadia
        arcadia:/arc/trunk/arcadia@3500000
        arcadia:/arc/tags/jupiter/stable-73-28/arcadia
        arcadia:/arc/tags/jupiter/stable-73-28/arcadia@3866506
    HG:
        arcadia-hg:/#default
        arcadia-hg:/#users/heretic/remove-check
        arcadia-hg:/#f0311b60db18
    ARC:
        arcadia-arc:/#trunk
        arcadia-arc:/#users/heretic/remove-check
        arcadia-arc:/#0c72621f2381c1b4772ab72df86888b89bc9c0e2

    :param url:                     Arcadia url
    :param use_arc_instead_of_aapi: Use Arc for SVN-style URLs
    :param minimize_mount_path:     Try to minimize mount path prefix (arc)
    :return:                  MountPoint context manager
    """
    url = url.strip()
    logging.info('Mount arc path %s', url)

    clone_path, commit_id, req_type = vcs_util.extract_path_details(url)
    logging.debug(
        "vcs_util.extract_path_details clone_path: %s commit_id: %s, req_type: %s",
        clone_path, commit_id, req_type,
    )

    if clone_path is None:
        raise ArcadiaApiNoSuchUrl('Url {} is not present in arcadia-api'.format(url))

    if req_type == vcs_util.ARC_VCS:
        if fallback:
            logging.warn("fallback is not respected while working with Arc VCS")

        logging.info('Using Arc VCS to mount, commit_id: {}'.format(commit_id))
        return arc.Arc(arc_oauth_token=arc_oauth_token).mount_path(clone_path, commit_id, fetch_all=fetch_all,
            minimize_mount_path=minimize_mount_path,
        )

    elif req_type == vcs_util.AAPI_HG:
        changeset = aapi.ArcadiaApi.hg_id(commit_id)

        if (
            not aapi.ArcadiaApi.has_hg_changeset(changeset)
            and not aapi.ArcadiaApi.wait_hg_changeset(changeset, timeout)
        ):
            if not fallback:
                raise aapi.ArcadiaApiNoSuchRevision(
                    'Changeset {}(referred as {}) is not present in arcadia-api. Waited {}s.'.format(
                        changeset,
                        commit_id,
                        timeout
                    )
                )

        else:
            return aapi.ArcadiaApi.mount_hg_path(clone_path, changeset)

    elif req_type == vcs_util.SVN:
        revision = commit_id
        if use_arc_instead_of_aapi:
            try:
                return arc.Arc(arc_oauth_token=arc_oauth_token).mount_svn_path(
                    clone_path, revision=revision, checkout_timeout=timeout,
                    minimize_mount_path=minimize_mount_path,
                )
            except arc.ArcNoSuchRevision:
                if not fallback:
                    raise
        else:
            now = time.time()
            utcnow = datetime.datetime.utcnow()
            try:
                if revision is None:
                    revision = aapi.ArcadiaApi.svn_head()

                if (
                    not aapi.ArcadiaApi.has_svn_revision(revision)
                    and not aapi.ArcadiaApi.wait_svn_revision(revision, timeout)
                ):
                    if not fallback:
                        raise aapi.ArcadiaApiNoSuchRevision(
                            'Revision {} is not present in arcadia-api. Waited {}s.'.format(revision, timeout)
                        )

                else:
                    return aapi.ArcadiaApi.mount_svn_path(clone_path, revision=revision)
            finally:
                if common.statistics.Signaler.instance is not None:
                    duration = int((time.time() - now) * 1000)
                    common.statistics.Signaler().push(dict(
                        type=ctst.SignalType.TASK_OPERATION,
                        kind=ctst.OperationType.AAPI,
                        date=utcnow,
                        timestamp=utcnow,
                        method="mount_svn",
                        duration=duration,
                    ))

    # fallback
    assert fallback

    parsed_url = svn.Arcadia.parse_url(url)

    if not aapi.ArcadiaApi.is_hg_scheme(url) and not parsed_url.path.endswith('/arcadia'):
        eh.check_failed('Cannot fallback for non-arcadia url: {}'.format(url))

    path = svn.Arcadia.get_arcadia_src_dir(url)

    # TODO: mounted_path_svnversion() doesn't work with fallback mode

    return context_managers.nullcontext(path)


def mounted_path_svnversion(path, arc_vcs=False):
    """
    Returns mounted path metadata.

    :param path: mount point path
    :return: metadata dict in following format
    {
        "date": <str>,
        "author": <str>,
        "revision": <str>,
        "repository": <str>,
    }

    if arc_vcs is True, revision can be of form `arc:<hash>`.
    """
    if arc_vcs:
        try:
            logging.info("getting information about HEAD revision of Arc repo")
            cmd = ["arc", "show", "HEAD", "--json"]
            out = subprocess.check_output(cmd, cwd=path)
            logging.debug("got output %ls", out)
        except subprocess.CalledProcessError as e:
            eh.check_failed("Cannot find svnversion. Caught {!r} while running `{}`.".format(e, " ".join(cmd)))

        try:
            data = json.loads(out)[0]["commits"][0]
            metadata = {
                'author': data['author'],
                'date': data['date'],
                'repository': "arc-vcs.yandex-team.ru",
            }
            if data.get('revision'):
                metadata['revision'] = data['revision']
            else:
                metadata['revision'] = 'arc:%s' % data['commit']
            return metadata
        except (ValueError, IndexError) as e:
            eh.check_failed("Unable to parse HEAD info from output {}, caught {!r}".format(out, e))

    with open(os.path.join(path, '__SVNVERSION__')) as f:
        return json.load(f)


def run_tool(source_root, tool, args=None, use_ya_dev=False, stdout=None, stderr=None, work_dir=None, timeout=600):
    env_vars = {}
    custom_fetcher_value = _get_custom_fetcher()
    if custom_fetcher_value is not None:
        env_vars['YA_CUSTOM_FETCHER'] = custom_fetcher_value

    with statistics.measure_time('ya_tool_{}'.format(tool)):
        with process.CustomOsEnviron(env_vars):
            tool_process = process.run_process(
                [_get_ya_tool(source_root), 'tool', tool] + args or [],
                log_prefix='run_tool_' + tool,
                check=False,
                work_dir=work_dir,
                outputs_to_one_file=False,
                stdout=stdout,
                stderr=stderr,
                timeout=timeout,
            )
    _check_ya_return_code(tool_process)


def _do_pgo_merge(
    source_root, merged_profile, raw_profiles,
    use_ya_dev=False, stdout=None, stderr=None, work_dir=None, timeout=600
):
    run_tool(
        source_root, 'llvm-profdata',
        ['merge', '-output=' + merged_profile] + raw_profiles,
        use_ya_dev=use_ya_dev, stdout=stdout, stderr=stderr, work_dir=work_dir, timeout=timeout
    )


def _python():
    if os.name == 'nt':
        return 'python'
    elif not common.system.inside_the_binary():
        return sys.executable
    else:
        return '/skynet/python/bin/python'


def _hardlink_tree(src, dst):
    names = os.listdir(src)

    if not os.path.isdir(dst):
        if os.path.exists(dst):
            os.remove(dst)
        os.mkdir(dst)

    for name in names:
        srcname = os.path.join(src, name)
        dstname = os.path.join(dst, name)

        if os.path.isdir(srcname):
            _hardlink_tree(srcname, dstname)
        else:
            if os.path.isfile(dstname):
                os.remove(dstname)
            elif os.path.isdir(dstname):
                shutil.rmtree(dstname)
            try:
                os.link(srcname, dstname)
            except OSError:
                logging.error('Unable to create hardlink from {} to {}'.format(srcname, dstname))
                raise


def hardlink_or_copy(src, dst):
    if common.system.inside_the_binary():
        from library.python import fs
        return fs.hardlink_or_copy(src, dst)
    try:
        os.link(src, dst)
    except OSError as e:
        if e.errno != errno.EXDEV:
            raise
        shutil.copy2(src, dst)


def apply_patch(task, arcadia_path, arcadia_patch, dest_dir, selective_checkout=False):
    if not arcadia_patch:
        return None, False

    tool_name = 'zipatcher'
    cmd_args = ['--arcadia-root', arcadia_path]

    if hasattr(task, 'ctx'):
        is_sdk1 = True
    elif hasattr(task, 'Parameters'):
        is_sdk1 = False
    else:
        raise eh.check_failed('Seems like your task is neither SDK1 nor SDK2 based o_O')

    def get_revision():
        url = task.ctx[consts.ARCADIA_URL_KEY] if is_sdk1 else getattr(task.Parameters, consts.ARCADIA_URL_KEY)
        return svn.Arcadia.parse_url(url).revision

    if arcadia_patch.startswith('zipatch:arc:'):
        revision = get_revision()

        arcadia_patch = arcadia_patch.strip()[len('zipatch:arc:'):].strip()
        eh.ensure(arcadia_patch, 'Incorrect specification of arcadia zipatch: missing pullrequest id')

        parts = arcadia_patch.split(':')
        eh.ensure(
            len(parts) >= 3,
            'Incorrect specification of arcadia zipatch: missing vault owner and name for ssh key'
        )

        arcadia_patch, key_owner, key_name = parts

        parts = arcadia_patch.strip().split('-')
        pr_iter = parts[1] if len(parts) == 2 and parts[1] else None

        cmd_args.extend(['--arcanum-id', parts[0], '--save-to-dir', dest_dir])
        if pr_iter is not None:
            cmd_args.extend(['--arcanum-iteration', pr_iter])

        env = os.environ.copy()
        env['SSH_KEY_OWNER'] = 'zomb-sandbox-rw'
        with process.CustomOsEnviron(env):
            run_tool(arcadia_path, tool_name, cmd_args)

        zipatcher_output_path = os.path.join(str(paths.get_logs_folder()), 'run_tool_' + tool_name + '.out.txt')

        with open(zipatcher_output_path, 'r') as f:
            zipatcher_result = json.loads(f.read())

        if is_sdk1:
            if revision is None:
                revision = zipatcher_result['base_revision']
                channel.channel.sandbox.set_task_context_value(task.id, consts.ARCADIA_URL_KEY,
                                                               task.ctx[consts.ARCADIA_URL_KEY] + '@' + str(revision))

        eh.ensure(
            int(revision) == int(zipatcher_result['base_revision']),
            'You can run {} with this patch only against revision {}'.format(
                task.type, zipatcher_result['base_revision']
            )
        )

        if pr_iter is None and is_sdk1:

            pr_iter = zipatcher_result.get('used_iteration')
            if pr_iter is None:
                task.set_info(
                    'Failed to update Context field for arcadia patch with actual iteration value '
                    'of Arcanum pullrequest: zipatcher report incomplete'
                )
            else:
                channel.channel.sandbox.set_task_context_value(
                    task.id, consts.ARCADIA_PATCH_KEY,
                    task.ctx[consts.ARCADIA_PATCH_KEY] + '-' + str(pr_iter)
                )

        return zipatcher_result.get('zipatch_path')

    elif selective_checkout:
        if arcadia_patch.startswith('zipatch:'):
            revision = get_revision()

            zipatch_path, _ = svn.Arcadia.fetch_patch(arcadia_patch, str(dest_dir))
            cmd_args.extend(['--file', zipatch_path, '--base-revision', revision])

            with process.CustomOsEnviron({'SVN_SSH_MUX_CP_DIR': svn.tunnel.SshTunnel.control_path_dir}):
                # 15 min should be enough for zipatch with up to 4-5 thousands of actions
                run_tool(arcadia_path, tool_name, cmd_args, timeout=15 * 60)

            return zipatch_path
        else:
            eh.check_failed('Only zipatch can be applied to sparse arcadia checkout')

    else:
        svn.Arcadia.apply_patch(arcadia_path, arcadia_patch, dest_dir)


def setup_frepkage(frepkage):
    assert frepkage.count(':') > 1, 'Expected format: "frepkage:<scheme>:<data>"'
    prefix = 'frepkage:'
    assert frepkage.startswith(prefix), frepkage

    _, scheme, data = frepkage.split(':', 2)

    if scheme == 'sbr':
        filename = channel.channel.task.sync_resource(int(data))
        dirname = os.path.abspath('frepkage')

        logging.debug("Extracting frepkage %s to %s", frepkage, dirname)
        with tarfile.open(filename) as tf:
            tf.extractall(dirname)

        os.chmod(os.path.join(dirname, 'ya-bin'), 0o755)
        return dirname
    else:
        raise NotImplementedError("Don't know what to do with '{}' scheme: {}".format(scheme, frepkage))
