# -*- coding: utf-8 -*-
import json
import logging
import re
import sandbox.sdk2 as sdk2
from sandbox.common.types.task import ReleaseStatus
from sandbox.projects.common import link_builder as lb
from sandbox.projects.release_machine import rm_notify
from sandbox.projects.release_machine import notify_helper
from sandbox.sandboxsdk.svn import Arcadia


def get_mention(task, login):
    try:
        return '@{}'.format(rm_notify.get_mention(task, person=login))
    except Exception:
        logging.error('Can not fetch tg login for %s', login, exc_info=True)
        return login


def get_task_link(task_id):
    return lb.task_link(task_id, link_name=str(task_id))


def prepare_revision_info(task, commit_info, only_title=False):
    commit_message = commit_info['msg'].replace('&', '&amp;').replace('<', '&lt;').replace('>', '&gt;')
    builders = {
        r'([A-Z]+-[0-9]+)': lb.st_link,
        r'REVIEW:\s+([0-9]+)': lb.review_link,
    }

    def build_link(builder):
        return lambda matchobj: builder(matchobj.group(1), link_name=matchobj.group(0))

    for pattern, builder in builders.items():
        commit_message = re.sub(pattern, build_link(builder), commit_message)
    if only_title:
        commit_message = commit_message.strip().split('\n')[0]

    return [
        'Revision {revision_link} by {author}'.format(
            revision_link=lb.revision_link(commit_info['revision']),
            author=get_mention(task, commit_info['author']),
        ),
        commit_message,
    ]


class Action(object):
    def __init__(self, text, url):
        self.text = text
        self.url = url

    def dump(self):
        return {'text': self.text, 'url': self.url}


def send_notification(task, notification, actions=None):
    logging.info('Notification text: %s', notification)
    if task.Parameters.notify and (task.Parameters.chats or task.Parameters.people):
        kwargs = {}
        if actions:
            kwargs['reply_markup'] = json.dumps({
                'inline_keyboard': [[action.dump()] for action in actions]
            })
        notify_helper.telegram(
            task,
            notification,
            chat_ids=task.Parameters.chats or (),
            people=task.Parameters.people or (),
            **kwargs
        )


def with_notify_parameters(parameters_class):
    class WithNotifyParameters(parameters_class):
        notify = sdk2.parameters.Bool('Notify about build/release', default=False)
        chats = sdk2.parameters.List('Telegram chat ids to send notification')
        people = sdk2.parameters.List('People or chat names to send notification')
    return WithNotifyParameters


def with_build_and_release_on_commit_notifier_parameters(parameters_class):
    class WithBuildAndReleaseOnCommitNotifierParameters(with_notify_parameters(parameters_class)):
        checkout_arcadia_from_url = sdk2.parameters.ArcadiaUrl()
        with sdk2.parameters.CheckGroup(
                'Release statuses to notify',
                default=[ReleaseStatus.STABLE]
        ) as release_statuses_to_notify:
            release_statuses_to_notify.choices = [(_, _) for _ in list(ReleaseStatus)]
    return WithBuildAndReleaseOnCommitNotifierParameters


class BuildAndReleaseOnCommitNotifier(object):
    class Parameters(with_build_and_release_on_commit_notifier_parameters(sdk2.Parameters)):
        pass

    @property
    def tool_name(self):
        raise NotImplementedError

    def prepare_tech_info(self, commit_info):
        return [
            'Build task: {task_link}'.format(task_link=get_task_link(self.id)),
        ]

    def prepare_build_notification(self, commit_info):
        return '\n'.join(
            [
                'New version of {tool} was built! Should it be released?'.format(
                    tool=self.tool_name,
                ),
            ] + self.prepare_tech_info(commit_info) + prepare_revision_info(self, commit_info)
        )

    def prepare_release_notification(self, commit_info, status):
        return '\n'.join(
            [
                '{tool} was released to {status}'.format(
                    tool=self.tool_name,
                    status=status,
                ),
            ] + self.prepare_tech_info(commit_info) + prepare_revision_info(self, commit_info)
        )

    def _notify(self, notification_builder, actions=None, *builder_args, **builder_kwargs):
        commit_info = Arcadia.log(
            self.Parameters.checkout_arcadia_from_url,
            limit=1
        )[0]
        notification = notification_builder(commit_info, *builder_args, **builder_kwargs)
        send_notification(self, notification, actions)

    def notify_on_build(self):
        self._notify(
            self.prepare_build_notification,
            actions=[
                Action(
                    text='Release task',
                    url=lb.task_link(self.id, plain=True)
                )
            ]
        )

    def notify_on_release(self, additional_parameters):
        logging.info(
            'Releasing to %s. Notifications set up for %s',
            additional_parameters['release_status'],
            ', '.join(self.Parameters.release_statuses_to_notify),
        )
        if additional_parameters['release_status'] in self.Parameters.release_statuses_to_notify:
            self._notify(self.prepare_release_notification, status=additional_parameters['release_status'])
