# -*- coding: utf-8 -*-
import json
import logging
import os
import re
from sandbox import sdk2
from sandbox.common import errors
from sandbox.common.types.task import ReleaseStatus
from sandbox.common.types.task import Status
from sandbox.projects.common.backend_build_pipeline.BuildAndReleaseOnCommitNotifier import (
    get_mention, get_task_link, prepare_revision_info, send_notification, with_notify_parameters
)
from sandbox.projects.common.build import parameters as build_parameters
from sandbox.projects.release_machine import changelogs as rm_ch
from sandbox.projects.release_machine.components import all as rmc
from sandbox.sdk2.vcs.svn import Arcadia


class DeliverAndNotifyTelegram(sdk2.Task):
    class Requirements(sdk2.Requirements):
        cores = 1
        ram = 1024

        class Caches(sdk2.Requirements.Caches):
            pass

    class Parameters(with_notify_parameters(sdk2.Parameters)):
        with sdk2.parameters.String(
                'Component name (if applicable)',
                default='',
                required=False,
        ) as component_name:
            component_name.choices = [(_, _) for _ in rmc.get_component_names()]
        env = sdk2.parameters.RadioGroup(
            'Environment',
            choices=[(_, _) for _ in list(ReleaseStatus)],
            default=ReleaseStatus.TESTING,
        )
        use_already_built = sdk2.parameters.Bool('Use already built backend')
        message = sdk2.parameters.String('Release message')
        include_commit_info_in_release_message = sdk2.parameters.Bool(
            'Include commit info in release message',
            default=False,
        )
        resource_type = sdk2.parameters.String('Resource type to build changelog')
        with use_already_built.value[False]:
            checkout_arcadia_from_url = sdk2.parameters.ArcadiaUrl()
            task_type = sdk2.parameters.String('Build task type')
        with use_already_built.value[True]:
            task = sdk2.parameters.Task('Build task')
        force = sdk2.parameters.Bool('Force release (without workflow check)', default=False)

    def build(self):
        if self.Context.build_task_id:
            return self.Context.build_task_id

        logging.info(
            'Launch build task %s with arcadia url: %s',
            self.Parameters.task_type,
            self.Parameters.checkout_arcadia_from_url,
        )

        build_task_class = sdk2.Task[self.Parameters.task_type]
        build_task = build_task_class(
            self,
            description=self.Parameters.message,
            **{
                build_parameters.ArcadiaUrl.name: self.Parameters.checkout_arcadia_from_url,
                build_parameters.ClearBuild.name: True,
            }
        ).enqueue()

        self.Context.build_task_id = build_task.id
        self.Context.save()
        raise sdk2.WaitTask(
            [build_task],
            Status.Group.FINISH | Status.Group.BREAK,
            wait_all=True
        )

    def get_build_task_id(self):
        if self.Parameters.use_already_built:
            build_task_id = self.Parameters.task.id
        else:
            build_task_id = self.build()

        build_task = sdk2.Task[build_task_id]
        if build_task.status not in [Status.SUCCESS, Status.RELEASED]:
            raise errors.TaskFailure(
                'Build was not successful: {}'.format(build_task.status),
            )

        self.Context.build_task_arcadia_url = build_task.Context.checkout_arcadia_from_url
        self.Context.save()
        return build_task_id

    def get_highest_tag(self):
        """Get highest tag made for component
        :return: highest tag name (prj_dir/tag_name)
        """
        assert self.Parameters.component_name

        c_info = rmc.get_component(self.Parameters.component_name)
        last_branch_num = c_info.last_branch_num
        last_tag_num = c_info.last_tag_num(last_branch_num)
        tag_folder_name = c_info.svn_cfg__tag_folder_name(last_branch_num, last_tag_num)
        logging.info('Fetched last tag number: %s', tag_folder_name)
        return os.path.join(self.Parameters.component_name, tag_folder_name)

    def get_build_task_version_info(self, task_id):
        """Get build_task version info.
        :return: version_info
        """
        prod_build_task = sdk2.Task[task_id]
        parsed_url = Arcadia.parse_url(
            prod_build_task.Context.checkout_arcadia_from_url
        )
        logging.info(
            'Fetched version info for task id "%s": %s',
            task_id,
            parsed_url,
        )
        return parsed_url

    def extract_branch_tag_num(self, tag_name):
        """Extract branch num, tag num from tag name
        Tag has format: stable-{branch_num}-{tag_num}
        :param tag_name: str
        :return: branch_num, tag_num
        """
        tag_regex = r'.*-([0-9]+)-([0-9]+)'
        match = re.match(tag_regex, tag_name)
        branch_num, tag_num = int(match.group(1)), int(match.group(2))
        return branch_num, tag_num

    def is_tag_less(self, tag1, tag2):
        """
        :return: tag1 < tag2
        """
        branch_num1, tag_num1 = self.extract_branch_tag_num(tag1)
        branch_num2, tag_num2 = self.extract_branch_tag_num(tag2)

        if (branch_num1, tag_num1) < (branch_num2, tag_num2):
            result = True
        else:
            result = False

        logging.info(
            'Tags comparison: "%s" < "%s" == %s',
            tag1, tag2, result,
        )
        return result

    def is_allowed_to_release_on_test(self):
        """Check if it is allowed to release build on test.
        :return: bool
        """
        assert not self.Parameters.force
        assert self.Parameters.env == ReleaseStatus.TESTING

        if not self.Parameters.component_name:
            return True

        # collect data
        c_info = rmc.get_component(self.Parameters.component_name)

        highest_tag = self.get_highest_tag()

        try:
            released_stable_resource = next(c_info.get_last_release(stage=ReleaseStatus.STABLE))
        except StopIteration:
            production_version_info = None
        else:
            production_version_info = self.get_build_task_version_info(
                released_stable_resource.build_task_id
            )

        try:
            released_testing_resource = next(c_info.get_last_release(stage=ReleaseStatus.TESTING))
        except StopIteration:
            test_version_info = None
        else:
            test_version_info = self.get_build_task_version_info(
                released_testing_resource.build_task_id
            )

        if self.Parameters.use_already_built:
            build_version_info = self.get_build_task_version_info(
                self.Parameters.task.id
            )
        else:
            build_version_info = Arcadia.parse_url(self.Parameters.checkout_arcadia_from_url)

        logging.info(
            'Check release allowance: highest tag "%s"; '
            'production version "%s"; testing version "%s"; build version "%s"',
            highest_tag, production_version_info, test_version_info, build_version_info
        )
        # if latest tag is not released and we don't release it
        if (
                production_version_info and
                production_version_info.tag and
                self.is_tag_less(production_version_info.tag, highest_tag) and
                highest_tag != build_version_info.tag
        ):
            logging.info('Not allowed to release on testing')
            return False

        logging.info('Check if newer version is released (from trunk)')
        # if newer version already released on test from trunk
        if (
                test_version_info and
                test_version_info.trunk and
                build_version_info.trunk and
                test_version_info.revision > build_version_info.revision
        ):
            logging.info('Release is not allowed, due to newer version already released')
            return False

        logging.info('Release allowed')
        return True

    def trunk_changelog(self, c_info, build_version_info, last_released_trunk_resource_revision):
        """
        Generate trunk changelog between current build and last released trunk resource.

        :param c_info: Release machine component info
        :param build_version_info: parsed arcadia url
        :param last_released_trunk_resource_revision: last resource revision
        :return: List of strings of changes
        """
        # if force release older version then changelog is msg of older version
        if last_released_trunk_resource_revision > build_version_info.revision:
            changelog_start_revision = build_version_info.revision
        else:
            # skip last revision of previous release
            changelog_start_revision = (int(last_released_trunk_resource_revision) + 1)
        path = "arcadia:/arc/trunk"
        changelog_master = rm_ch.ChangeLogMaster(
            changelog_start_revision,
            path, changelog_start_revision,
            path, build_version_info.revision,
            [rm_ch.PathsFilter([
                c_info.changelog_cfg__set_paths_importance(i) for i in c_info.changelog_cfg__dirs
            ])],
        )
        changelog = [
            '\n'.join(prepare_revision_info(self, i.vcs_info, only_title=True))
            for i in changelog_master.get_changelog()
        ]
        logging.info("Got changelog:\n%s", json.dumps(changelog, indent=2))
        return changelog

    def changelog(self, build_version_info):
        """Build changelog for testing releases from trunk
        :param build_version_info: parsed arcadia url
        :return: list of changelog messages
        """
        if (
                not build_version_info.trunk or
                self.Parameters.env != ReleaseStatus.TESTING
        ):
            return []

        if not self.Parameters.component_name or not self.Parameters.resource_type:
            return []

        c_info = rmc.get_component(self.Parameters.component_name)
        last_released_trunk_resource = sdk2.Resource.find(
            type=self.Parameters.resource_type,
            attrs={
                "released": ReleaseStatus.TESTING,
                "arcadia_trunk": True,
            }
        ).order(-sdk2.Resource.id).first()

        return self.trunk_changelog(c_info, build_version_info, last_released_trunk_resource.arcadia_revision)

    @staticmethod
    def format_version(version_info):
        """Format version from url to human-readable format and create url on
        source code in arc.
        :param version_info: ArcadiaUrl
        :return: text_version
        """
        msg_parts = []

        if version_info.trunk:
            msg_parts.append('Source: TRUNK')
        elif version_info.branch:
            msg_parts.extend([
                'Source: BRANCH',
                'Branch: {}'.format(version_info.branch),
            ])
        elif version_info.tag:
            msg_parts.extend([
                'Source: TAG',
                'Tag: {}'.format(version_info.tag),
            ])
        else:
            raise RuntimeError(
                'Unknown source for version: {}'.format(version_info)
            )

        return '\n'.join(
            msg_parts + ['Revision: {}'.format(version_info.revision)]
        )

    def notify(self, build_task_id):
        """Send notification about release to telegram chat.
        Notification works only from rm-release-bot.
        :param build_task_id: str
        :return: None
        """
        version_info = self.get_build_task_version_info(build_task_id)
        logging.info('Building changelog for version: %s', version_info)
        if version_info.trunk:
            changelog = self.changelog(version_info)
            changelog = '\n\n'.join(['Changelog:'] + changelog)
        else:
            changelog = ''

        formatted_version = self.format_version(version_info)
        logging.info('Formatted version: %s', formatted_version)

        notification = (
            '{released} {component} by {author}\n'
            'Build task: {build_task}\n'
            'Release task: {release_task}\n'
            'Environment: {env}\n'
            '{version_fmt}\n'
            '{log}'
        ).format(
            released='Force released' if self.Parameters.force else 'Released',
            component=self.Parameters.component_name or '',
            author=get_mention(self, self.author),
            build_task=get_task_link(build_task_id),
            release_task=get_task_link(self.id),
            env=self.Parameters.env,
            version_fmt=formatted_version,
            log=changelog,
        )
        send_notification(self, notification)

    def on_execute(self):
        if (
                not self.Parameters.force and
                self.Parameters.env == ReleaseStatus.TESTING and
                not self.is_allowed_to_release_on_test()
        ):
            raise errors.TaskFailure('It is not allowed to make release on test at this moment')

        build_task_id = self.get_build_task_id()
        release_kwargs = dict(
            task_id=build_task_id,
            type=self.Parameters.env,
        )
        if self.Parameters.include_commit_info_in_release_message and self.Context.build_task_arcadia_url:
            commit_info = Arcadia.log(
                self.Context.build_task_arcadia_url,
                limit=1
            )[0]
            commit_message = commit_info['msg']
            release_kwargs['subject'] = '{message}: {subject}'.format(
                message=self.Parameters.message,
                subject=commit_message.strip().split('\n')[0],
            )
            release_kwargs['comments'] = commit_message
        else:
            release_kwargs['subject'] = self.Parameters.message
        self.server.release(**release_kwargs)
        self.notify(build_task_id)
