# -*- coding: utf-8 -*-
import os
from sandbox import sdk2
from sandbox.common import errors
from sandbox.sandboxsdk import environments
from sandbox.sandboxsdk.svn import Arcadia
from sandbox.projects.common.backend_build_pipeline.BuildAndReleaseOnCommitNotifier import BuildAndReleaseOnCommitNotifier
from sandbox.projects.common.build.parameters import ArcadiaUrl
from sandbox.projects.common.nanny import nanny


class ExportTemplateResourcesAndNotifyTelegram(BuildAndReleaseOnCommitNotifier, nanny.ReleaseToNannyTask2, sdk2.Task):
    """
    Task for rendering jinja2 templates, exporting results as resources and notify about build and release to telegram chats
    """
    TEMPLATE_EXTS = (
        '.j2',
        '.tmpl',
        '.template',
    )

    @property
    def tool_name(self):
        return self.Parameters.export_name

    class Parameters(BuildAndReleaseOnCommitNotifier.Parameters):
        template_resources = sdk2.parameters.List('Template resources to export')
        checkout_arcadia_from_url = ArcadiaUrl()
        export_name = sdk2.parameters.String('Name of export in notification', default='Template resources')

    class Requirements(sdk2.Task.Requirements):
        environments = (
            environments.PipEnvironment("Jinja2"),
        )

    def render_template(self, template_path, result_path, template_params):
        from jinja2 import Template

        with open(template_path, 'r') as template_file:
            template_content = template_file.read()

        template = Template(template_content)
        result_content = template.render(**template_params)

        with open(result_path, 'w') as result_file:
            result_file.write(result_content)

    def export(self, resource_cls):
        commit_info = Arcadia.log(
            self.Parameters.checkout_arcadia_from_url,
            limit=1
        )[0]
        self.Context.commit_info = {
            k: commit_info[k]
            for k in ('msg', 'revision')
        }
        self.Context.save()

        commit_revision = commit_info['revision']
        arcadia_path = Arcadia.trunk_url(
            resource_cls.arcadia_build_path,
            revision=commit_revision,
        )
        if not Arcadia.check(arcadia_path):
            raise errors.TaskFailure(
                'Given resource path does not exists, path: {}, resource type: {}'.format(
                    arcadia_path,
                    resource_cls.__name__,
                )
            )

        os.mkdir(resource_cls.name)
        base_name = os.path.basename(resource_cls.arcadia_build_path)
        file_name = os.path.join(resource_cls.name, base_name)
        Arcadia.export(arcadia_path, file_name)

        name, ext = os.path.splitext(file_name)
        if ext in self.TEMPLATE_EXTS:
            self.render_template(file_name, name, getattr(resource_cls, 'template_params', {}))
            config_name = name
        else:
            config_name = file_name

        resource = resource_cls(
            self,
            'Source file {}, rev: {}'.format(
                resource_cls.arcadia_build_path,
                commit_revision,
            ),
            config_name,
        )
        resource_data = sdk2.ResourceData(resource)
        resource_data.ready()

    def on_execute(self):
        for resource_name in self.Parameters.template_resources:
            resource_cls = sdk2.Resource[resource_name]
            self.export(resource_cls)
        self.notify_on_build()

    def on_release(self, additional_parameters):
        if self.Context.commit_info:
            if not additional_parameters.get('release_subject'):
                additional_parameters['release_subject'] = '{name} r{revision}: {subject}'.format(
                    name=self.tool_name,
                    revision=self.Context.commit_info['revision'],
                    subject=self.Context.commit_info['msg'].strip().split('\n')[0],
                )
            if not additional_parameters.get('release_comments'):
                additional_parameters['release_comments'] = self.Context.commit_info['msg']
        super(ExportTemplateResourcesAndNotifyTelegram, self).on_release(additional_parameters)
        self.notify_on_release(additional_parameters)
