import copy
import json
import logging
import requests
from sandbox import sdk2
from sandbox.projects.common import link_builder as lb
import six
from six.moves.urllib import parse as urlparse


class GenerateExtServicesYasmPanelsBaseTask(sdk2.Task):
    """
    Base task for ext services yasm panels generation
    """
    class Requirements(sdk2.Task.Requirements):
        tasks_resource = None

    class Parameters(sdk2.Task.Parameters):
        yasm_api_endpoint = sdk2.parameters.String(
            'Yasm api endpoint',
            default='https://yasm.yandex-team.ru',
        )
        ext_services_base_path = sdk2.parameters.String(
            'Yasm panel tree base for ext services',
            required=True,
        )
        ext_services_alert_prefix = sdk2.parameters.String(
            'Prefix for ext services alerts',
            required=True,
        )
        ctypes = sdk2.parameters.List(
            'Ctypes',
            default=['prod'],
        )
        abc_slug = sdk2.parameters.String(
            'ABC service slug',
            required=True,
        )
        juggler_namespace = sdk2.parameters.String(
            'Juggler namespace',
            required=False,
        )
        juggler_tags = sdk2.parameters.JSON(
            'Juggler tags per ctype',
            required=False,
        )
        use_last_binary = sdk2.parameters.Bool(
            'Use last binary to run the task',
            default=False,
        )
        dry_run = sdk2.parameters.Bool(
            'Do not make actual requests',
            default=False,
        )

    class AlertThresholds(object):
        def __init__(self, warn, crit):
            self.warn = warn
            self.crit = crit

        @property
        def warn_range(self):
            return [min(self.warn, self.crit), max(self.warn, self.crit)]

        @property
        def crit_range(self):
            return [self.crit, None] if self.crit > self.warn else [None, self.crit]

    class FailPercentParams(object):
        SIGNAL_PREFIX = None

        @classmethod
        def code_signal(cls, service, code):
            return '<unistat|push>-{signal_prefix}.{service}_response_{code}_ammm'.format(
                signal_prefix=cls.SIGNAL_PREFIX,
                service=service,
                code=code
            )

        @classmethod
        def perc_signal(cls, service, fail_codes):
            failed_codes = 'sum({})'.format(','.join([cls.code_signal(service, code) for code in fail_codes]))
            all_codes = 'sum({})'.format(','.join([cls.code_signal(service, code) for code in cls.all_codes()]))
            return 'or(perc({failed},{all}), 0)'.format(failed=failed_codes, all=all_codes)

        @classmethod
        def default_fail_codes(cls):
            raise NotImplementedError()

        @classmethod
        def all_codes(cls):
            raise NotImplementedError()

    ALERT_NAME_TEMPLATE = '{prefix}.{itype}.{ctype}.{service}.{alert_type}'
    FAIL_PERC_ALERT = 'fail-perc'

    def _get_alerts(self):
        return {
            self.FAIL_PERC_ALERT: self.FailPercentParams.perc_signal
        }

    def recursive_merge_dict(self, *sources):
        iter_sources = iter(sources)
        result = copy.deepcopy(next(iter_sources))
        for source in iter_sources:
            for key, value in source.items():
                if key not in result:
                    result[key] = copy.deepcopy(value)
                elif isinstance(source[key], dict) and isinstance(result[key], dict):
                    result[key] = self.recursive_merge_dict(result[key], source[key])
        return result

    @staticmethod
    def slice_in_chunks(l, n):
        return [l[i:i+n] for i in range(0, len(l), n)]

    def _get_panel_template(self):
        """
        Returns pair (key, content)
        """
        raise NotImplementedError()

    def _get_ext_services(self):
        """
        Returns ext services enum
        """
        raise NotImplementedError()

    def _get_tags(self, ctype):
        """
        Returns dict of configuration -> yasm tags
        """
        raise NotImplementedError()

    def _get_ext_services_alerts_config(self, services):
        """
        Returns dict with alerts config
        """
        raise NotImplementedError()

    def _update_panel_templates(self, key, content):
        logging.info('Going to update panel %s', key)
        if self.Parameters.dry_run:
            logging.info('Dry run - not really updating')
            return

        response = requests.post(
            urlparse.urljoin(self.Parameters.yasm_api_endpoint, '/srvambry/tmpl/panels/update?key={}'.format(key)),
            json={
                'content': content,
            }
        )
        response.raise_for_status()

    def _generate_ext_services_menu_node(self, name, template_key, metrics, tags, graphs, children=None):
        substitutions = {
            key: ', '.join(value)
            for key, value in tags.items()
        }
        substitutions['metric_names'] = metrics if isinstance(metrics, six.string_types) else ', '.join(metrics)
        substitutions['graphs'] = graphs if isinstance(graphs, six.string_types) else ', '.join(graphs)
        node = dict(
            name=name,
            title=name,
            type='template_panel',
            item_id=template_key,
            substitutions=substitutions,
        )
        if children:
            node['children'] = children
        return node

    def _generate_ext_services_menu(self, template_key, ctype, services):
        tree = dict(
            name='ext_services',
            title='Ext services',
            type='dummy',
            children=[
                dict(
                    name=tag_name,
                    title=tag_name,
                    type='dummy',
                    children=[
                        self._generate_ext_services_menu_node(
                            name='_{}_{}'.format(tag_name, i + 1),
                            template_key=template_key,
                            metrics=[service.name for service in service_chunk],
                            tags=tags,
                            graphs=['time', 'status'],
                        )
                        for i, service_chunk in enumerate(self.slice_in_chunks(services, 10))
                    ] + [
                        self._generate_ext_services_menu_node(
                            name=service.name,
                            template_key=template_key,
                            metrics=service.name,
                            tags=tags,
                            graphs=['time', 'status', 'time_alert', 'status_alert'],
                        )
                        for service in services
                    ],
                )
                for tag_name, tags in self._get_tags(ctype).items()
            ],
        )
        subtree_path = self.Parameters.ext_services_base_path.format(ctype=ctype)
        logging.info(
            'Going to post subtree to %s\n%s',
            subtree_path,
            json.dumps(tree, indent=4),
        )
        if self.Parameters.dry_run:
            logging.info('Dry run - not really posting')
            return

        response = requests.post(
            urlparse.urljoin(self.Parameters.yasm_api_endpoint, '/srvambry/menutree/replace'),
            json={
                'path': subtree_path,
                'tree': tree,
            }
        )
        response.raise_for_status()

    def _generate_ext_services_alerts(self, ctype, services):
        alerts = []
        config = self._get_ext_services_alerts_config(services)
        for tag_name, tags in self._get_tags(ctype).items():
            for service in services:
                for alert_type, signal_func in self._get_alerts().items():
                    config_item = config[service.name][alert_type]
                    alert = {
                        'name': self.ALERT_NAME_TEMPLATE.format(
                            prefix=self.Parameters.ext_services_alert_prefix,
                            itype=tags['itype'][0],
                            ctype=ctype,
                            service=service.name,
                            alert_type=alert_type,
                        ),
                        'description': 'Generated by sandbox task {}'.format(lb.task_link(self.id, plain=True)),
                        'signal': signal_func(
                            service=service.name,
                            **config_item['signal_kwargs']
                        ),
                        'tags': tags,
                        'crit': config_item['thresholds'].crit_range,
                        'warn': config_item['thresholds'].warn_range,
                        'mgroups': ['ASEARCH'],
                        'abc': self.Parameters.abc_slug,
                    }
                    if config_item.get('value_modify') is not None:
                        alert['value_modify'] = config_item['value_modify']
                    if self.Parameters.juggler_namespace:
                        alert['juggler_check'] = {
                            'host': '{}-{}-ext-services-{}'.format(self.FailPercentParams.SIGNAL_PREFIX, tag_name, ctype),
                            'namespace': self.Parameters.juggler_namespace,
                            'service': '{service}-{alert_type}'.format(
                                service=service.name,
                                alert_type=alert_type,
                            ),
                            'tags': self.Parameters.juggler_tags.get(ctype) or [],
                            'flaps': {
                                'critical': 120,
                                'boost': 0,
                                'stable': 30
                            },
                            'aggregator_kwargs': {
                                'nodata_mode': 'force_ok',
                            },
                        }
                    alerts.append(alert)
        logging.info(
            'Going to replace alerts on prefix %s\n%s',
            self.Parameters.ext_services_alert_prefix,
            json.dumps(alerts, indent=4),
        )
        if self.Parameters.dry_run:
            logging.info('Dry run - not really replacing')
            return

        response = requests.post(
            urlparse.urljoin(self.Parameters.yasm_api_endpoint, '/srvambry/alerts/replace'),
            json={
                'prefix': self.Parameters.ext_services_alert_prefix,
                'alerts': alerts,
            }
        )
        logging.info(
            json.dumps(response.json(), indent=4)
        )
        response.raise_for_status()

    def _generate_ext_services(self, template_key, ctype):
        services = self._get_ext_services()
        self._generate_ext_services_menu(template_key, ctype, list(services))
        self._generate_ext_services_alerts(ctype, list(services))

    def on_save(self):
        if self.Parameters.use_last_binary:
            self.Requirements.tasks_resource = sdk2.service_resources.SandboxTasksBinary.find(
                attrs={'task_cls': self.__class__.__name__}
            ).first().id

    def on_execute(self):
        key, content = self._get_panel_template()
        self._update_panel_templates(key, content)
        for ctype in self.Parameters.ctypes:
            self._generate_ext_services(key, ctype)
