# coding: utf-8

import re
import os
import json
import logging
import requests

from six.moves.urllib import parse as urlparse

from sandbox.common.types.client import Tag
from sandbox.common.types.task import Status
from sandbox.common import rest
from sandbox.projects.common import utils
from sandbox.projects.common import error_handlers as eh
from sandbox.projects.common.nanny import nanny
from sandbox.projects.common.constants import constants
from sandbox.projects.common.build import parameters as build_params
from sandbox.projects.common.build.YaMake import YaMakeTask
from sandbox.sandboxsdk import svn
from sandbox.sandboxsdk import parameters as sp
from sandbox.sandboxsdk import process
import sandbox.sandboxsdk.task as sdk_task
import sandbox.common.types.misc as ctm

ARC_GENERATOR_PATH = 'gencfg/custom_generators/balancer_gencfg'


class ReleaseStatus(sp.SandboxRadioParameter):
    choices = [(_, _) for _ in ("none", "prestable", "stable", "testing", "unstable")]
    description = '"stable" Will be immediately enqueued to production deploy, so use with care'
    default_value = "none"
    name = 'release_status'


class ReleaseUniqDataOnly(sp.SandboxBoolParameter):
    name = 'release_uniq_data'
    description = "Release only if last released resource has different md5"
    default_value = True


class SkipTest(sp.SandboxBoolParameter):
    name = 'skip_test'
    description = "Don't test configs. I DO KNOW what I'm doing"
    default_value = False


class CheckTestsStatus(sp.SandboxBoolParameter):
    name = 'check_tests_status'
    description = 'Check tests status (if not skipped)'
    default = True
    required = False


class CommitMessage(sp.SandboxStringParameter):
    name = 'commit_message'
    description = 'Commit message'
    default_value = ''


class CommitAuthor(sp.SandboxStringParameter):
    name = 'commit_author'
    description = 'Commit author'
    default_value = ''


class FixedPatchMode(sp.SandboxBoolParameter):
    name = 'fixed_patch_mode'
    description = 'Fixed patch mode, see MINOTAUR-1414'
    default_value = True


YAMAKE_TASK_ID = 'yamake_task_id'


class UtilsYaMakeTaskId(sp.TaskSelector):
    name = YAMAKE_TASK_ID
    description = 'Prebuilt utils and configs'


FAILED_STATUSES = tuple(Status.Group.BREAK) + (Status.FAILURE, Status.DELETED)
TEST_TASK_ID = 'test_task_id'


class BaseTask(nanny.ReleaseToNannyTask, sdk_task.SandboxTask):
    """Generate, test and release tarball with balancer configs"""

    type = None
    execution_space = 888
    required_ram = 8192
    dns = ctm.DnsType.DNS64
    client_tags = Tag.Group.LINUX & ~Tag.LINUX_LUCID
    input_parameters = [
        ReleaseStatus,
        ReleaseUniqDataOnly,
        SkipTest,
        CheckTestsStatus,
        CommitMessage,
        CommitAuthor,
        build_params.ArcadiaUrl,
        build_params.ArcadiaPatch,
        UtilsYaMakeTaskId,
        FixedPatchMode,
    ]

    test_ctx = {
        'use_parallel': False,
    }

    resource_for_autorelease = None
    generator_root = None

    base_description_text = "Build and test balancer's configs in task"

    def on_release(self, additional_parameters):
        pattern = re.compile('[A-Z]+-[0-9]+')
        tickets_ids = list(set(pattern.findall(self.ctx['commit_message'])))
        st_api = 'http://st-api.yandex-team.ru'
        handler = '/v2/issues/%s/comments'
        token = self.get_vault_data(self.ctx['vault_owner'], self.ctx['vault_name'])
        request_headers = {
            "Authorization": token,
            "Content-Type": "application/json",
        }

        for tickets_id in tickets_ids:
            session = requests.Session()
            session.headers = request_headers

            text = "{0} (({1}/{2} {2})):\n\n  {3}/{4} {4})) by {5}@:\n    //{6}//".format(
                self.base_description_text,
                'https://sandbox.yandex-team.ru/task',
                self.id,
                'Commit ((https://a.yandex-team.ru/commit',
                self.ctx['commit_revision'],
                self.ctx['commit_author'],
                self.ctx['commit_message'].encode('utf-8')
            )
            text = json.dumps({'text': text})

            try:
                session.post(urlparse.urljoin(st_api, handler % tickets_id), data=text)

            except requests.RequestException as err:
                logging.warn("Request to ST ended with error: %s" % err)

        nanny.ReleaseToNannyTask.on_release(self, additional_parameters)
        # Set ttl for released resources to 183 days
        self.mark_released_resources(self.ctx['release_status'], ttl=183)

    def run_build(self):
        pass

    def on_execute(self):
        # MINOTAUR-817
        if not utils.check_all_subtasks_done():
            utils.restart_broken_subtasks()

        arcadia_url = self.ctx[constants.ARCADIA_URL_KEY]
        patch = self.ctx.get(constants.ARCADIA_PATCH_KEY)

        if not self.ctx.get(YAMAKE_TASK_ID):
            yamake_task = self.create_subtask(
                task_type=YaMakeTask.type,
                description='Build {}/utils'.format(ARC_GENERATOR_PATH),
                input_parameters={
                    'targets': '{}/utils'.format(ARC_GENERATOR_PATH),
                    constants.BUILD_SYSTEM_KEY: constants.SEMI_DISTBUILD_BUILD_SYSTEM,
                    constants.USE_AAPI_FUSE: True,   # for MINOTAUR-1126 (experimental, can break something)
                    constants.BUILD_TYPE_KEY: constants.RELEASE_BUILD_TYPE,
                    constants.STRIP_BINARIES: True,  # default is False
                    constants.ARCADIA_URL_KEY: arcadia_url,
                    constants.ARCADIA_PATCH_KEY: patch,
                    'arts': '{}/utils/utils'.format(ARC_GENERATOR_PATH),
                    # auto mode will work with patch also (fix after r3643940)
                    constants.CHECKOUT_MODE: constants.CHECKOUT_MODE_AUTO,
                },
            )
            self.ctx[YAMAKE_TASK_ID] = yamake_task.id
            utils.wait_all_subtasks_stop()

        # MINOTAUR-1449
        utils.check_subtasks_fails()

        # log generator build time for audit purposes
        try:
            self.ctx["generator_build_time"] = 0  # default
            generator_task_data = rest.Client().task[self.ctx[YAMAKE_TASK_ID]].read()
            self.ctx["generator_build_time"] = generator_task_data["intervals"]["execute"][0]["duration"]
        except Exception as exc:
            eh.log_exception("Cannot get generator build time", exc)

        # Okay, we have pre-built generator as for now. Let's generate some config files
        # TODO(mvel): rename self.generator_root to something else, it is NOT AN GENERATOR ROOT at all!
        # Please don't forget derivative tasks when renaming.

        if utils.get_or_default(self.ctx, FixedPatchMode):
            # experimental code for MINOTAUR-1414
            self.generator_root = os.path.join(self.abs_path(), ARC_GENERATOR_PATH)
            checkout_url = svn.Arcadia.append(arcadia_url, ARC_GENERATOR_PATH)
            svn.Arcadia.checkout(
                url=checkout_url,
                path=self.generator_root,
            )
            logging.info("Checked path: %s, generator_root: %s", checkout_url, self.generator_root)

            if patch:
                svn.Arcadia.apply_patch(self.abs_path(), patch, self.abs_path())
        else:
            self.generator_root = svn.Arcadia.get_arcadia_src_dir(
                svn.Arcadia.append(arcadia_url, ARC_GENERATOR_PATH)
            )
            logging.info("Arcadia cache root: %s", self.generator_root)

            if patch:
                svn.Arcadia.apply_patch(self.generator_root, patch, self.abs_path())

        with self.memoize_stage.build:
            process.run_process(
                ['bash', './install.sh'],
                work_dir=self.generator_root,
                log_prefix='install'
            )
            self.run_build()

        with self.memoize_stage.release:
            if self.ctx['release_status'] != 'none':
                check_already_released = False
                if self.ctx['release_uniq_data']:
                    check_already_released = self.resource_for_autorelease

                releaser = self.create_subtask(
                    task_type='RELEASE_ANY',
                    inherit_notifications=True,
                    input_parameters={
                        'check_already_released': check_already_released,
                        'release_task_id': self.id,
                        'release_status': self.ctx['release_status'],
                    },
                    description='balancer configs (task id: {}) autorelease'.format(str(self.id))
                )

                self.info = "Subtask {} runned, waiting it's decision about release.\n\n".format(releaser.id)
