# -*- coding: utf-8 -*-
import os
import multiprocessing

from sandbox import common

from sandbox.sandboxsdk import paths
from sandbox.sandboxsdk.parameters import LastReleasedResource
from sandbox.sandboxsdk.parameters import ResourceSelector
from sandbox.sandboxsdk.parameters import SandboxArcadiaUrlParameter
from sandbox.sandboxsdk.parameters import SandboxStringParameter
from sandbox.sandboxsdk.parameters import SandboxBoolParameter
from sandbox.sandboxsdk.parameters import SandboxRadioParameter
from sandbox.sandboxsdk.process import run_process
from sandbox.sandboxsdk.environments import VirtualEnvironment
from sandbox.common.types.client import Tag
from sandbox.projects.balancer import resources as balancer_resources
from sandbox.projects.common.build.YaMake import YaMakeTask
from sandbox.projects.common import apihelpers
from sandbox.projects.common.search import gdb
from sandbox.projects.common import utils
import sandbox.projects.common.build.parameters as build_params
import sandbox.projects.common.constants as consts
import sandbox.sandboxsdk.svn as svn


RUNTIME_TESTS = 'balancer/test'
ARCADIA_TRUNK_URL = 'svn+ssh://arcadia.yandex.ru/arc/trunk/arcadia'


class ArcadiaUrl(SandboxArcadiaUrlParameter):
    name = consts.ARCADIA_URL_KEY
    description = 'Svn url for arcadia'
    default_value = svn.Arcadia.trunk_url()
    required = True


class BalancerBuildType(build_params.BuildType):
    default_value = consts.RELEASE_BUILD_TYPE


class BalancerSanitize(build_params.Sanitize):
    default_value = None


class BalancerBuildDeps(SandboxBoolParameter):
    name = 'build_deps'
    description = 'Build test dependencies'
    default_value = False


class AppendGdb(SandboxBoolParameter):
    # MINOTAUR-1259
    name = 'append_gdb'
    description = 'Append GDB toolkit to release'
    default_value = False


class BalancerExecutable(LastReleasedResource):
    name = 'executable_resource_id'
    description = 'Executable(balancer ver)'
    resource_type = [
        balancer_resources.BALANCER_EXECUTABLE,
        balancer_resources.BALANCER_IMPROXY_EXECUTABLE,
    ]

    @common.utils.classproperty
    def default_value(cls):
        balancer = apihelpers.get_last_released_resource(balancer_resources.BALANCER_EXECUTABLE, arch='linux')
        return balancer.id if balancer else None


class BalancerTestTools(ResourceSelector):
    name = 'balancer_test_tools_id'
    description = 'balancer test tools resource'
    resource_type = [balancer_resources.BALANCER_TEST_TOOLS]
    required = False

    @common.utils.classproperty
    def default_value(cls):
        resource = apihelpers.get_last_resource(cls.resource_type[0])
        return resource.id if resource else None


class BalancerTestFilters(SandboxStringParameter):
    name = consts.TEST_FILTERS
    description = 'Filter tests expression'
    default_value = ''


class TestTool(SandboxRadioParameter):
    name = 'test_tool'
    description = 'Test tool'
    default_value = 'ya_make'
    choices = [
        ('ya make', 'ya_make'),
        ('py.test', 'pytest'),
    ]


class ParallelTests(SandboxBoolParameter):
    name = 'use_parallel'
    description = 'Use paralleling'
    default_value = True


class ClearBuild(SandboxBoolParameter):
    name = consts.CLEAR_BUILD_KEY
    description = 'Clear build'
    default_value = True


class CheckReturnCode(SandboxBoolParameter):
    name = 'check_tests_return_code'
    description = 'Check return code'
    default_value = False


class UseSystemPython(SandboxBoolParameter):
    name = consts.USE_SYSTEM_PYTHON
    description = 'Use system Python to build python libraries'
    default_value = False


class BaseBalancerTask(YaMakeTask):
    input_parameters = [
        ArcadiaUrl,
        BalancerBuildType,
        BalancerSanitize,
        AppendGdb,
    ] + build_params.get_aapi_parameters()

    type = 'BASE_BALANCER_TASK'

    @property
    def arcadia_url(self):
        return self.ctx[ArcadiaUrl.name]

    def is_trunk(self):
        return self.arcadia_url.startswith(svn.Arcadia.trunk_url()) or self.arcadia_url.startswith(ARCADIA_TRUNK_URL)

    def pre_execute(self):
        if utils.get_or_default(self.ctx, AppendGdb):
            gdb.append_to_release(self)
        if not self.is_trunk():
            self.ctx[consts.CHECKOUT] = not utils.get_or_default(self.ctx, build_params.UseArcadiaApiFuse)
        YaMakeTask.pre_execute(self)


class BaseTestBalancerTask(BaseBalancerTask):
    input_parameters = BaseBalancerTask.input_parameters + [
        BalancerBuildDeps,
        BalancerExecutable,
        BalancerTestTools,
        BalancerTestFilters,
        TestTool,
        ParallelTests,
        ClearBuild,
        CheckReturnCode,
        UseSystemPython,
    ]

    type = 'BASE_TEST_BALANCER_TASK'

    client_tags = Tag.Group.LINUX & ~Tag.LINUX_LUCID

    def get_targets(self):
        raise NotImplementedError()

    def test_params(self):
        return []

    def pytest_pip_deps(self):
        return []

    def sync_res(self, res_cls):
        res_id = self.ctx[res_cls.name]
        if res_id:
            return self.sync_resource(res_id)

    def __get_test_tools(self):
        test_tools = self.sync_res(BalancerTestTools)

        if not test_tools:
            return None

        if os.path.isdir(test_tools):
            return test_tools

        # extract tarball
        test_tools_dir = self.get_path('test_tools_dir')
        if not os.path.exists(test_tools_dir):
            os.makedirs(test_tools_dir)
        run_process([
            'tar',
            '-xvzf', test_tools,
            '-C', test_tools_dir,
            '--strip-components', '3',  # strip balancer/test/tools prefix
        ], log_prefix='tar', check=True)
        return test_tools_dir

    def __common_test_params(self):
        result = [
            ('balancer', self.sync_res(BalancerExecutable)),
        ]
        test_tools = self.__get_test_tools()
        if test_tools:
            result.append(('test_tools_dir', test_tools))
        return result

    def __joined_params(self):
        return self.__common_test_params() + self.test_params()

    def __ya_make_test_params(self):
        return ' '.join(['{}={}'.format(name, value) for name, value in self.__joined_params()])

    @property
    def test_threads(self):
        use_parallel = self.ctx[ParallelTests.name]
        if use_parallel:
            return multiprocessing.cpu_count()
        else:
            return 1

    @property
    def test_filters(self):
        result = self.ctx[BalancerTestFilters.name]
        if result:
            return result
        else:
            return None

    @property
    def def_flags(self):
        if not self.ctx[BalancerBuildDeps.name]:
            return '-DBUILD_DEPS=no'
        else:
            return None

    @property
    def __ya_make_ctx(self):
        result = {
            consts.TESTS_REQUESTED: True,
            consts.TEST_PARAMS_KEY: self.__ya_make_test_params(),
            consts.TEST_THREADS: self.test_threads,
            consts.CHECK_RETURN_CODE: self.check_rc,
        }
        if self.def_flags:
            result[consts.DEFINITION_FLAGS_KEY] = self.def_flags
        return result

    def pre_execute(self):
        self.ctx.update(self.__ya_make_ctx)
        BaseBalancerTask.pre_execute(self)

    def get_path(self, name):
        return os.path.join(paths.get_logs_folder(), name)

    @property
    def junit_report_path(self):
        return self.get_path('junit_report.xml')

    @property
    def output_dir(self):
        return self.get_path('logs')

    @property
    def clear_build(self):
        return self.ctx[ClearBuild.name]

    @property
    def check_rc(self):
        return self.ctx[CheckReturnCode.name]

    def get_project_url(self, name):
        if '@' in self.arcadia_url:
            arcadia_url, rev = self.arcadia_url.split('@')
        else:
            arcadia_url = self.arcadia_url
            rev = None
        result = arcadia_url + '/' + name
        if rev is not None:
            result += '@' + rev
        return result

    def __get_tests(self):
        tests_path = self.abs_path('test')
        tests_arcadia_url = self.get_project_url(RUNTIME_TESTS)
        svn.Arcadia.export(tests_arcadia_url, tests_path)
        return tests_path

    def pytest_execute(self):
        targets = self.get_targets()
        targets = [trg[len(RUNTIME_TESTS) + 1:] for trg in targets if trg.startswith(RUNTIME_TESTS)]

        tests_root = self.__get_tests()
        tests_paths = [os.path.join(tests_root, trg) for trg in targets]

        requirements_paths = [os.path.join(path, 'requirements.txt') for path in tests_paths]

        cmd = [
            'py.test',
            '--junit-xml={}'.format(self.junit_report_path),
            '--log_dir={}'.format(self.output_dir),
            '-n={}'.format(self.test_threads),
            '--sandbox',
        ]
        for name, value in self.__joined_params():
            cmd.append('--{}={}'.format(name, value))
        if self.test_filters:
            cmd.append('-k={}'.format(self.test_filters))
        cmd.extend(tests_paths)

        with VirtualEnvironment(use_system=True) as venv:
            for req_path in requirements_paths:
                venv.pip('-r {}'.format(req_path))
            for dep in self.pytest_pip_deps():
                venv.pip(dep)
            run_process(cmd, log_prefix='run_test', check=self.check_rc)

    def on_execute(self):
        test_tool = self.ctx[TestTool.name]
        if test_tool == 'ya_make':
            YaMakeTask.on_execute(self)
        else:
            self.pytest_execute()
