# -*- coding: utf8 -*-

import logging
import os
import sandbox.sandboxsdk.svn as sdk_svn
import sandbox.sandboxsdk.task as sdk_task
import sandbox.sandboxsdk.process as sdk_process
import subprocess
import time

import sandbox.common.types.client as ctc

from sandbox.projects.common.nanny import nanny
from sandbox.sandboxsdk.parameters import SandboxBoolParameter


class ForceReleaseParameter(SandboxBoolParameter):
    """
        Release OCSP as soon as possible
    """
    name = 'force_release'
    description = 'Release OCSP ASAP'
    default_value = False


class FetchOcspResponse(nanny.ReleaseToNannyTask, sdk_task.SandboxTask):
    """
    Generate ocsp response files for search balancers (base class)
    """
    execution_space = 1024
    required_ram = 1024
    client_tags = ctc.Tag.LINUX_PRECISE & ctc.Tag.IPV4

    input_parameters = [
        ForceReleaseParameter
    ]

    def _release_hook(self):
        releaser = self.create_subtask(
            task_type='RELEASE_ANY',
            inherit_notifications=True,
            input_parameters={
                'check_already_released': str(self._get_ocsp_resource_type()),
                'release_task_id': self.id,
                'release_status': 'stable'
            },
            description='ocsp_responses.tgz (task id: {}) autorelease'.format(str(self.id))
        )

        self.info = "Subtask {} runned, waiting it's decision about release.\n\n".format(releaser.id)

    def on_release(self, additional_parameters):
        nanny.ReleaseToNannyTask.on_release(self, additional_parameters)
        self.mark_released_resources('stable', ttl=10)

    def on_execute(self):
        with self.memoize_stage.test:
            certs_dir = sdk_svn.Arcadia.get_arcadia_src_dir(
                self.ctx[self._get_svn_cert_url()]
            )
            current_time = int(time.time())
            os.mkdir(os.path.join(os.path.abspath(''), 'ocsp'))

            counter = 0
            for cert in os.listdir(certs_dir):
                if cert.endswith('.pem') and cert.startswith('allCAs-'):
                    ocsp_uri_proc = sdk_process.run_process(
                        [
                            'openssl', 'x509', '-noout', '-ocsp_uri',
                            '-in', '{}/{}'.format(certs_dir, cert)
                        ],
                        work_dir=certs_dir,
                        stdout=subprocess.PIPE,
                        log_prefix='ocsp_uri',
                    )
                    ocsp_uri = ocsp_uri_proc.stdout.read().rstrip()

                    if not ocsp_uri:
                        logging.warn('WANRNING: There is no ocsp uri for {}'.format(cert))
                        continue

                    issuer_proc = sdk_process.run_process(
                        [
                            'openssl', 'x509', '-noout', '-issuer',
                            '-in', '{}/{}'.format(certs_dir, cert)
                        ],
                        work_dir=certs_dir,
                        stdout=subprocess.PIPE,
                        log_prefix='issuer'
                    )
                    issuer = issuer_proc.stdout.read().rstrip()

                    if issuer == 'issuer= /C=FR/O=KEYNECTIS/CN=CLASS 2 KEYNECTIS CA':
                        CA_cert = '{}/root/CA-Certplus.pem'.format(certs_dir)
                        no_nonce = False
                        header = False

                    elif issuer == 'issuer= /C=PL/O=Unizeto Technologies S.A./OU=Certum Certification Authority/CN=Certum Level IV CA':
                        CA_cert = '{}/root/CA-Unizeto.pem'.format(certs_dir)
                        no_nonce = '-no_nonce'
                        header = False

                    elif issuer == 'issuer= /C=RU/O=Yandex LLC/OU=Yandex Certification Authority/CN=Yandex CA':
                        CA_cert = '{}/root/CA-Yandex.pem'.format(certs_dir)
                        no_nonce = '-no_nonce'
                        header = False

                    elif issuer == 'issuer= /C=BE/O=GlobalSign nv-sa/CN=GlobalSign Organization Validation CA - SHA256 - G2':
                        CA_cert = '{}/root/CA-GlobalSign.pem'.format(certs_dir)
                        no_nonce = ''
                        header = ['-header', 'HOST', 'ocsp2.globalsign.com']

                    elif issuer == 'issuer= /C=BE/O=GlobalSign nv-sa/CN=GlobalSign Organization Validation CA - G2':
                        CA_cert = '{}/root/CA-GlobalSignSha1.pem'.format(certs_dir)
                        no_nonce = False
                        header = ['-header', 'HOST', 'ocsp2.globalsign.com']

                    else:
                        logging.warn('WANRNING: There is no CAfile for {}'.format(cert))
                        continue

                    tmp_der_file = '{}_temporary.der'.format(cert.split('.pem')[0])

                    ocsp_generate_args = [
                        'openssl', 'ocsp',
                        '-CAfile', CA_cert,
                        '-issuer', CA_cert,
                        '-cert', '{}/{}'.format(certs_dir, cert),
                        '-url', ocsp_uri,
                        '-respout', os.path.join(os.path.abspath(''), tmp_der_file)
                    ]
                    if no_nonce:
                        ocsp_generate_args.append(no_nonce)
                    if header:
                        ocsp_generate_args.extend(header)

                    try:
                        sdk_process.run_process(
                            ocsp_generate_args,
                            work_dir=certs_dir,
                            log_prefix='generate'
                        )

                    except Exception as err:
                        logging.critical(
                            'ERROR: unexpected error due receiving response for {}: {}'.format(
                                cert, err
                            )
                        )
                        continue

                    ocsp_text_proc = sdk_process.run_process(
                        [
                            'openssl', 'ocsp',
                            '-respin', os.path.join(os.path.abspath(''), tmp_der_file),
                            '-noverify', '-text'
                        ],
                        work_dir=certs_dir,
                        stdout=subprocess.PIPE,
                        log_prefix='verify'
                    )

                    for line in ocsp_text_proc.stdout:
                        if 'This Update' in line:
                            date = line.rstrip().split('Update:')[1]
                            this_update = int(time.mktime(time.strptime(date, ' %b %d %H:%M:%S %Y %Z')))
                        if 'Next Update' in line:
                            date = line.rstrip().split('Update:')[1]
                            next_update = int(time.mktime(time.strptime(date, ' %b %d %H:%M:%S %Y %Z')))

                    if this_update < current_time < next_update:
                        os.rename(
                            os.path.join(os.path.abspath(''), tmp_der_file),
                            os.path.join(os.path.abspath(''), 'ocsp', '{}.der'.format(cert.split('.pem')[0]))
                        )
                        counter += 1
                    else:
                        logging.critical(
                            'ERR: Status times invalid in {}. This Update: {}; Next Update: {}'.format(
                                tmp_der_file,
                                this_update,
                                next_update
                            )
                        )
                        os.remove(os.path.join(os.path.abspath(''), tmp_der_file))
                        continue

            if counter == 0:
                raise Exception('There are no valid ocsp files at all')

            sdk_process.run_process(
                [
                    'tar', '-czf',
                    os.path.join(os.path.abspath(''), 'ocsp_responses.tgz'),
                    'ocsp'
                ],
                work_dir=os.path.abspath(''),
                log_prefix='tar_files'
            )

            ocsp_responses = self.create_resource(
                arch='any',
                attributes={'ttl': 1},
                description=self.descr,
                resource_path='ocsp_responses.tgz',
                resource_type=self._get_ocsp_resource_type(),
            )
            self.mark_resource_ready(ocsp_responses)

            if not self.ctx[ForceReleaseParameter.name]:
                self.wait_time(43200)

        self._release_hook()


__Task__ = FetchOcspResponse
