import os
import six

from sandbox.projects.common.base_search_quality.tree import htmldiff
from sandbox.projects.common.search import queries as sq


def GetFactorName(FACTOR_NAMES, index):
    if index < len(FACTOR_NAMES):
        return FACTOR_NAMES[index]

    return "UNKNOWN FACTOR"


class DiffForSingleUrl:
    def __init__(self):
        self.DifferentRelevance = []
        self.DifferentFactorValues = {}

    def __eq__(self, other):
        return (
            self.DifferentRelevance == other.DifferentRelevance
            and self.DifferentFactorValues == other.DifferentFactorValues
        )

    def __repr__(self):
        return (
            '<DiffForSingleUrl\n    DifferentRelevance:\n    %s\n    DifferentFactorValues:\n    %s>' % (
                self.DifferentRelevance, self.DifferentFactorValues
            )
        )

    def PrintDifferences(self, url, FACTOR_NAMES, file):
        file.write("<table border=1>\r\n")

        if len(self.DifferentRelevance) > 0:
            file.write(
                "<tr><td width=150>relevance</td><td width=100 style='background-color: %s'>%s"
                "</td><td width=100 style='background-color: %s'>%s</td></tr>" % (
                    htmldiff.COLOR_REMOVED, self.DifferentRelevance[0],
                    htmldiff.COLOR_ADDED, self.DifferentRelevance[1]))

        for factorNumber, oldAndNewFactorValues in self.DifferentFactorValues.iteritems():
            file.write(
                "<tr><td width=150>%s</td><td width=100 style='background-color: %s'>%s"
                "</td><td width=100 style='background-color: %s'>%s</td></tr>" % (
                    GetFactorName(FACTOR_NAMES, factorNumber), htmldiff.COLOR_REMOVED,
                    oldAndNewFactorValues[0], htmldiff.COLOR_ADDED, oldAndNewFactorValues[1]))

        file.write("</table>\r\n")


class DiffsForSingleQuery:
    def __init__(self):
        self.Dolbilka1NotFoundUrls = set()
        self.Dolbilka2NotFoundUrls = set()

        self.DiffsForUrls = {}

    def __eq__(self, other):
        return (
            self.Dolbilka1NotFoundUrls == other.Dolbilka1NotFoundUrls and
            self.Dolbilka2NotFoundUrls == other.Dolbilka2NotFoundUrls and
            self.DiffsForUrls == other.DiffsForUrls
        )

    def __repr__(self):
        return (
            "<DiffsForSingleQuery\n"
            "  Dolbilka1NotFoundUrls:\n"
            "  %s\n"
            "  Dolbilka2NotFoundUrls:\n"
            "  %s\n"
            "  DiffsForUrls:\n"
            "  %s>" % (
                self.Dolbilka1NotFoundUrls,
                self.Dolbilka2NotFoundUrls,
                self.DiffsForUrls
            )
        )

    def HasFoundOrLostDocuments(self):
        return len(self.Dolbilka1NotFoundUrls) > 0 or len(self.Dolbilka2NotFoundUrls) > 0

    def PrintDifferences(self, queryNumber, FACTOR_NAMES, printDifferentFactors, file):
        if len(self.Dolbilka2NotFoundUrls) > 0:
            file.write(htmldiff.StartBlock("disappeared documents", htmldiff.COLOR_REMOVED))

            for url in self.Dolbilka2NotFoundUrls:
                htmldiff.WriteData(file, url)

            file.write(htmldiff.EndBlock())

        if len(self.Dolbilka1NotFoundUrls) > 0:
            file.write(htmldiff.StartBlock("new documents", htmldiff.COLOR_ADDED))

            for url in self.Dolbilka1NotFoundUrls:
                htmldiff.WriteData(file, url)

            file.write(htmldiff.EndBlock())

        if printDifferentFactors:
            for url, diffForSingleUrl in self.DiffsForUrls.iteritems():
                file.write(htmldiff.StartBlock(url))
                diffForSingleUrl.PrintDifferences(url, FACTOR_NAMES, file)
                file.write(htmldiff.EndBlock())


def _FormatQueryToOutput(query):
    cgiParams = sq.parse_cgi_params(query)

    try:
        query += "\r\nuser request: %s" % (cgiParams["user_request"][0])
    except Exception:
        pass

    try:
        query += "\r\ntext: %s" % (cgiParams["text"][0])
    except Exception:
        pass

    return query


class DiffsBetweenDolbilkas:
    def __init__(self):
        self.Dolbilka1FoundNothingForQueries = set()
        self.Dolbilka2FoundNothingForQueries = set()

        self.DiffsForQueries = {}

        self.DolbilkaOutput1 = set()
        self.DolbilkaOutput2 = set()

        self.TotalNonEmptyResponseQueries = 0
        self.TotalDocsCompared = 0

    def __eq__(self, other):
        return (
            self.Dolbilka1FoundNothingForQueries == other.Dolbilka1FoundNothingForQueries and
            self.Dolbilka2FoundNothingForQueries == other.Dolbilka2FoundNothingForQueries and
            self.DiffsForQueries == other.DiffsForQueries
        )

    def __repr__(self):
        return (
            "<DiffsBetweenDolbilkas\n"
            "Dolbilka1FoundNothingForQueries:\n"
            "%s\n"
            "Dolbilka2FoundNothingForQueries:\n"
            "%s\n"
            "DiffsForQueries:\n%s>" % (
                self.Dolbilka1FoundNothingForQueries,
                self.Dolbilka2FoundNothingForQueries,
                self.DiffsForQueries
            )
        )

    def _HasNewEmptyResponses(self):
        return len(self.Dolbilka1FoundNothingForQueries) > 0 or len(self.Dolbilka2FoundNothingForQueries) > 0

    def HasDifferences(self):
        return self._HasNewEmptyResponses() or len(self.DiffsForQueries) > 0

    def HasNewFoundOrLostDocuments(self):
        if self._HasNewEmptyResponses():
            return True
        for diffForSingleQuery in self.DiffsForQueries.values():
            if diffForSingleQuery.HasFoundOrLostDocuments():
                return True
        return False

    def PrintDifferences(self, USER_REQUESTS, FACTOR_NAMES, printDifferentFactors, file):
        if len(self.Dolbilka2FoundNothingForQueries) > 0:
            file.write(htmldiff.StartBlock("disappeared answers for requests:", htmldiff.COLOR_REMOVED))

            for req in self.Dolbilka2FoundNothingForQueries:
                file.write(htmldiff.StartBlock("query # " + str(req)))

                htmldiff.WriteDataBlock(file, "query string", _FormatQueryToOutput(USER_REQUESTS[req]))

                for url in self.DolbilkaOutput1[req]:
                    htmldiff.WriteData(file, url)

                file.write(htmldiff.EndBlock())

            file.write(htmldiff.EndBlock())

        if len(self.Dolbilka1FoundNothingForQueries) > 0:
            file.write(htmldiff.StartBlock("new answers for requests:", htmldiff.COLOR_ADDED))

            for req in self.Dolbilka1FoundNothingForQueries:
                file.write(htmldiff.StartBlock("query # " + str(req)))

                htmldiff.WriteDataBlock(file, "query string", _FormatQueryToOutput(USER_REQUESTS[req]))

                for url in self.DolbilkaOutput2[req]:
                    htmldiff.WriteData(file, url)

                file.write(htmldiff.EndBlock())

            file.write(htmldiff.EndBlock())

        for queryNumber in sorted(self.DiffsForQueries.keys()):
            diffForSingleQuery = self.DiffsForQueries[queryNumber]

            if (
                printDifferentFactors
                or len(diffForSingleQuery.Dolbilka1NotFoundUrls) > 0
                or len(diffForSingleQuery.Dolbilka2NotFoundUrls) > 0
            ):

                file.write(htmldiff.StartBlock("query # " + str(queryNumber), htmldiff.COLOR_CHANGED))

                if queryNumber >= len(USER_REQUESTS):
                    raise Exception((queryNumber, len(USER_REQUESTS)))
                htmldiff.WriteDataBlock(file, "query string", _FormatQueryToOutput(USER_REQUESTS[queryNumber]))

                diffForSingleQuery.PrintDifferences(queryNumber, FACTOR_NAMES, printDifferentFactors, file)

                file.write(htmldiff.EndBlock())

    def GetDifferentFactors(self):
        differentFactors = {}

        for queryNumber, diffsForSingleQuery in self.DiffsForQueries.iteritems():
            for url, diffsForSingleUrl in diffsForSingleQuery.DiffsForUrls.iteritems():
                for factorNumber, oldAndNewFactorValues in diffsForSingleUrl.DifferentFactorValues.iteritems():
                    currentDiff = abs(oldAndNewFactorValues[0] - oldAndNewFactorValues[1])

                    try:
                        changedFactor = differentFactors[factorNumber]

                        changedFactor.Count = changedFactor.Count + 1

                        maxDiff = abs(changedFactor.Value1 - changedFactor.Value2)

                        if currentDiff > maxDiff:
                            changedFactor.Value1 = oldAndNewFactorValues[0]
                            changedFactor.Value2 = oldAndNewFactorValues[1]

                        changedFactor.AbsDiffsSum = changedFactor.AbsDiffsSum + currentDiff

                    except KeyError:
                        changedFactor = ChangedFactor(
                            1, oldAndNewFactorValues[0], oldAndNewFactorValues[1], currentDiff
                        )
                        differentFactors[factorNumber] = changedFactor

                    changedFactor.queries.setdefault(queryNumber, {url: oldAndNewFactorValues})

        return differentFactors


def _CompareResultsForSingleUrl(url, foundUrls1, foundUrls2):
    diff = DiffForSingleUrl()

    isDifferent = False

    url1 = foundUrls1[url]
    url2 = foundUrls2[url]

    if url1.relev != url2.relev:
        diff.DifferentRelevance = [int(url1.relev), int(url2.relev)]
        isDifferent = True

    ln = min(len(url1.factors), len(url2.factors))

    for i in six.moves.xrange(ln):
        if abs(url1.factors[i] - url2.factors[i]) > 1e-8:
            diff.DifferentFactorValues[i] = [url1.factors[i], url2.factors[i]]
            isDifferent = True

    if isDifferent:
        return diff
    else:
        return None


def _CompareResultsForSingleQuery(diffs, queryNumber, foundUrls1, foundUrls2):
    urls1 = set(foundUrls1.keys())
    urls2 = set(foundUrls2.keys())

    diff1 = urls1 - urls2
    if len(diff1) > 0:
        if queryNumber not in diffs.DiffsForQueries:
            diffs.DiffsForQueries[queryNumber] = DiffsForSingleQuery()

        diffs.DiffsForQueries[queryNumber].Dolbilka2NotFoundUrls = diff1

    diff2 = urls2 - urls1
    if len(diff2) > 0:
        if queryNumber not in diffs.DiffsForQueries:
            diffs.DiffsForQueries[queryNumber] = DiffsForSingleQuery()

        diffs.DiffsForQueries[queryNumber].Dolbilka1NotFoundUrls = diff2

    for url in urls1 & urls2:
        diffForSingleUrl = _CompareResultsForSingleUrl(url, foundUrls1, foundUrls2)

        if diffForSingleUrl is not None:
            if queryNumber not in diffs.DiffsForQueries:
                diffs.DiffsForQueries[queryNumber] = DiffsForSingleQuery()

            diffs.DiffsForQueries[queryNumber].DiffsForUrls[url] = diffForSingleUrl

        diffs.TotalDocsCompared += 1


def CompareResults(dolbilkaOutput1, dolbilkaOutput2):
    diffs = DiffsBetweenDolbilkas()
    diffs.DolbilkaOutput1 = dolbilkaOutput1
    diffs.DolbilkaOutput2 = dolbilkaOutput2

    queries_numbers1 = set(dolbilkaOutput1.keys())
    queries_numbers2 = set(dolbilkaOutput2.keys())

    diffs.Dolbilka1FoundNothingForQueries = queries_numbers2 - queries_numbers1
    diffs.Dolbilka2FoundNothingForQueries = queries_numbers1 - queries_numbers2

    for queryNumber in (queries_numbers1 & queries_numbers2):
        _CompareResultsForSingleQuery(diffs, queryNumber, dolbilkaOutput1[queryNumber], dolbilkaOutput2[queryNumber])

    diffs.TotalNonEmptyResponseQueries = len(queries_numbers1) + len(diffs.Dolbilka1FoundNothingForQueries)

    return diffs


class ChangedFactor:
    def __init__(self, count, value1, value2, absDiffsSum):
        self.Count = count

        self.Value1 = value1
        self.Value2 = value2

        self.AbsDiffsSum = absDiffsSum

        # queries[queryNumber] -> urls[url] -> (value1, value2)
        self.queries = {}


def _WriteIframe(file, src):
    file.write("""<iframe src = "%s" width="100%%" height=400 scrolling=auto>"
    <p>Your browser does not support iframes.</p>
</iframe>""" % (src))


def WriteDiffs(diffs, FACTOR_NAMES, USER_REQUESTS, dir):
    differentFactors = diffs.GetDifferentFactors()
    hasNewFoundOrLostDocuments = diffs.HasNewFoundOrLostDocuments()

    def WriteAllDifferencesGroupedByQueryUrlFactor():
        file = open(os.path.join(dir, "groupByQueryUrlFactor.html"), "w")

        def WriteBody():
            diffs.PrintDifferences(USER_REQUESTS, FACTOR_NAMES, True, file)

        htmldiff.WriteDiff(file, WriteBody, addButtons=False)

        file.close()

    def WriteAllDifferencesGroupedByFactorQueryUrl():
        file = open(os.path.join(dir, "groupByFactorQueryUrl.html"), "w")

        def WriteBody():
            for factorNumber in sorted(differentFactors.keys()):
                changedFactor = differentFactors[factorNumber]

                file.write(htmldiff.StartBlock("%s %s" % (factorNumber, GetFactorName(FACTOR_NAMES, factorNumber))))

                for queryNumber, queryUrls in changedFactor.queries.iteritems():
                    file.write(htmldiff.StartBlock("query # " + str(queryNumber)))

                    htmldiff.WriteDataBlock(file, "query string", _FormatQueryToOutput(USER_REQUESTS[queryNumber]))

                    for url, oldAndNewFactorValues in queryUrls.iteritems():
                        file.write(htmldiff.StartBlock(url))

                        file.write("<table border=1>\r\n")
                        file.write(
                            "<tr><td width=100 style='background-color: %s'>%s</td>"
                            "<td width=100 style='background-color: %s'>%s</td></tr>" % (
                                htmldiff.COLOR_REMOVED, oldAndNewFactorValues[0],
                                htmldiff.COLOR_ADDED, oldAndNewFactorValues[1])
                        )
                        file.write("</table>\r\n")

                        file.write(htmldiff.EndBlock())

                    file.write(htmldiff.EndBlock())

                file.write(htmldiff.EndBlock())

        htmldiff.WriteDiff(file, WriteBody, addButtons=False)

        file.close()

    def WriteDifferentUlrs():
        file = open(os.path.join(dir, "differentUrls.html"), "w")

        def WriteBody():
            diffs.PrintDifferences(USER_REQUESTS, FACTOR_NAMES, False, file)

        htmldiff.WriteDiff(file, WriteBody, addButtons=False)

        file.close()

    def WriteIndex():
        file = open(os.path.join(dir, "index.html"), "w")

        def WriteBody():
            file.write("""<table border=1>
<tr> <td><b>total number of queries&nbsp;</b></td> <td align=right>{0}</td> </tr>
<tr> <td><b>number of queries with no documents found for&nbsp;</b></td> <td align=right>{1}</td> </tr>
<tr> <td><b>total number of found documents compared&nbsp;</td> <td align=right>{2}</td> </tr>
</table>
<br>
<table border=1>
<tr>
    <td><b>factor number</b></td>
    <td><b>factor name</b></td>
    <td><b>affected documents</b></td>
    <td colspan=2 align=center><b>max diff</b></td>
    <td align=center><b>avg diff</b></td> </tr>
""".format(len(USER_REQUESTS), len(USER_REQUESTS) - diffs.TotalNonEmptyResponseQueries, diffs.TotalDocsCompared))

            for factorNumber in sorted(differentFactors.keys()):
                changedFactor = differentFactors[factorNumber]
                file.write(
                    "<tr> <td>%s</td>"
                    "<td>%s</td>"
                    "<td align=right>%s</td>"
                    "<td align=right>%s</td>"
                    "<td align=right>%s</td>"
                    "<td align=right>%s</td>"
                    "</tr>\r\n" % (
                        factorNumber,
                        GetFactorName(FACTOR_NAMES, factorNumber),
                        changedFactor.Count, str(changedFactor.Value1),
                        str(changedFactor.Value2),
                        str(changedFactor.AbsDiffsSum / changedFactor.Count)
                    )
                )

            file.write("""</table><p>""")

            if hasNewFoundOrLostDocuments:
                file.write("<b>diff in filtration (new found or lost documents)</b><p>")

            file.write(htmldiff.StartBlock("all diffs grouped by query-&gt;url-&gt;factor", open=False))
            _WriteIframe(file, "groupByQueryUrlFactor.html")
            file.write(htmldiff.EndBlock())

            file.write(htmldiff.StartBlock("all diffs grouped by factor-&gt;query-&gt;url", open=False))
            _WriteIframe(file, "groupByFactorQueryUrl.html")
            file.write(htmldiff.EndBlock())

            file.write(htmldiff.StartBlock("affected documents only", open=False))
            _WriteIframe(file, "differentUrls.html")
            file.write(htmldiff.EndBlock())

        htmldiff.WriteDiff(file, WriteBody, addLegend=False, addButtons=False)

        file.close()

    WriteAllDifferencesGroupedByQueryUrlFactor()
    WriteAllDifferencesGroupedByFactorQueryUrl()
    WriteDifferentUlrs()
    WriteIndex()
