# -*- coding: utf-8 -*-

import six
import logging

from . import snippets_diff_representation
from .tree import htmldiff
from .tree import exception as ne
from sandbox.projects.common.differ import printers

# глобальный кортеж из двух списков факторов
# (заполняется в соответствующих тасках)
GLOBAL_FACTOR_NAMES = ([], [])


def _ensure_str(txt):
    return txt.encode('utf-8') if isinstance(txt, unicode) else str(txt)


def _doc_get_docid(node):
    try:
        ret = "BinaryDocId="
        if "Route" in node.props():
            ret += str(node.GetPropValue("Route")) + "/"
        ret += str(node.GetPropValue("DocHash"))
        return ret
    except ne.ElementNotFoundException:
        return "DocId=" + node.GetPropValue("DocId")


class GroupingNode(htmldiff.KeyNode):
    def __init__(self):
        super(GroupingNode, self).__init__("Grouping", [])

    def GetKeyValue(self, node):
        try:
            return "Attr=\"{}\" Mode={}".format(node.GetPropValue("Attr"), node.GetPropValue("Mode"))
        except ne.ElementNotFoundException:
            return "IsFlat={}".format(node.GetPropValue("IsFlat"))


class SearcherPropNode(htmldiff.KeyValueMapNode):
    def __init__(self):
        super(SearcherPropNode, self).__init__("SearcherProp", [])


class FactorMappingNode(htmldiff.KeyValueMapNode):
    def __init__(self):
        super(FactorMappingNode, self).__init__("FactorMapping", ["Head"])


class GroupNode(htmldiff.KeyNode):
    def __init__(self):
        super(GroupNode, self).__init__("Group", ["Grouping"])

    def GetKeyValue(self, node):
        category_name = node.GetPropValue("CategoryName", False)

        if category_name is not None:
            key = "CategoryName=" + category_name
        else:
            key = ""

        documents = node._nodes.get("Document", None)

        if documents and len(documents) == 1:
            key += ";" + _doc_get_docid(documents[0])

        return key


class DocumentNode(htmldiff.KeyNode):
    def __init__(self):
        super(DocumentNode, self).__init__("Document", ["Grouping", "Group"])

    def GetKeyValue(self, node):
        return _doc_get_docid(node)


def _split_values(values, separator):
    if values.endswith(separator):
        # SEARCH-2331
        values = values[:-1]

    result = values.split(separator)
    if len(result) == 1 and not result[0]:
        return []
    else:
        return result


def _write_relev_factors_diff(diff_file, query_number, property_path, values1, values2, changed_props):
    separator = ";" if property_path[-1] == "_AllFactors" else " "
    factors1 = _split_values(values1[0], separator)
    factors2 = _split_values(values2[0], separator)
    write_objects_diff(
        diff_file, query_number, property_path + ["relev_factor"],
        factors1, factors2, changed_props,
        diff_as_factors=True,
    )
    return True


class FirstStageAttributeNode(htmldiff.KeyValueMapNode):
    def __init__(self):
        super(FirstStageAttributeNode, self).__init__("FirstStageAttribute", ["Grouping", "Group", "Document"])

    def WriteDiff(self, file, queryNumber, propertyPath, values1, values2, changedProps):
        property_name = propertyPath[-1]
        if property_name in ['_RelevFactors', "_AllFactors"]:
            return _write_relev_factors_diff(file, queryNumber, propertyPath, values1, values2, changedProps)
        else:
            super(FirstStageAttributeNode, self).WriteDiff(
                file, queryNumber, propertyPath, values1, values2, changedProps
            )


class GtaRelatedAttributeNode(htmldiff.KeyValueMapNode):
    def __init__(self):
        super(GtaRelatedAttributeNode, self).__init__(
            "GtaRelatedAttribute",
            ["Grouping", "Group", "Document", "ArchiveInfo"]
        )

    def WriteDiff(self, file, queryNumber, propertyPath, values1, values2, changedProps):
        property_name = propertyPath[-1]
        if property_name == '_SnippetsExplanation':
            snippets_diff_representation.repr_entities(file, str(values1), str(values2))
            changedProps.add(queryNumber, "SEE FULL DIFF!!!", htmldiff.COLOR_NODIFF)
            return True
        elif property_name == 'EventLog':
            text1 = '\n'.join(values1)
            text2 = '\n'.join(values2)
            file.write(printers.HtmlBlock.side_by_side_data(text1, text2))
            return True
        elif property_name in ['_RelevFactors', '_AllFactors']:
            return _write_relev_factors_diff(file, queryNumber, propertyPath, values1, values2, changedProps)
        else:
            return super(GtaRelatedAttributeNode, self).WriteDiff(
                file, queryNumber, propertyPath, values1, values1, changedProps
            )


class BinFactorNode(htmldiff.KeyValueMapNode):
    def __init__(self):
        super(BinFactorNode, self).__init__("BinFactor", ["Grouping", "Group", "Document"])


class _DiffWriter(object):

    LIST_PROPERTY_FMT = "{}[{}]"
    DICT_PROPERTY_FMT = "{}.{}"
    FACTOR_FMT = '{:<17.11f}'

    FORMAT = "{:<50}{:<17}{}"
    HEAD = FORMAT.format("Name", "Old", "New")

    def __init__(self, diff_file, query_number, changed_props, diff_as_factors=False):
        self.__diff_file = diff_file
        self.__query_number = query_number
        self.__changed_props = changed_props
        self.__diff_as_factors = diff_as_factors
        if diff_as_factors:
            self.FORMAT = "{:<17}{:<17}{:<17}{}"
            self.HEAD = self.FORMAT.format("Old", "New", "Delta", "Name")

    def write_diff(self, property_path, value1, value2):
        """
        Generate diff between two lists or two dicts
        """

        if isinstance(value1, (list, tuple)) and isinstance(value2, (list, tuple)):
            self.__write_list_diff(property_path, value1, value2)

        elif isinstance(value1, dict) and isinstance(value2, dict):
            self.__write_dict_diff(property_path, value1, value2)
        else:
            raise Exception("Failed to write dict for values {} and {}".format(value1, value2))

    def __write_list_diff(self, property_path, list1, list2):
        """
        Generate diff between two lists
        """

        min_length = min(len(list1), len(list2))

        # change
        self.__write_change_diff(
            property_path, self.LIST_PROPERTY_FMT, (
                (i, list1[i], list2[i]) for i in six.moves.xrange(min_length)
            )
        )
        # add or remove
        if len(list1) < len(list2):
            color, delta = htmldiff.COLOR_ADDED, list2[min_length:]
        else:
            color, delta = htmldiff.COLOR_REMOVED, list1[min_length:]

        self.__write_delta_diff(
            property_path, self.LIST_PROPERTY_FMT, (
                (i + min_length, elt) for i, elt in enumerate(delta)
            ),
            color,
        )

    def __write_dict_diff(self, property_path, dict1, dict2):
        """Generate diff between two dicts"""

        keys1 = set(dict1.keys())
        keys2 = set(dict2.keys())

        # change
        self.__write_change_diff(
            property_path, self.DICT_PROPERTY_FMT, ((key, dict1[key], dict2[key]) for key in keys1 & keys2)
        )
        # add or remove
        self.__write_delta_diff(
            property_path, self.DICT_PROPERTY_FMT, ((k, dict1[k]) for k in keys1 - keys2), htmldiff.COLOR_REMOVED
        )
        self.__write_delta_diff(
            property_path, self.DICT_PROPERTY_FMT, ((k, dict2[k]) for k in keys2 - keys1), htmldiff.COLOR_ADDED
        )

    @staticmethod
    def __has_detailed_diff(value1, value2):
        """
        Check whether values are both lists or dicts
        """

        return (
            isinstance(value1, (list, tuple)) and
            isinstance(value2, (list, tuple))
        ) or (
            isinstance(value1, dict) and
            isinstance(value2, dict)
        )

    def __format_factor_value(self, value):
        return value if value == "-" else self.FACTOR_FMT.format(float(value))

    @staticmethod
    def __numeric_factor_value(value):
        return 0.0 if value == "-" else float(value)

    def __add_property_diff(self, change_diff, property_name, value1, value2):
        if self.__diff_as_factors:
            delta = self.__numeric_factor_value(value2) - self.__numeric_factor_value(value1)
            value1 = self.__format_factor_value(value1)
            value2 = self.__format_factor_value(value2)
            diff = self.FORMAT.format(value1, value2, delta, property_name)
        else:
            diff = self.FORMAT.format(property_name, value1, value2)
        change_diff.append(diff)

    def __write_delta_diff(self, property_path, fmt_str, items, color):
        change_diff = [self.HEAD]
        add = (color == htmldiff.COLOR_ADDED)
        for key, value in items:
            value = _ensure_str(value)
            property_name = self.__get_prop_name(property_path[-1], key, fmt_str, add)
            new_property_path = property_path[:-1] + [property_name]
            if self.__changed_props is not None:
                self.__changed_props.add(self.__query_number, new_property_path, color, value)
            value1 = "-" if add else value
            value2 = value if add else "-"
            self.__add_property_diff(change_diff, property_name, value1, value2)

        if len(change_diff) > 1:
            htmldiff.WriteData(self.__diff_file, "\r\n".join(change_diff), color)

    def __write_change_diff(self, property_path, fmt_str, items):
        change_diff = [self.HEAD]
        for key, value1, value2 in items:
            value1 = _ensure_str(value1)
            value2 = _ensure_str(value2)

            if value1 == value2:
                continue

            property_name = self.__get_prop_name(property_path[-1], key, fmt_str)
            new_property_path = property_path[:-1] + [property_name]
            if self.__has_detailed_diff(value1, value2):
                self.write_diff(new_property_path, value1, value2)
            else:
                if self.__changed_props is not None:
                    self.__changed_props.add_property(
                        self.__query_number, new_property_path, value1, value2
                    )
                self.__add_property_diff(change_diff, property_name, value1, value2)

        if len(change_diff) > 1:  # 1 is for HEAD
            htmldiff.WriteData(self.__diff_file, "\r\n".join(change_diff), htmldiff.COLOR_CHANGED)

    @staticmethod
    def __get_prop_name(last_prop_path, key, fmt_str, factor_list_num=0):
        if last_prop_path == "relev_factor":
            try:
                return "[{:4}] {}".format(key, GLOBAL_FACTOR_NAMES[factor_list_num][int(key)])
            except Exception as e:
                # probably, there was no option to set GLOBAL_FACTOR_NAMES list
                # for example, in case of using local sandbox it's meaningless to remote copy all resources with names
                logging.debug("Cannot get factor name, print it by number\nException:\n%s", e)
        return fmt_str.format(last_prop_path, key)


def write_objects_diff(
    diff_file,
    query_number,
    property_path,
    values1,
    values2,
    changed_props,
    diff_as_factors=False,
):
    _DiffWriter(
        diff_file,
        query_number,
        changed_props,
        diff_as_factors=diff_as_factors,
    ).write_diff(property_path, values1, values2)


def make_node_types_dict():
    return htmldiff.node_types_dict([
        GroupingNode(),
        SearcherPropNode(),
        FactorMappingNode(),
        GroupNode(),
        DocumentNode(),
        FirstStageAttributeNode(),
        GtaRelatedAttributeNode(),
        BinFactorNode(),
    ])
