#!/usr/bin/env python
# -*- coding: utf-8 -*-
"""
    Используется в основном для задач FastBasesearchAcceptance
    и для FastCheckBasesearchResponses.
"""

from sandbox.sandboxsdk import parameters as sp
from sandbox.sandboxsdk.channel import channel

from sandbox.projects.common.search import params_handler
from sandbox.projects.common import error_handlers as eh
from sandbox.projects.common import utils
from sandbox.projects import resource_types

_FROM_PRODUCTION = 'from_production'
_EXISTING = 'existing'
_PROD_EXISTING_CHOICE = [
    ('From production', _FROM_PRODUCTION),
    ('Existing', _EXISTING),
]


class ExecutableBase(sp.LastReleasedResource):
    name = 'executable{}_resource_id'
    description = 'BaseSearch Executable {}'
    resource_type = [resource_types.BASESEARCH_EXECUTABLE]
    group = 'Runtime source selection'


class Executable1(ExecutableBase):
    name = ExecutableBase.name.format(1)
    description = ExecutableBase.description.format(1)


class Executable2(ExecutableBase):
    name = ExecutableBase.name.format(2)
    description = ExecutableBase.description.format(2)


class ExecutableSourceBase(sp.SandboxStringParameter):
    name = "executable{}_source"
    description = 'Executable source {}'
    choices = _PROD_EXISTING_CHOICE
    default_value = _FROM_PRODUCTION
    group = 'Runtime source selection'


class ExecutableSource1(ExecutableSourceBase):
    name = ExecutableSourceBase.name.format(1)
    description = ExecutableSourceBase.description.format(1)
    sub_fields = {
        _EXISTING: [Executable1.name]
    }


class ExecutableSource2(ExecutableSourceBase):
    name = ExecutableSourceBase.name.format(2)
    description = ExecutableSourceBase.description.format(2)
    sub_fields = {
        _EXISTING: [Executable2.name]
    }


class DynamicModelsBase(sp.LastReleasedResource):
    name = 'dynamic_models{}_res_id'
    description = 'Dynamic models archive {}'
    resource_type = [
        resource_types.DYNAMIC_MODELS_ARCHIVE,
        resource_types.DYNAMIC_MODELS_ARCHIVE_BASE,
    ]
    group = 'Runtime source selection'


class DynamicModels1(DynamicModelsBase):
    name = DynamicModelsBase.name.format(1)
    description = DynamicModelsBase.description.format(1)


class DynamicModels2(DynamicModelsBase):
    name = DynamicModelsBase.name.format(2)
    description = DynamicModelsBase.description.format(2)


class DynamicModelsSourceBase(sp.SandboxStringParameter):
    name = "dynamic_models_archive{}_source"
    description = 'Dynamic models source {}'
    choices = _PROD_EXISTING_CHOICE
    default_value = _FROM_PRODUCTION
    group = 'Runtime source selection'


class DynamicModelsSource1(DynamicModelsSourceBase):
    name = DynamicModelsSourceBase.name.format(1)
    description = DynamicModelsSourceBase.description.format(1)
    sub_fields = {
        _EXISTING: [DynamicModels1.name]
    }


class DynamicModelsSource2(DynamicModelsSourceBase):
    name = DynamicModelsSourceBase.name.format(2)
    description = DynamicModelsSourceBase.description.format(2)
    sub_fields = {
        _EXISTING: [DynamicModels2.name]
    }


def get_source_for_other_resources(fast=False):
    class OtherResourcesSource(sp.SandboxStringParameter):
        name = "other_resources_source"
        description = 'Source for db, queries and config'
        choices = _PROD_EXISTING_CHOICE
        sub_fields = {
            _EXISTING: [res.name for res in get_other_existing_res(fast)]
        }
        default_value = _FROM_PRODUCTION
        group = 'Runtime source selection'

    return OtherResourcesSource


def get_other_existing_res(fast=False):
    res = []
    for tier_name in get_basesearch_production_shard_tag_names(fast):
        res += [
            get_db_by_tier(tier_name),
            get_initial_config_by_tier(tier_name),
            get_queries_by_tier(tier_name),
        ]
    return res


def get_db_by_tier(tier=""):
    class DatabaseParameter(sp.LastReleasedResource):
        name = 'db_{}_res_id'.format(tier)
        description = 'Database for {}'.format(tier)
        resource_type = resource_types.SEARCH_DATABASE
        group = 'Runtime source selection'

    return DatabaseParameter


def get_initial_config_by_tier(tier=""):
    class ConfigParameter(sp.LastReleasedResource):
        name = 'initial_config_{}_res_id'.format(tier)
        description = 'Config for {}'.format(tier)
        resource_type = resource_types.SEARCH_CONFIG
        group = 'Runtime source selection'

    return ConfigParameter


def get_queries_by_tier(tier=""):
    class QueriesParameter(sp.LastReleasedResource):
        name = 'queries_{}_res_id'.format(tier)
        description = 'Queries for {}'.format(tier)
        resource_type = resource_types.PLAIN_TEXT_QUERIES
        group = 'Runtime source selection'

    return QueriesParameter


class GetAllFactors(sp.SandboxBoolParameter):
    name = 'get_all_factors'
    description = 'Get all factors (dbgrlv=da and gta=_RelevFactors)'
    default_value = False
    group = "Responses setup"


class PoliteMode(sp.SandboxBoolParameter):
    name = 'polite_mode'
    description = 'Polite mode'
    default_value = False
    group = "Responses setup"


class IgnoreGotError(sp.SandboxBoolParameter):
    name = 'ignore_got_error'
    description = 'Ignore basesearch GotError'
    default_value = False
    group = "Responses setup"


def get_basesearch_production_shard_tag_names(fast=False):
    return ["PlatinumTier0"]


RES_ATTR_PREFIX = "TE_web_base_prod_resources_"
QUERIES_ATTR_PREFIX = "TE_web_base_prod_queries_all_"

_resource_counter = 0


def get_models_archive(ctx, n=1):
    if ctx.get(DynamicModelsSourceBase.name.format(n), _FROM_PRODUCTION) == _FROM_PRODUCTION:
        models_res_id = utils.get_and_check_last_released_resource_id(resource_types.DYNAMIC_MODELS_ARCHIVE_BASE)
    else:
        models_res_id = ctx.get(DynamicModelsBase.name.format(n))

    return models_res_id


def get_basesearch_exe(ctx, n=1):
    if ctx.get(ExecutableSourceBase.name.format(n), _FROM_PRODUCTION) == _FROM_PRODUCTION:
        base_exe_res_id = utils.get_and_check_last_released_resource_id(
            resource_types.BASESEARCH_EXECUTABLE,
            arch='linux',
        )
    else:
        base_exe_res_id = ctx.get(ExecutableBase.name.format(n))

    return base_exe_res_id


def _get_fake_resource():
    class FakeResource(object):
        def get_host(self):
            return "host123"

    global _resource_counter
    _resource_counter += 1

    res = FakeResource()
    res.id = _resource_counter

    return res


def get_shard_tags_and_resources(dev_mode=False, fast=False):
    result = {}
    source = channel.task.ctx.get("other_resources_source")

    for shard_tag in get_basesearch_production_shard_tag_names(fast):
        if dev_mode:
            result[shard_tag] = {
                "db": _get_fake_resource(),
                "config": _get_fake_resource(),
                "queries": _get_fake_resource(),
            }
        elif source == _FROM_PRODUCTION:
            db, queries = params_handler.get_last_interrelated_resources(
                resource_types.SEARCH_DATABASE, RES_ATTR_PREFIX + shard_tag,
                resource_types.PLAIN_TEXT_QUERIES, QUERIES_ATTR_PREFIX + shard_tag,
            )
            config = utils.get_and_check_last_resource_with_attribute(
                resource_types.SEARCH_CONFIG, RES_ATTR_PREFIX + shard_tag,
            )
            result[shard_tag] = {
                "db": db,
                "config": config,
                "queries": queries,
            }
        elif source == _EXISTING:
            result[shard_tag] = {
                "db": channel.sandbox.get_resource(channel.task.ctx['db_{}_res_id'.format(shard_tag)]),
                "config": channel.sandbox.get_resource(channel.task.ctx['initial_config_{}_res_id'.format(shard_tag)]),
                "queries": channel.sandbox.get_resource(channel.task.ctx['queries_{}_res_id'.format(shard_tag)]),
            }
        else:
            eh.fail('Wrong type of source: {}'.format(source))

    return result


CTX_KEY_SOURCES = [
    ExecutableSource1,
    ExecutableSource2,
    DynamicModelsSource1,
    DynamicModelsSource2,
    get_source_for_other_resources(),
]


def reset_resources_ids(task):
    """
        Если надо брать ресурс из продакшена - убираем скопированное значение.
        SEARCH-1381
    """
    for source in CTX_KEY_SOURCES:
        ctx_val = task.ctx.get(source.name)
        if ctx_val == _FROM_PRODUCTION:
            for ctx_key in source.sub_fields[_EXISTING]:
                task.ctx[ctx_key] = 0
