# -*- coding: utf-8 -*-

import os.path
import tempfile
import shutil
import logging
import urllib

import cStringIO
import cPickle

from sandbox import sandboxsdk

from sandbox.projects.common import error_handlers as eh
from sandbox.projects.common import string
from sandbox.projects.common.search import queries as sq

from . import node_types
from . import threadPool
from .tree import htmldiff
from .tree import node as tree_node


class _HtmlDiffFileSet(object):
    """Set of output files"""

    def __init__(self, diff_path=".", max_diff_size=0, max_diff_files=0, number_field_width=1):

        self.__max_diff_files = max_diff_files
        self.__diff_files = 0

        self.__max_diff_size = max_diff_size
        self.__diff_size = 0
        self.__fileobj = None

        self.__diff_path = diff_path
        self.__number_field_width = number_field_width

        self.__reopen()

    def write(self, query_number, data):
        """Write next chunk of data to current diff file"""

        if self.__max_diff_size and (self.__diff_size + len(data) > self.__max_diff_size):
            logging.info(
                "Diff is big enough (%s > %s)\nExiting...",
                self.__diff_size + len(data),
                self.__max_diff_size,
            )
            return False

        self.__fileobj.write(htmldiff.StartBlock("<b>query:</b> {0}".format(query_number)))
        self.__fileobj.write(data)
        self.__fileobj.write(htmldiff.EndBlock())
        self.__fileobj.write("\n<hr>\n")

        self.__diff_size += len(data)
        return True

    def next(self, start_number, end_number):
        """Switch to next diff file"""

        if self.__max_diff_files and self.__diff_files > self.__max_diff_files:
            logging.info(
                "Too many files with diff (%s > %s)\nExiting...", self.__diff_files, self.__max_diff_files
            )
            return False

        if not self.__diff_size:
            return False

        diff_filename = os.path.join(
            self.__diff_path,
            "queries_{:0{width}}-{:0{width}}.html".format(
                start_number,
                end_number,
                width=self.__number_field_width
            )
        )
        logging.info("Dumping diff portion to %s", diff_filename)

        with open(diff_filename, "w") as diff_fileobj:
            def write_body():
                self.__fileobj.seek(0)
                shutil.copyfileobj(self.__fileobj, diff_fileobj)

            htmldiff.WriteDiff(diff_fileobj, write_body)

        self.__reopen()
        return True

    def __reopen(self):
        """Open temporary file for diff data"""

        if self.__fileobj is not None:
            self.__fileobj.close()
        self.__fileobj = tempfile.TemporaryFile(dir=self.__diff_path)
        self.__diff_size = 0
        self.__diff_files += 1


class _HtmlDiffParams(object):
    """Parameters to thread worker"""

    def __init__(self, **args):
        self.args = args


def _fill_factor_index_to_name(response):
    result = {}

    def get_info_from_node(mapping_node):
        key = mapping_node.GetPropValue("Key")
        value = mapping_node.GetPropValue("Value", parse=False)
        result[value] = key

    tree_node.WalkPath(response, ["Head", "FactorMapping"], get_info_from_node)
    return result


def _patch_bin_factors(response, factor_index_to_name):

    def patch_node(bin_factor_node):
        bin_factor_key = bin_factor_node.GetPropValue("Key")
        factor_name = factor_index_to_name.get(bin_factor_key)
        bin_factor_node.SetPropValue("Key", "{} ({})".format(bin_factor_key, factor_name))

    tree_node.WalkPath(response, ["Grouping", "Group", "Document", "BinFactor"], patch_node)


def _patch_response(response):

    factor_index_to_name = _fill_factor_index_to_name(response)
    _patch_bin_factors(response, factor_index_to_name)

    searcher_props = response._nodes.get("SearcherProp")
    if searcher_props:
        props_to_remove = []

        for sp in searcher_props:
            key = sp.GetPropValue("Key")
            parsers = {
                "scheme.json_local_standard.nodump": htmldiff.json_dict_to_tree,
                "QueryData.debug": htmldiff.convert_query_data_proto_to_tree
            }
            if key in parsers.keys():
                try:
                    value_as_tree = parsers[key](sp.GetPropValue("Value"))
                    response._nodes["SearcherProp-{}".format(key)] = [value_as_tree]
                    props_to_remove.append(sp)
                except Exception as e:
                    logging.warning('can not apply parser <{}>:\n{}'.format(key, str(e)))

        for sp in props_to_remove:
            searcher_props.remove(sp)


def _query_field(cgi_params, field, title):
    if field not in cgi_params:
        return ""
    return "\r\n{}: {}".format(title, string.all_to_str(cgi_params[field][0]))


def _write_diff_func(file_obj, query_number, cgi_params, query, r1, r2, changed_props, custom_node_types_dict=None):
    s = string.all_to_str(query)
    try:
        s += _query_field(cgi_params, "lr", "region")
        s += urllib.unquote(_query_field(cgi_params, "user_request", "user request"))
        s += _query_field(cgi_params, "text", "text")
    except Exception as e:
        eh.log_exception("Shit happens, SEARCH-2316", e)
        logging.debug("cgi_params: %s", repr(cgi_params))
        logging.debug("u: %s", repr(_query_field(cgi_params, "user_request", "user request")))
        logging.debug("t: %s", repr(_query_field(cgi_params, "text", "text")))
        logging.debug("s: %s", repr(s))
        raise

    htmldiff.WriteDataBlock(file_obj, "<b>query string</b>", s)

    _patch_response(r1)
    _patch_response(r2)

    node_types_dict = node_types.make_node_types_dict()
    if custom_node_types_dict:
        node_types_dict.update(custom_node_types_dict)

    htmldiff.write_tree_diff(file_obj, query_number, r1, r2, node_types_dict, changed_props)


def _process_html_diff(data, params):
    changed_props = htmldiff.ChangedProps()

    query_start_number = None
    query_end_number = None
    queries_count = 0

    diff_file_set = _HtmlDiffFileSet(
        diff_path=params.args["diff_path"],
        max_diff_files=params.args["max_diff_files"],
        max_diff_size=params.args["max_diff_size"],
        number_field_width=params.args["number_field_width"],
    )

    for query_number, query, r1, r2 in data:
        if r1 == r2:
            continue

        r1 = cPickle.loads(r1)
        r2 = cPickle.loads(r2)

        if r1.Compare(r2):
            continue

        # TODO: rewrite write_diff_func method
        out_file = cStringIO.StringIO()
        cgi_params = sq.parse_cgi_params(query)
        _write_diff_func(
            out_file, query_number, cgi_params, query, r1, r2, changed_props,
            custom_node_types_dict=params.args["custom_node_types_dict"]
        )

        if not diff_file_set.write(query_number, out_file.getvalue()):
            # indeed, this is an error (huge diff, etc)
            return None

        if query_start_number is None:
            query_start_number = query_number

        query_end_number = query_number
        queries_count += 1

        if queries_count % params.args["queries_per_file"] == 0:
            if not diff_file_set.next(query_start_number, query_end_number):
                # indeed, this is an error (huge diff, etc)
                return None
            query_start_number = None

    diff_file_set.next(query_start_number, query_end_number)

    return changed_props


def write_html_diff(
    queries,
    responses1,
    responses2,
    changed_props,
    diff_indexes,
    diff_path=".",
    custom_node_types_dict={},
    use_processes=False,
    queries_per_file=100,
    max_diff_size=0,
    max_diff_files=0,
    ctx=None,
):
    """
    Generates a set of html file with diff between two sets of responses
    """

    if not diff_indexes:
        return

    eh.verify(
        len(responses1) == len(responses2),
        "Responses should have same length"
    )

    data = [(i, queries[i], responses1[i], responses2[i]) for i in diff_indexes]

    # adjust number of subprocesses, because of chunks are generated by each worker
    number_of_processes = sandboxsdk.util.system_info()['ncpu']
    while number_of_processes > 1 and (len(data) // number_of_processes < 5):
        number_of_processes -= 1

    params = _HtmlDiffParams(
        diff_path=diff_path,
        custom_node_types_dict=custom_node_types_dict,
        queries_per_file=queries_per_file,
        max_diff_files=max_diff_files,
        max_diff_size=max_diff_size,
        number_field_width=len(str(data[-1][0]))
    )

    props = threadPool.process_data(
        _process_html_diff, data, params,
        use_processes=use_processes,
        process_count=number_of_processes,
        ctx=ctx,
    )

    eh.verify(props is not None, "Got empty props, check _process_html_diff return value. ")

    for p in props:
        changed_props.combine(p)
