# -*- coding: utf-8 -*-
"""
Search response saver tools

Please obey PEP8 coding standards and Sandbox StyleGuide here
https://wiki.yandex-team.ru/sandbox/codestyleguide

Initial author: alexeykruglov@
Responsibles/maintainers: mvel@
"""

import os
import random
import logging
import datetime
import time
import struct
import cPickle
import collections

from . import basesearch_response_parser as brp
from . import response_diff
from . import threadPool
from . import node_types
from . import node_patchers

from .response_saver_params import USE_BINARY_RESPONSES_KEY  # noqa
from .response_saver_params import UNSTABLE_OUTPUT_KEY
from .response_saver_params import GROUP_NAME  # noqa
from .response_saver_params import create_response_saver_params  # noqa
from .response_saver_params import DefaultResponseSaverParams

from .tree import htmldiff

from sandbox.sandboxsdk.channel import channel
from sandbox.sandboxsdk import paths

from sandbox.projects import resource_types
from sandbox.projects.common import file_utils as fu
from sandbox.projects.common import error_handlers as eh
from sandbox.projects.common import utils
from sandbox.projects.common import utils2
from sandbox.projects.common.search import factors_decompressor
from sandbox.projects.common.search.eventlog import eventlog
from sandbox.projects.common.search.response import state as response_state
from sandbox.projects.common.search.response import cgi as response_cgi


def create_resources(task):
    is_binary = utils.get_or_default(task.ctx, DefaultResponseSaverParams.UseBinaryResponses)
    responses_name = "responses." + ["txt", "bin"][is_binary]
    resource_type = [resource_types.BASESEARCH_HR_RESPONSES, resource_types.SEARCH_PROTO_RESPONSES][is_binary]
    task.ctx['out_resource_id'] = str(task.create_resource(
        task.descr,
        responses_name,
        resource_type,
        arch='any',
    ).id)

    if utils.get_or_default(task.ctx, DefaultResponseSaverParams.TestWithCacheDisabled):
        task.ctx['without_cache_responses_id'] = task.create_resource(
            task.descr,
            "disabled_cache_" + responses_name,
            resource_type,
            arch='any',
        ).id

    if utils.get_or_default(task.ctx, DefaultResponseSaverParams.SavePatchedQueriesParameter):
        task.ctx['patched_queries_resource_id'] = task.create_resource(
            'Patched queries for: ' + task.descr,
            'patched_queries.txt',
            resource_types.PLAIN_TEXT_QUERIES,
            arch='any',
        ).id


def get_patched_queries_resource(task):
    resource_id = task.ctx.get('patched_queries_resource_id')
    if not resource_id:
        return None
    return channel.sandbox.get_resource(resource_id)


def get_factor_names(search_component, responses_res_id):
    resource_path = "factor_names.txt"
    factor_names = search_component.get_factor_names()
    fu.write_file(resource_path, factor_names)
    res = channel.task.create_resource(
        description="factor_names",
        resource_path=resource_path,
        resource_type=resource_types.OTHER_RESOURCE,
        attributes={"responses_id": "resource:{}".format(responses_res_id)},
    )
    channel.task.mark_resource_ready(res)

    # записывает в глобальную переменную кортеж из двух одинаковых массивов имен факторов
    # нужно для корректной замены номеров факторов на имена в случае unstable diff
    node_types.GLOBAL_FACTOR_NAMES = ([line.split()[-1] for line in factor_names.split("\n") if line],) * 2


def save_unstable_responses(n, responses_new):
    responses_new_file_name = "basesearch_responses_{}.txt".format(n + 1)
    brp.write_responses(responses_new_file_name, responses_new)
    r = channel.task.create_resource(
        "responses {}".format(n + 1),
        responses_new_file_name,
        resource_types.BASESEARCH_HR_RESPONSES
    )
    channel.task.mark_resource_ready(r)


def check_responses_size(ctx, responses, responses_resource):
    if not responses:
        logging.warning("No responses, nothing to check for size/avg size")
        return

    min_output_size = utils.get_or_default(ctx, DefaultResponseSaverParams.MinOutputSizeParameter)
    min_avg_output_size = utils.get_or_default(ctx, DefaultResponseSaverParams.MinAvgOutputSizeParameter)
    output_size = os.path.getsize(responses_resource.path)
    avg_output_size = output_size / len(responses)
    logging.info("Average output size: %s", avg_output_size)
    if min_output_size:
        eh.ensure(
            output_size >= min_output_size,
            "Total output size is too small: {}".format(output_size)
        )
    if min_avg_output_size:
        eh.ensure(
            avg_output_size >= min_avg_output_size,
            "Average output size is too small: {}".format(avg_output_size)
        )


def _get_response_file_pattern(save_responses_immediately, n):
    if save_responses_immediately:
        return channel.task.log_path("responses_{}_%s.txt".format(n))
    else:
        return None


def write_unstable_html_diff(
    queries,
    responses_1, responses_2,
    custom_node_types_dict,
    diff_indexes,
    use_multiprocessing=True,
):
    channel.task.ctx[UNSTABLE_OUTPUT_KEY] = True
    unstable_diff = channel.task.create_resource(
        "Unstable diff", "unstable_diff", resource_types.EXTENDED_INFO
    )
    paths.make_folder(unstable_diff.path)
    response_diff.write_html_diff(
        queries,
        responses_1, responses_2,
        htmldiff.ChangedProps(),
        diff_indexes,
        diff_path=unstable_diff.path,
        custom_node_types_dict=custom_node_types_dict,
        use_processes=use_multiprocessing
    )
    channel.task.mark_resource_ready(unstable_diff)
    unstable_diff_html = utils2.resource_redirect_link(unstable_diff.id, 'unstable diff')
    channel.task.set_info(
        "Search is not stable, see {0}".format(unstable_diff_html),
        do_escape=False
    )
    return unstable_diff


def save_responses(
    ctx,
    search_component,
    responses_resource,
    queries_patchers=list(),
    relevance_threads=0,
    ignore_empty_response=False,
    prepare_session_callback=None,
    custom_node_types_dict=None,
    response_patchers=None,
    transform_if_response_is_xml=False,
    save_unpatched_responses=False,
    need_max_timeout=True,
    need_complete_answer=False,
    need_dbgrlv=True,
    remove_search_info=True,
    patched_queries_resource=None,
    on_start_get_responses=None,
    on_end_get_responses=None,
    factor_names=False,
    test_with_doubled_queries=False,
    retry_on_removed_docs=False,
):
    """
        Получает ответы от поискового компонента. Перед началом обстрела поисковый компонент
        не должен быть запущен.

        :param ctx: Контекст задачи. Здесь содержится самый главный параметр: запросы, на которые мы получаем ответы.
        :param search_component: Объект поискового компонента, наследник SearchComponent.
        :param responses_resource: Объект ресурса, в который будут сохранены ответы.
        :param queries_patchers: Список патчеров запросов перед обстрелом. Выполняются последовательно.
        :param relevance_threads: Устанавливает cgi-параметр relthreads (по умолчанию 0)
        :param ignore_empty_response: Игнорировать пустой ответ (см. basesearch_response_parser)
        :param prepare_session_callback: Действие, которое следует выполнить до старта поискового компонента.
        :param custom_node_types_dict: Перекрыть стандартный список типов узлов в дереве ответа
            (см. модуль response_diff)
        :param response_patchers: Список функций-преобразователей, применяемых последовательно к исходному запросу.
        :param transform_if_response_is_xml: костыль для xml-поиска (см. basesearch_response_parser)

        :param save_unpatched_responses: Сохранять ответы в изначальном виде. По умолчанию, ответы патчатся сразу
            и в таком виде сохраняются.

        :param need_max_timeout: Выставляет для всех запросов максимальный таймаут "&timeout=4294967295" (по умолчанию)
        :param need_complete_answer: Требует полного ответа (без неответов)

        :param need_dbgrlv: Получать все факторы (по умолчанию).
            Опция проставляет в запрос дополнительно параметры "&dbgrlv=da&gta=_RelevFormula".

        :param remove_search_info: Убирать из запросов параметр "&search_info=da" (по умолчанию).
            Режим search_info=da добавляет в выдачу много (нестабильных) отладочных свойств,
            например, имя хоста, поэтому по умолчанию их следует убирать. Подробности
            см. SEARCH-898 и связанные с ним ревью.

        :param patched_queries_resource: Объект ресурса для сохранения пропатченных запросов.
        :param on_start_get_responses: Действие, которое следует выполнить после старта, но перед обстрелом.
        :param on_end_get_responses: Действие, которое следует выполнить после обстрела, но до остановки компонента.
        :param factor_names: Получать список имен факторов как отдельный ресурс (по умолчанию выключено).
        :param test_with_doubled_queries Дублировать каждый запрос.
        :param retry_on_removed_docs: Перезапрашивать неполные ответы (содержащие RemovedDoc)

        :return responses: ответы в текстовом виде
    """

    dp = DefaultResponseSaverParams
    get_all_factors = utils.get_or_default(ctx, dp.GetAllFactorsParameter)
    use_dcfm = utils.get_or_default(ctx, dp.UseDCFM)
    save_responses_immediately = utils.get_or_default(ctx, dp.SaveResponsesImmediatelyParameter)
    ignore_got_error = utils.get_or_default(ctx, dp.IgnoreGotErrorParameter)
    remove_ig = utils.get_or_default(ctx, dp.RemoveIg)
    get_responses_from_binary = utils.get_or_default(ctx, dp.UseBinaryResponses)
    recheck_n_param = utils.get_or_default(ctx, dp.RecheckResultsNTimesParameter)
    additional_cgi_params = utils.get_or_default(ctx, dp.AdditionalCgiParams)
    decompress_factors = utils.get_or_default(ctx, dp.DecompressFactors)

    # Initialize decompressor once (not to deal with multithreaded resource downloading, it is a mess)
    # Decompressor will be needed when responses stability check is executed below
    factors_decompressor.FactorsDecompressor.init()
    factors_decompressor.PipedFactorsDecompressor.init()

    need_component_start = not search_component.is_running()

    queries = get_queries(ctx)

    if additional_cgi_params:
        # SEARCH-1659
        if additional_cgi_params[0] != '&':
            additional_cgi_params = '&' + additional_cgi_params

        def additional_cgi_params_patcher(query_url):
            return query_url + additional_cgi_params
        queries_patchers.append(additional_cgi_params_patcher)

    # response_saver can only work with responses in protobuf format
    def _ensure_protobuf_as_default_type(query_url):
        if not response_cgi.get_request_type(query_url):
            query_url += '&ms=proto'
        return query_url

    def _add_log_rearrange_worked_rule(query_url):
        return query_url + '&debug=log_rearrange_worked_rule'

    queries_patchers.append(_ensure_protobuf_as_default_type)
    queries_patchers.append(_add_log_rearrange_worked_rule)

    queries = [
        _patch_query(
            q,
            need_max_timeout=need_max_timeout,
            get_all_factors=get_all_factors,
            relevance_threads=relevance_threads,
            queries_patchers=queries_patchers,
            need_dbgrlv=need_dbgrlv,
            remove_search_info=remove_search_info,
            use_dcfm=use_dcfm,
            get_responses_from_binary=get_responses_from_binary,
        ) for q in queries
    ]

    if patched_queries_resource is not None:
        fu.write_lines(patched_queries_resource.path, queries)
        channel.task.mark_resource_ready(patched_queries_resource)

    http_url = "http://localhost:{}".format(search_component.port)
    http_collection = search_component.http_collection

    fetch_response_patchers = [
        remove_eventlog,
        remove_config_info_props,
        node_patchers.compress_formula_and_factors,
    ]
    if remove_ig:
        fetch_response_patchers.append(_remove_ig_props)
    if response_patchers:
        fetch_response_patchers += response_patchers

    def _get_responses(session_number, prepared_queries, resolve_factor_names=False):
        if prepare_session_callback:
            prepare_session_callback()

        if need_component_start:
            search_component.start()
            search_component.wait()

        try:
            search_component.warmup_request()  # Defeat SEARCH-616

            if resolve_factor_names:
                get_factor_names(search_component, responses_resource.id)

            if on_start_get_responses:
                on_start_get_responses()

            thread_params = _ThreadParams(
                base_url=http_url,
                base_collection=http_collection,
                transform_if_response_is_xml=transform_if_response_is_xml,
                save_response_file_pattern=_get_response_file_pattern(save_responses_immediately, session_number),
                ignore_got_error=ignore_got_error,
                ignore_empty_response=ignore_empty_response,
                patch_responses=not save_unpatched_responses,
                response_patchers=fetch_response_patchers,
                need_complete_answer=need_complete_answer,
                get_responses_from_binary=get_responses_from_binary,
                retry_on_removed_docs=retry_on_removed_docs,
                decompress_factors=decompress_factors,
            )

            logging.info("starting threads")
            start_time = time.time()

            thread_responses = search_component.use_component(
                lambda: threadPool.process_data(_thread_func, prepared_queries, thread_params, ctx=ctx)
            )

            logging.info("threads ended")
            logging.info("processing time: %s", str(datetime.timedelta(seconds=(time.time() - start_time))))
        finally:
            if on_end_get_responses:
                on_end_get_responses()

            if need_component_start:
                search_component.stop()

        return thread_responses

    custom_comparer = response_state.has_incomplete_responses if need_complete_answer else None
    if test_with_doubled_queries and not get_responses_from_binary:
        # для бинарных ответов эта опция не работает
        logging.info("Double shooting started")
        responses_doubled = _get_responses(0, _double_queries(queries), resolve_factor_names=factor_names)
        responses = responses_doubled[::2]
        responses_2 = responses_doubled[1::2]
        diff_indexes = []
        if not brp.compare_responses(
            responses, responses_2,
            custom_comparer=custom_comparer,
            diff_indexes=diff_indexes,
        ):
            logging.info("Double shooting found some issues")
            save_unstable_responses(0, responses_2)
            unstable_diff = write_unstable_html_diff(
                queries,
                responses, responses_2,
                custom_node_types_dict,
                diff_indexes,
                use_multiprocessing=ctx.get('use_multiprocessing', True),
            )
            check_unanswers_in_unstable_diffs(search_component, unstable_diff)
        else:
            logging.info("Double shooting finished correctly. No diff found.")
    else:
        responses = _get_responses(0, queries, resolve_factor_names=factor_names)

    if get_responses_from_binary:
        with open(responses_resource.path, "w") as f:
            for response in responses:
                str_size = struct.pack('i', len(response))
                f.write(str_size)
                f.write(response)

        channel.task.mark_resource_ready(responses_resource)
        return

    brp.write_responses(responses_resource.path, responses)

    if save_unpatched_responses:
        # patch only after save
        responses = brp.patch_responses(
            responses,
            response_patchers=fetch_response_patchers,
            use_multiprocessing=ctx.get('use_multiprocessing', True)
        )

    check_responses_size(ctx, responses, responses_resource)

    # recheck (works only for hr responses)
    if recheck_n_param >= 1:
        responses_old = responses

        # send the same requests again and compare with first responses
        for n in range(1, recheck_n_param + 1):
            logging.info("Rechecking results. attempt # %s", n)

            responses_new = _get_responses(n, queries)

            if save_unpatched_responses:
                responses_new = brp.patch_responses(
                    responses_new,
                    response_patchers=fetch_response_patchers,
                    use_multiprocessing=ctx.get('use_multiprocessing', True)
                )

            # True - no diff
            diff_indexes = []
            if not brp.compare_responses(
                responses_old, responses_new,
                custom_comparer=custom_comparer,
                diff_indexes=diff_indexes,
            ):
                logging.info("Recheck attempt # %s found some issues", n)
                save_unstable_responses(n, responses_new)
                unstable_diff = write_unstable_html_diff(
                    queries,
                    responses_old, responses_new,
                    custom_node_types_dict,
                    diff_indexes,
                    use_multiprocessing=ctx.get('use_multiprocessing', True),
                )
                check_unanswers_in_unstable_diffs(search_component, unstable_diff)
            else:
                logging.info("Recheck attempt # %s is done successfully", n)

    _check_fail_rate(ctx, queries, responses)

    # caller should be able to analyze responses manually
    return responses


def check_unanswers_in_unstable_diffs(search_component, unstable_diff):
    """
    Obtains unanswers info from eventlog. Turns task into FAILURE when
    no eventlog available. On low unanswer rate restarts the task once.
    On high unanswer rate fails the task. See SEARCH-2862 for details.
    :param search_component: search component
    :param unstable_diff: resource with unstable diff data
    :return: None
    """
    error_message = (
        "Search is not stable, "
        "see resource:{} for unstable diff, SEARCH-2862 for details. ".format(unstable_diff.id)
    )
    if not hasattr(search_component, "event_log_path"):
        logging.warning("No eventlog path available, cannot analyze unanswers")
        # In case of no information about eventlog: just fail task.
        # Unstable responses are always evil problem.
        eh.check_failed(error_message)
        return

    # SEARCH-2862
    # get previous unanswer percent (or None)
    not_responded_reqids_percent_counted = channel.task.ctx.get(eventlog.NOT_RESPONDED_REQIDS_PERCENT)

    # get current unanswer percent
    unanswers_percent = eventlog.locate_problems(
        channel.task,
        eventlog.get_evlogdump(),
        search_component.event_log_path,
    )
    if not unanswers_percent:
        # zero unanswer rate while having unstable diff is certainly a big trouble
        eh.check_failed(error_message)
        return

    # currently we have some unanswers (unanswers_percent > 0)
    if not_responded_reqids_percent_counted:
        # task was restarted at least once, and we still have some unanswers
        channel.task.set_info(
            "Current unanswer percent: {}, previous unanswer percent: {}. "
            "{}".format(
                unanswers_percent, not_responded_reqids_percent_counted, error_message,
            )
        )
        if unanswers_percent < 1.0:
            channel.task.set_info("Unanswer rate is acceptably low, so we won't fail the task. ")
            return

        # high unanswer rate
        eh.check_failed(error_message)

    # Restart task once: give it a try
    # See https://st.yandex-team.ru/SEARCH-2862#1483010222000
    channel.sandbox.server.restart_task(channel.task.id)


class _ThreadParams(object):
    def __init__(
        self,
        base_url=None,
        base_collection=None,
        save_response_file_pattern=None,
        pid=None,
        ignore_got_error=None,
        ignore_empty_response=None,
        response_patchers=None,
        transform_if_response_is_xml=None,
        patch_responses=None,
        need_complete_answer=False,
        get_responses_from_binary=False,
        retry_on_removed_docs=False,
        decompress_factors=True,
    ):
        self.base_url = base_url
        self.base_collection = base_collection
        self.save_response_file_pattern = save_response_file_pattern
        self.pid = pid
        self.ignore_got_error = ignore_got_error
        self.ignore_empty_response = ignore_empty_response
        self.response_patchers = response_patchers
        self.transform_if_response_is_xml = transform_if_response_is_xml
        self.patch_responses = patch_responses
        self.need_complete_answer = need_complete_answer
        self.get_responses_from_binary = get_responses_from_binary
        self.retry_on_removed_docs = retry_on_removed_docs
        self.decompress_factors = decompress_factors


def get_queries(ctx):
    return _get_queries(
        ctx[DefaultResponseSaverParams.QueriesParameter.name],
        queries_limit=utils.get_or_default(ctx, DefaultResponseSaverParams.QueriesLimitParameter),
    )


def _get_queries(resource_id, queries_limit=0):
    file_name = channel.task.sync_resource(resource_id)
    queries = fu.read_lines(file_name)
    if queries_limit > 0:
        queries = queries[0:queries_limit]
    eh.verify(queries, "Queries list is empty. Please check your input parameters or plan resources. ")
    return queries


def _patch_query(
    q,
    need_max_timeout=True,
    get_all_factors=True,
    relevance_threads=0,
    queries_patchers=None,
    need_dbgrlv=True,
    remove_search_info=True,
    use_dcfm=False,
    get_responses_from_binary=False,
):
    if get_responses_from_binary:
        q = q.replace("&hr=da", "")
        q = response_cgi.remove_protobuf_compression(q)
    else:
        q += "&hr=da"

    if queries_patchers:
        for patcher in queries_patchers:
            q = patcher(q)

    if need_max_timeout:
        q = response_cgi.force_timeout(q)

    gta_param = ["gta", "fsgta"]["&haha=da" in q]

    if need_dbgrlv:
        q += "&dbgrlv=da&{}=_RelevFormula".format(gta_param)

    if remove_search_info:
        q = q.replace("&search_info=da", "").replace("&search_info=1", "").replace("&search_info=yes", "")

    if get_all_factors:
        # FIXME # Note: we pass DCFM unconditionally (see SEARCH-841)
        # FIXME # If search component does not support it, it will just ignore it
        # TODO: enable it forever
        q += "&{}=_RelevFactors".format(gta_param)
        if use_dcfm:
            q += "&pron=dcfm"

    if relevance_threads > 0:
        q += "&relthreads={}".format(relevance_threads)

    return q


def _double_queries(queries_list):
    return [q for q in queries_list for _ in (0, 1)]


def _thread_func(queries, params):
    piped_decompressor = None

    if params.decompress_factors:
        piped_decompressor = factors_decompressor.PipedFactorsDecompressor()
        piped_decompressor.open_pipe()

    # (response as str, response as obj)
    queries = list(queries)  # usually it is a generator
    responses = [None for _ in range(len(queries))]
    incomplete_indexes = []

    def fetch(index, query):
        if query.startswith('?'):
            # cropped query without collection
            url = '{}/{}{}'.format(params.base_url, params.base_collection, query)
        elif query.startswith("http"):
            # case with fill query (with some host and port)
            url = '{}/{}{}'.format(params.base_url, params.base_collection, query[query.find("?"):])
        else:
            # cropped query with collection
            url = params.base_url + query

        try:
            def callback(response):
                if not response_state.is_answer_complete(response):
                    incomplete_indexes.append(index)

            response_as_str = None
            parsed_response = None
            for fetch_attempt in range(0, 3):
                response_as_str, parsed_response, checking_result = brp.fetch_response(
                    url,
                    parse_and_check_response=True,
                    remove_non_stable_props=True if params.patch_responses else False,
                    response_patchers=params.response_patchers if params.patch_responses else None,
                    pickle=True,
                    save_response_file=params.save_response_file_pattern,
                    ignore_got_error=params.ignore_got_error,
                    ignore_empty_response=params.ignore_empty_response,
                    transform_if_response_is_xml=params.transform_if_response_is_xml,
                    callback=callback if params.need_complete_answer else None,
                    get_responses_from_binary=params.get_responses_from_binary,
                    piped_decompressor=piped_decompressor if params.decompress_factors else None,
                )

                if parsed_response and params.retry_on_removed_docs:
                    has_removed_docs = brp.has_removed_docs(parsed_response)
                    if has_removed_docs:
                        logging.warning("Retrying incomplete query, attempt %s (see SEARCH-2136)", fetch_attempt)
                        continue
                break

        except Exception:
            logging.exception("Cannot fetch response for url %s\nquery index: %d", url, index)
            raise

        if params.get_responses_from_binary:
            responses[index] = response_as_str
        else:
            responses[index] = parsed_response

    for idx, q in enumerate(queries):
        fetch(idx, q)

    attempt = 1
    while len(incomplete_indexes) > 0 and attempt <= 5:
        logging.info(
            "process %s, num of incomplete_indexes: %s, attempt %s",
            os.getpid(), len(incomplete_indexes), attempt
        )

        if attempt > 1:
            time.sleep(30)

        incomplete_indexes_copy = incomplete_indexes[:]
        incomplete_indexes = []

        for idx in incomplete_indexes_copy:
            fetch(idx, queries[idx])

        attempt += 1

    if params.need_complete_answer and len(incomplete_indexes) > 0:
        logging.info(
            "process %s, num of incomplete_indexes: %s, unanswered_query_example: %s",
            os.getpid(), len(incomplete_indexes), queries[incomplete_indexes[0]]
        )

    if params.decompress_factors:
        piped_decompressor.close_pipe()

    return responses


def remove_config_info_props(response):
    """
        Remove build-specific properties (ConfigInfo_*)
        https://wiki.yandex-team.ru/jandekspoisk/kachestvopoiska/search-components-common/config-info/
        We ought to remove them from responses as they
        will not be comparable by md5 otherwise.
    """
    searcher_props = response.nodes().get("SearcherProp")
    if not searcher_props:
        return

    # remove unstable configuration info (exe/cfg info)
    searcher_props = [prop for prop in searcher_props if not prop.GetPropValue("Key").startswith('ConfigInfo_') or prop.GetPropValue("Key").startswith('DebugConfigInfo_')]
    response.nodes()["SearcherProp"] = searcher_props


def remove_eventlog(response):
    """
        Remove EventLog field from search/idl/meta.proto::TReport
        This field always produces noise. See SEARCH-4561
    """
    if "EventLog" in response.nodes():
        del response.nodes()["EventLog"]


def _remove_ig_props(response):
    """
        Remove SearcherProp/Ig (after SEARCH-1186)
    """
    searcher_props = response.nodes().get("SearcherProp")
    if not searcher_props:
        return

    searcher_props = [prop for prop in searcher_props if not prop.GetPropValue("Key") == 'Ig']
    response.nodes()["SearcherProp"] = searcher_props


def _check_fail_rate(ctx, queries, responses):
    """
    Count max fail rate and fail task if limit is reached (SEARCH-2084)
    """
    max_fail_rate = float(utils.get_or_default(ctx, DefaultResponseSaverParams.MaxFailRate))
    if max_fail_rate <= 0:
        return  # nothing to check

    errors = collections.Counter()
    errors_sample = {}

    fail_count = 0
    query_index = 0
    for response in responses:
        response = cPickle.loads(response)
        _, code, text = brp.get_error_info(response)
        if code:

            errors[text] += 1
            # sample 10 queries
            if text not in errors_sample:
                errors_sample[text] = []
            if random.random() > 0.9 and len(errors_sample[text]) < 10:
                errors_sample[text].append(queries[query_index])

            fail_count += 1

        query_index += 1

    fail_rate = float(fail_count) / len(responses)
    if fail_rate > max_fail_rate:
        message = "Fail rate threshold reached: {} > {}, see logs for details".format(
            fail_rate, max_fail_rate
        )
        for text in errors:
            logging.info("Encountered %s errors of type '%s'", errors[text], text)
            for query in errors_sample[text]:
                logging.info("    Query sample: %s", query)

        logging.error(message)
        eh.check_failed(message)
