import base64
import re
from bz2 import decompress

PATTERN = re.compile(r"</*strong>")


def global_compare(str1, str2):
    """Main function to compare snippets"""
    before = _process_string(str1)
    after = _process_string(str2)
    changes_in_algo_groups = _compare_dicts(before, after, _compare_groups)
    return changes_in_algo_groups


def _process_string(string):
    deciphered_blocks = _decipher(string)
    groups = _group_candidates(deciphered_blocks)
    return groups


def _decipher(string):
    blocks = []
    if 'binaryFactors=' not in string:
        from_base64 = base64.decodestring(string)
        for block in decompress(from_base64).split('\n'):
            if block:
                blocks.append(SnippetExplanation(block.split('\t')))
    return blocks


def _group_candidates(blocks):
    blocks = sorted(blocks, key=lambda explanation: explanation.Algo)
    groups = {}
    point = 0

    def make_dict(group):
        dict_result = {}
        for Snippet in group:
            dict_result[Snippet.get_bare_text()] = Snippet
        return dict_result

    for number, element in enumerate(blocks):
        if blocks[number-1].Algo != element.Algo and number != 0:
            groups[blocks[number-1].Algo] = make_dict(blocks[point:number])
            point = number
    if blocks[point:]:
        groups[blocks[point].Algo] = make_dict(blocks[point:])
    return groups


def _compare_dicts(first, second, compare_func=None):
    result = {
        'Removed': {},
        'Added': {},
        'Changed': {},
        'Unchanged': {},
    }
    for key in first.keys():
        if key in second:
            comparison = _compare(first[key], second[key], compare_func=compare_func)

            if comparison.get('Changed'):
                result['Changed'][key] = comparison['Changed']
        else:
            result['Removed'][key] = first[key]
    for key in second.keys():
        if key not in first:
            result['Added'][key] = second[key]
    return result


def _compare(x, y, compare_func=None):
    if x == y:
        cmp_dict = {}  # {'Unchanged': x}  # uncomment it to see unchanged info in diff
    else:
        cmp_dict = compare_func(x, y) if compare_func is not None else {'Changed': (x, y)}
    return cmp_dict


def _compare_factors(factors_before, factors_after):
    return _compare_dicts(factors_before, factors_after)


def _compare_groups(before, after):
    return _compare_dicts(before, after, SnippetExplanation.get_difference)


class SnippetExplanation:
    def __init__(self, input_list):
        fragments_num = int(input_list[0])
        self.Fragments = ''.join(input_list[1:fragments_num + 1])

        data = input_list[fragments_num + 1:]
        self.Algo = data[0]
        self.Number = data[1]
        self.Coordinates = data[2]
        self.Factors = self._parse_factors(data[3])

        self.Dict = {
            'Numbers': self.Number,
            'Factors': self.Factors,
            'Coordinates': self.Coordinates,
            'Fragments': self.Fragments,
        }

    def get_bare_text(self):
        return re.sub(PATTERN, '', self.Fragments)

    @staticmethod
    def _parse_factors(factors):
        factors_dict = {}
        for pair in factors.split(' '):
            if pair:
                factor, value = pair.split(':')
                factors_dict[factor] = value
        return factors_dict

    def get_difference(self, other):
        result = {
            'Numbers': _compare(self.Number, other.Number),
            'Fragments': _compare(self.Fragments, other.Fragments),
            'Factors': _compare(self.Factors, other.Factors, compare_func=_compare_factors),
            'Coordinates': _compare(self.Coordinates, other.Coordinates),
        }
        self.Dict = result
        if any(['Changed' in value for value in result.values()]):
            return {"Changed": self}
        else:
            return {"Unchanged": self}
